﻿using System;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;

namespace nack
{
    /// <summary>
    /// プログラム本体。
    /// </summary>
    class Program
    {
        /// <summary>
        /// メイン処理。
        /// </summary>
        /// <param name="args">引数。</param>
        static void Main(string[] args)
        {
            var dir = ".";
            var fileEncoding = Encoding.UTF8;
            string grepPattern = null;
            var regexOptions = RegexOptions.None;

            var i = 0;

            for (;  i < args.Length; i++)
            {
                switch (args[i])
                {
                    case "-i": // 大文字・小文字を無視する場合。
                        regexOptions |= RegexOptions.IgnoreCase;
                        break;
                    case "-s": // ファイルがShift-JISの場合。
                        fileEncoding = Encoding.GetEncoding(932);
                        break;
                    case "--": // 以降をオプションとして解析しない。
                        i++;
                        goto loopEnd;
                    default:
                        goto loopEnd;
                }
            }
        loopEnd:

            var restLength = args.Length - i;
            if (restLength == 0)
            {
                Usage();
                return;
            }
            else if (restLength == 1)
            {
                grepPattern = args[i];
            }
            else if (restLength >= 2)
            {
                dir = args[i];
                grepPattern = args[i + 1];
            }

            Run(dir, fileEncoding, grepPattern, regexOptions);
        }

        /// <summary>
        /// 使い方を表示する。
        /// </summary>
        private static void Usage()
        {
            Console.WriteLine("Usage: nack.exe [-i][-s] [dir] pattern");
            Console.WriteLine("  -i: ignore case.");
            Console.WriteLine("  -s: Shift-JIS encoding.");
        }

        /// <summary>
        /// メイン処理の本体を実行する。
        /// </summary>
        /// <param name="dir">検索するディレクトリ。</param>
        /// <param name="fileEncoding">読み込むファイルのエンコーディング。</param>
        /// <param name="grepPattern">検索するパターン。</param>
        /// <param name="regexOptions">検索オプション。</param>
        private static void Run(string dir, Encoding fileEncoding, string grepPattern, RegexOptions regexOptions)
        {
            var grep = new Grep();
            var files = new Find().FindFiles(dir,
                Properties.Settings.Default.FileMatchPattern,
                Properties.Settings.Default.DirNotMatchPattern);

            foreach (var file in files)
            {
                var matched = false;
                foreach (var res in grep.GrepFile(file, fileEncoding, grepPattern, regexOptions))
                {
                    if (matched == false)
                    {
                        // ファイル名を表示する。
                        var dirName = Path.GetDirectoryName(file);
                        var baseName = Path.GetFileName(file);
                        Console.ForegroundColor = ConsoleColor.Green;
                        Console.WriteLine(baseName + " [" + dirName + "]");
                        matched = true;
                    }

                    // 行番号を表示する。
                    Console.ForegroundColor = ConsoleColor.Yellow;
                    Console.Write(res.LineNo + ": ");

                    // 行の文字列を表示する。
                    foreach (var str in res.FoudnStringList)
                    {
                        if (str.Type == FoundStringType.Plain)
                        {
                            Console.ResetColor();
                            Console.Write(str.Value);
                        }
                        else if (str.Type == FoundStringType.Matched)
                        {
                            Console.ResetColor();
                            Console.ForegroundColor = ConsoleColor.Black;
                            Console.BackgroundColor = ConsoleColor.Yellow;
                            Console.Write(str.Value);
                        }
                        else
                        {
                            throw new Exception();
                        }
                    }
                    Console.WriteLine("");
                }
                if (matched)
                {
                    // 空行を表示する。
                    Console.WriteLine("");
                    Console.ResetColor();
                }
            }
        }
    }
}
