/*
 * hexdump.c
 *
 * Utility functions to display the content of any memory region, in a
 * similar style to the hexadecimal output from the unix od command.
 *
 * $Id$
 *
 * Written by Keith Marshall <keithmarshall@users.sourceforge.net>
 * Copyright (C) 2014, 2016, MinGW.org Project
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice, this permission notice, and the following
 * disclaimer shall be included in all copies or substantial portions of
 * the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OF OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */
#include <ctype.h>
#include <stdio.h>

#define HEXDUMP_INLINE static __inline__ __attribute__((__always_inline__))

#ifndef HEXDUMP_OFFSET_FORMAT
#define HEXDUMP_OFFSET_FORMAT  "%04x"
#endif

HEXDUMP_INLINE const unsigned char
*show_bytes( const unsigned char *dp, int count )
{
  /* Internal helper function: prints a sequence of up to sixteen
   * byte values, as stored contiguously in memory, into a display
   * field of fifty character positions; this allows for a display
   * of two hexadecimal digits per byte value, with a single space
   * separating each value from its predecessor, and two further
   * padding spaces, before appending a textual representation
   * of the printable bytes within the sequence.
   *
   * Returns the memory address of the byte immediately following
   * the last byte in the displayed sequence.
   */
  int fw;
  for( fw = 50; count-- > 0; fw -= printf( " %02x", *dp++ ) );
  while( fw-- > 0 ) putchar( 0x20 );
  return dp;
}

HEXDUMP_INLINE const unsigned char
*show_text( const unsigned char *dp, int count )
{
  /* Internal helper function: prints the textual representation of
   * a sequence of up to sixteen contiguous bytes in memory; those
   * bytes which represent a printable character are displayed as
   * such; all others are replaced by a single '.' for display.
   *
   * Returns the memory address of the byte immediately following
   * the last byte in the displayed sequence.
   */
  unsigned int c;
  while( count-- > 0 ) putchar( isprint( c = *dp++ ) ? c : '.' );
  return dp;
}

HEXDUMP_INLINE const unsigned char
*show_hexdump( unsigned offset, const unsigned char *dp, int count )
{
  /* Internal helper function: combines the effects of show_bytes()
   * and show_text() above, to format a single hexdump record for a
   * contiguous sequence of up to sixteen bytes in memory; each such
   * record is prefixed by a caller specified offset, which is taken
   * to represent the offset of the current sequence from the start
   * of the data buffer being dumped.
   *
   * Returns the memory address of the byte immediately following
   * the last byte in the displayed sequence.
   */
  printf( HEXDUMP_OFFSET_FORMAT " ", offset );
  dp = show_text( dp, show_bytes( dp, ((count > 16) ? 16 : count) ) - dp );
  putchar( '\n' );
  return dp;
}

const void *hexdump( void *data, size_t len )
{
  /* Entry point for a general purpose hexdump display function.
   * Displays the content of a memory buffer of arbitrary length,
   * formatting it as a sequence of records of up to sixteen bytes
   * each, in conventional hexdump style.
   *
   * Returns the memory address of the byte immediately following
   * the last byte in the specified data buffer region.
   */
  const void *dp = data, *ep = data + len;
  while( dp < ep ) dp = show_hexdump( dp - data, dp, ep - dp );
  return dp;
}

/* $RCSfile$: end of file */
