/*
 * chktime.c
 *
 * Program to test the hypothesis that MSVCRT.DLL's time_t is equivalent
 * to __time32_t, irrespective of the default behaviour as documented for
 * MSVCR80.DLL and later, where time_t becomes equivalent to __time64_t,
 * unless the user defines the _USE_32BIT_TIME_T macro.
 *
 * $Id$
 *
 * Written by Keith Marshall <keithmarshall@users.sourceforge.net>
 * Copyright (C) 2014-2016, MinGW.org Project
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice, this permission notice, and the following
 * disclaimer shall be included in all copies or substantial portions of
 * the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OF OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 *
 * All MinGW headers assume that <_mingw.h> is included; we want to avoid
 * possible conflicts with typedefs from standard headers, where we want
 * a customized variant; thus, we avoid including as many of the standard
 * system headers as practicible.  However, we do need <dlfcn.h>, to get
 * the inline implementation for dlsym(); we know that it doesn't create
 * any unwanted conflicts, so we go ahead and include it, allowing it to
 * include <_mingw.h> for us, while we copy any required fragments from
 * other system headers, as appropriate.
 *
 */
#include <dlfcn.h>

/* The two possible internal representations of time_t, effectively as
 * defined by time.h; we want only these from time.h, while excluding any
 * possible ambiguity in mapping to time_t itself.
 */
typedef __int32 __time32_t;
typedef __int64 __time64_t;

/* In the case of time_t itself, we override the normal time.h definition,
 * preferring a custom type which can represent either of the preceding
 * internal time types.
 */
typedef union { __time32_t _32bit; __time64_t _64bit; } time_t;

/* We reproduce the definition of size_t from sys/types.h, the prototype
 * for memset() from string.h, and the prototype for printf() from stdio.h,
 * to avoid including any of these.
 */
typedef unsigned __int32 size_t;
_CRTIMP void* __cdecl __MINGW_NOTHROW memset( void *, int, size_t );
_CRTIMP int __cdecl __MINGW_NOTHROW printf( const char *, ... );

/* Program implementation begins with this...
 */
static void eval( const char *fname )
{
  /* ...helper function to lookup the entry point for the specified
   * time() function within the program's (DLL) address space, and to
   * invoke it according to a generic time() function template.
   */
  time_t (*fp)( time_t * );
  if( (fp = dlsym( RTLD_DEFAULT, fname )) == (void *)(0) )
    /*
     * The specified function is not supported in the current
     * program runtime environment; simply add a line record to
     * the results table, reporting this, instead of attempting
     * to invoke a non-existent API function.
     */
    printf( "%8s():%47s\n", fname, "*** function not supported ***" );

  else
  { /* The specified function is available; we set aside a buffer
     * in which to capture its return state, initialize that with
     * a recognizable bit pattern, (recurring 0xaa bytes), so that
     * we can easily see what is subsequently changed as we invoke
     * the specified function...
     */
    time_t now;
    fp( memset( &now, 0xaa, sizeof( time_t ) ));

    /* ...and tabulate the return state, as a single line record
     * in our results table.
     */
    printf( "%8s(): 0x%016I64x%+22I32d%+22I64d\n", fname,
       	now._64bit, now._32bit, now._64bit
      );
  }
}

int main()
{ /* Define a generic underlining string, which may be used by the
   * ULINE macro, (defined below), to underline table column headings
   * with a maximum width of 23 characters.
   */
  char underline[24];
  memset( underline, '-', sizeof(underline) );
  underline[sizeof(underline)-1] = '\0';

  /* Print a table heading.
   */
# define ULINE(LEN) underline + sizeof(underline) - LEN - 1
  printf( "Seconds elapsed since the beginning of the unix epoch,\n"
      "as reported by MSVCRT.DLL time functions with time_t interpreted as:--\n"
      "\n%30s   %-22s%s\n%30s%22s%22s\n", "Uninterpreted [*]",
      "32-bit (__time32_t)", "64-bit (__time64_t)",
      ULINE(18), ULINE(20), ULINE(20)
    );

  /* Evaluate and tabulate the effect of calling each of the possible
   * MSVCRT.DLL implementations of the time() function.
   */
  eval( "time" );
  eval( "_time32" );
  eval( "_time64" );

  /* Close the results table, and print a footnote.
   */
  printf( "%10s%20s%22s%22s\n\n"
      "[*] This is a hexadecimal representation of the content of the 64-bit\n"
      "    results buffer, AFTER the function call; note that this was filled\n"
      "    with 0xaa bytes BEFORE the function call, and that any bytes which\n"
      "    retain this value after the call may be assumed to have remained\n"
      "    untouched, when calling the specified function.\n",
      ULINE(9), ULINE(18), ULINE(20), ULINE(20)
    );

  /* Always terminate as if successful.
   */
  return 0;
}

/* $RCSfile$: end of file */
