/*
 * chkstat.c
 *
 * Program to test the hypothesis that MSVCRT.DLL retains the 32-bit
 * time_t format, (equivalent to __time32_t), for its representation
 * of file system time stamps within any instance of the _stat family
 * of structures, (other than those which explicitly declare their
 * timestamp fields to be of type __time64_t), irrespective of MSDN
 * documentation which suggests that __time64_t becomes the default
 * unless _USE_32BIT_TIME_T is defined.
 *
 * $Id$
 *
 * Written by Keith Marshall <keithmarshall@users.sourceforge.net>
 * Copyright (C) 2016, MinGW.org Project
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice, this permission notice, and the following
 * disclaimer shall be included in all copies or substantial portions of
 * the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OF OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 *
 * All MinGW headers assume that <_mingw.h> is included; we want to avoid
 * possible conflicts with typedefs from standard headers, where we want
 * a customized variant; thus, we avoid including as many of the standard
 * system headers as practicible.  However, we do need <dlfcn.h>, to get
 * the inline implementation for dlsym(); we know that it doesn't create
 * any unwanted conflicts, so we go ahead and include it, allowing it to
 * include <_mingw.h> for us, while we copy any required fragments from
 * other system headers, as appropriate.
 *
 */
#include <dlfcn.h>

/* The two possible internal representations of time_t, effectively as
 * defined by time.h; we want only these from time.h, while excluding any
 * possible ambiguity in mapping to time_t itself, so just typedef them
 * locally, rather than include the header file.
 */
typedef __int32 __time32_t;
typedef __int64 __time64_t;

/* Similarly, the following three normally come from <sys/types.h>, but
 * it is safer for us to just define them here.
 */
typedef unsigned int _dev_t;
typedef unsigned short _mode_t;
typedef short _ino_t;

/* Here, we definitely want to avoid <sys/stat.h>, to define our own
 * custom variant of the stat structure...
 */
struct _stat
{ union
  { /* ...making it a union of the normal structure definitions, with
     * totally unambiguous declarations for the sizes of the time and
     * file size fields, in each case.
     */
    struct __stat32
    { _dev_t		st_dev;
      _ino_t		st_ino;
      _mode_t		st_mode;
      short		st_nlink;
      short		st_uid;
      short		st_gid;
      _dev_t		st_rdev;
      __int32		st_size;
      __time32_t	st_atime;
      __time32_t	st_mtime;
      __time32_t	st_ctime;
    } stat32i32;

    struct _stat32i64
    { _dev_t		st_dev;
      _ino_t		st_ino;
      _mode_t		st_mode;
      short		st_nlink;
      short		st_uid;
      short		st_gid;
      _dev_t		st_rdev;
      __int64		st_size;
      __time32_t	st_atime;
      __time32_t	st_mtime;
      __time32_t	st_ctime;
    } stat32i64;

    struct _stat64i32
    { _dev_t		st_dev;
      _ino_t		st_ino;
      _mode_t		st_mode;
      short		st_nlink;
      short		st_uid;
      short		st_gid;
      _dev_t		st_rdev;
      __int32		st_size;
      __time64_t	st_atime;
      __time64_t	st_mtime;
      __time64_t	st_ctime;
    } stat64i32;

    struct __stat64
    { _dev_t		st_dev;
      _ino_t		st_ino;
      _mode_t		st_mode;
      short		st_nlink;
      short		st_uid;
      short		st_gid;
      _dev_t		st_rdev;
      __int64		st_size;
      __time64_t	st_atime;
      __time64_t	st_mtime;
      __time64_t	st_ctime;
    } stat64i64;
  };
};

/* To aid identification of regions of the result buffer which may
 * not be updated when _stat(), (or any of its variants), is called,
 * we will initially fill the data space with a recurring recognizable
 * pattern; for this we will use memset(), which is normally prototyped
 * in <string.h>, and which also requires the definition of size_t from
 * <stddef.h>; to avoid including these headers, we may conveniently
 * reproduce the requisite typedef and prototype here...
 */
typedef unsigned __int32 size_t;
_CRTIMP void* __cdecl __MINGW_NOTHROW memset (void *, int, size_t);

/* ...together with these, to avoid including <stdio.h>.
 */
_CRTIMP int __cdecl __MINGW_NOTHROW printf (const char *, ...);
_CRTIMP int __cdecl __MINGW_NOTHROW close (int);

/* This pair are local to this test suite, but are implemented in their
 * own separate translation units.
 */
int __cdecl __MINGW_NOTHROW tmpfile_create (char *);
const void* __cdecl __MINGW_NOTHROW hexdump (void *, size_t);

/* A macro to facilitate underlining of column headings; we initialize
 * the content of its static buffer in our main() function.
 */
static char underline[32];
#define ULINE(LEN) underline + sizeof(underline) - LEN - 1

static void first_header( const char *first, const char *second )
{
  /* Print the table header for the first section of analytical results
   * in each single function test...
   */
  printf(
      "\nInterpreting as respectively %s, and %s return data,"
      "\nthe embedded file sizes and time stamp values are represented as:--"
      "\n\n%43s%26s\n%43s%26s\n", first, second, first, second,
       ULINE(24), ULINE(24)
    );
}
/* ...and the footer line, used to separate each analytical section
 * from the next.
 */
static void table_footer( void )
{ printf( "%17s%26s%26s\n", ULINE(17), ULINE(24), ULINE(24) ); }

/* Macros to specify the format for each table row...
 */
#define DECFMT		"%17s:%+25I64d%+26I64d\n"
#define HEXFMT		"%17s:       0x%016I64x        0x%016I64x\n"
/*
 * ...and to identify the specific content for each row.
 */
#define ATIME(CLASS)	(__time64_t)(refdata.CLASS.st_atime)
#define CTIME(CLASS)	(__time64_t)(refdata.CLASS.st_ctime)
#define MTIME(CLASS)	(__time64_t)(refdata.CLASS.st_mtime)

#define FSIZE(CLASS)	(__int64)(refdata.CLASS.st_size)

static void epoch_header( const char *first, const char *second )
{
  /* Print the separator between the first tabulated data section,
   * and the following interpretation of time stamp values.
   */
  table_footer();
  printf(
      "\nWhen expressed as elapsed seconds since the beginning of "
      "\nthe unix epoch, the above timestamp values represent:--"
      "\n\n%43s%26s\n%43s%26s\n", first, second,
       ULINE(24), ULINE(24)
    );
}

static void fsize_header( const char *first, const char *second )
{
  /* Print the final table separator, introducing the analysis of
   * file size data as decimal values.
   */
  table_footer();
  printf(
      "\nThe decimal value equivalents for the above file sizes are:--"
      "\n\n%43s%26s\n%43s%26s\n", first, second,
       ULINE(24), ULINE(24)
    );
}

static int chkstat( const char *fname, const char *refname )
{
  /* Evaluate the data returned by a single member of the _stat()
   * family of functions, treating all as generically similar.
   */
  int (*fn)( const char *, struct _stat * );

  /* First check that the function of interest is actually present,
   * in the current runtime environment...
   */
  printf( "\nChecking for %s()... ", fname );
  if( (fn = dlsym( RTLD_DEFAULT, fname )) == (void *)(0) )
  {
    /* ...bailing out, if not.
     */
    printf( "no\n%s() is not available in the current run time environment\n\n",
	fname
      );
    return 0;
  }
  else
  { /* Next, check that the specified function actually works, with
     * the return buffer initialized with a predefined pattern...
     */
    struct _stat refdata;
    printf( "yes\nAnalysing behaviour of function %s()...", fname );
    if( fn( refname, memset( &refdata, 0xaa, sizeof( refdata ) )) == -1 )
      /*
       * ...again, bailing out if not.
       */
      printf( " failed\n\n" );

    else
    { /* Print a hexdump of the data returned by the working function...
       */
      printf( "\n\nContent of returned[*] stat data structure is:--\n\n" );
      hexdump( &refdata, sizeof( refdata ) );

      /* ...followed by a field-by-field analysis, with respect to each
       * of the potentially applicable structural layouts.
       */
      first_header( "_stat", "_stat32i64" );
      printf( HEXFMT, "Creation time", CTIME(stat32i32), CTIME(stat32i64) );
      printf( HEXFMT, "Last access time", ATIME(stat32i32), ATIME(stat32i64) );
      printf( HEXFMT, "Last write time", MTIME(stat32i32), MTIME(stat32i64) );
      printf( HEXFMT, "File size", FSIZE(stat32i32), FSIZE(stat32i64) );

      epoch_header( "_stat (__time32_t)", "_stat32i64 (__time32_t)" );
      printf( DECFMT, "Creation time", CTIME(stat32i32), CTIME(stat32i64) );
      printf( DECFMT, "Last access time", ATIME(stat32i32), ATIME(stat32i64) );
      printf( DECFMT, "Last write time", MTIME(stat32i32), MTIME(stat32i64) );

      fsize_header( "_stat (int32_t)", "_stat32i64 (int64_t)" );
      printf( DECFMT, "File size", FSIZE(stat32i32), FSIZE(stat32i64) );
      table_footer();

      first_header( "_stat64i32", "_stat64" );
      printf( HEXFMT, "Creation time", CTIME(stat64i32), CTIME(stat64i64) );
      printf( HEXFMT, "Last access time", ATIME(stat64i32), ATIME(stat64i64) );
      printf( HEXFMT, "Last write time", MTIME(stat64i32), MTIME(stat64i64) );
      printf( HEXFMT, "File size", FSIZE(stat64i32), FSIZE(stat64i64) );

      epoch_header( "_stat64i32 (__time64_t)", "_stat64 (__time64_t)" );
      printf( DECFMT, "Creation time", CTIME(stat64i32), CTIME(stat64i64) );
      printf( DECFMT, "Last access time", ATIME(stat64i32), ATIME(stat64i64) );
      printf( DECFMT, "Last write time", MTIME(stat64i32), MTIME(stat64i64) );

      fsize_header( "_stat64i32 (int32_t)", "_stat64 (int64_t)" );
      printf( DECFMT, "File size", FSIZE(stat64i32), FSIZE(stat64i64) );
      table_footer();

      /* Returning non-zero indicates that the specified function did
       * exist, and appeared to work as advertised.
       */
      return 1;
    }
  }
}

int main()
{ /* Begin by creating, and opening, a new file on which to execute
   * the stat() function tests.
   */
  int fd;
  char fname[] = "./chkfind.XXXXXX";
  if( (fd = tmpfile_create( fname )) >= 0 )
  {
    /* Define a generic underlining string, which may be used by the
     * ULINE macro, (defined above), to underline table column headings
     * with a maximum width of 23 characters.
     */
    memset( underline, '-', sizeof(underline) );
    underline[sizeof(underline)-1] = '\0';

    /* Evaluate each of the potentially available functions, in turn...
     */
    if( chkstat( "_stat", fname )
      | chkstat( "_stat32", fname )
      | chkstat( "_stat64", fname )
      | chkstat( "_stat32i64", fname )
      | chkstat( "_stat64i32", fname )
      | chkstat( "_stati64", fname )
      ) printf( "\n\n"
	/*
	 * ...adding a footnote to the resultant output, when at least
	 * one of them appears to work as advertised.
	 */
	"[*] In each case, the return data structure is filled with a\n"
	"    repeating pattern of 0xaa bytes, before invoking the function\n"
	"    which is to be analysed.  Any byte sequences which continue to\n"
	"    exhibit this pattern, after the function has returned, may be\n"
        "    assumed to have remained untouched during the function call.\n"
      );

    /* Closing the temporary file, when we are done with it, causes the
     * system to delete it automatically.
     */
    close( fd );
  }
  /* Always terminate indicating successful completion.
   */
  return 0;
}

/* $RCSfile$: end of file */
