/*
 * chkfind.c
 *
 * Program to test the hypothesis that MSVCRT.DLL retains the 32-bit
 * time_t format, (equivalent to __time32_t), for its representation
 * of file system time stamps within any instance of the _finddata_t
 * structure, irrespective of MSDN documentation which suggests that
 * _finddata_t may now be equivalent to _finddata64_t, in which the
 * time stamps are represented as 64-bit __time64_t.
 *
 * $Id$
 *
 * Written by Keith Marshall <keithmarshall@users.sourceforge.net>
 * Copyright (C) 2014-2016, MinGW.org Project
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice, this permission notice, and the following
 * disclaimer shall be included in all copies or substantial portions of
 * the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OF OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 *
 * All MinGW headers assume that <_mingw.h> is included; we want to avoid
 * possible conflicts with typedefs from standard headers, where we want
 * a customized variant; thus, we avoid including as many of the standard
 * system headers as practicible.  However, we do need <dlfcn.h>, to get
 * the inline implementation for dlsym(); we know that it doesn't create
 * any unwanted conflicts, so we go ahead and include it, allowing it to
 * include <_mingw.h> for us, while we copy any required fragments from
 * other system headers, as appropriate.
 *
 */
#include <dlfcn.h>

/* The two possible internal representations of time_t, effectively as
 * defined by time.h; we want only these from time.h, while excluding any
 * possible ambiguity in mapping to time_t itself, so just typedef them
 * locally, rather than include the header file.
 */
typedef __int32 __time32_t;
typedef __int64 __time64_t;

/* The following are normally defined in io.h, but we MUST NOT include
 * it, because we want to define our own custom representation for the
 * _finddata_t structure, such that it can represent either one of the
 * underlying _finddata32_t and _finddata64_t structures.
 */
#define FILENAME_MAX  (260)

struct _finddata_t
{ /* Not the way it's defined in the headers, but this custom variant
   * allows it to represent the equivalent of either of the above.
   */
  union
  { struct _finddata32_t
    { /* This is the MSVCR80.DLL and later representation, equivalent to
       * the all 32-bit member _finddata_t of earlier runtime versions...
       */
      unsigned	attrib;
      __time32_t	time_create;
      __time32_t	time_access;
      __time32_t	time_write;
      __int32	size;
      char		name[FILENAME_MAX];
    } _32bit;

    struct _finddata64_t
    { /* ...while this is the all 64-bit member alternative, which is used
       * by those function variants which explicitly require 64-bit members;
       * (it is NOT equivalent to _finddata_t, when that implicitly adopts
       * the 64-bit time_t representation, for that retains a 32-bit file
       * size member, and thus the equivalent is _finddata64i32_t).
       */
      unsigned	attrib;
      __time64_t	time_create;
      __time64_t	time_access;
      __time64_t	time_write;
      __int64	size;
      char		name[FILENAME_MAX];
    } _64bit;
  };
};

/* All variants of _findfirst() require a typedef for intptr_t; this
 * is normally provided in stdint.h, but we prefer not to include it,
 * and anyway, it's just as convenient to typedef it here.
 */
typedef __int32 intptr_t;

/* To aid identification of regions of the result buffer which may
 * not be updated when _findfirst() is called, we will initially fill
 * the data space with a recurring recognizable pattern; for this we
 * will use memset(), normally prototyped in string.h, which also
 * requires the definition of size_t from sys/types.h; to avoid
 * including these headers, we may conveniently reproduce the
 * requisite typedef and prototype here...
 */
typedef unsigned __int32 size_t;
_CRTIMP void* __cdecl __MINGW_NOTHROW memset (void *, int, size_t);

/* ...together with these, to avoid including <stdio.h>.
 */
_CRTIMP int __cdecl __MINGW_NOTHROW printf (const char *, ...);
_CRTIMP int __cdecl __MINGW_NOTHROW close (int);

/* This pair are local to this test suite, but are implemented in
 * their own separate translation units.
 */
int __cdecl __MINGW_NOTHROW tmpfile_create (char *);
const void* __cdecl __MINGW_NOTHROW hexdump (void *, size_t);

static void timdump( const char *fmt, struct _finddata_t *refdata )
# define TIMESTAMP_VALUES(CLASS) refdata->_32bit.CLASS, refdata->_64bit.CLASS
{
  /* Print the analysis of the timestamp fields, within the
   * returned data buffer.
   */
  printf( fmt, "Creation time", TIMESTAMP_VALUES( time_create ) );
  printf( fmt, "Last access time", TIMESTAMP_VALUES( time_access ) );
  printf( fmt, "Last write time", TIMESTAMP_VALUES( time_write ) );
}

static int chkfind( const char *fname, const char *refname )
{
  /* Evaluate the data returned by a single member of the _findfirst()
   * family of functions, treating all as generically similar.
   */
  intptr_t (*fn)( const char *, struct _finddata_t * );

  /* Define a generic underlining string, which may be used by the
   * ULINE macro, (defined below), to underline table column headings
   * with a maximum width of 23 characters.
   */
  char underline[24];
# define ULINE(LEN) underline + sizeof(underline) - LEN - 1
  memset( underline, '-', sizeof(underline) );
  underline[sizeof(underline)-1] = '\0';

  /* First check that the function of interest is actually present,
   * in the current runtime environment...
   */
  printf( "\nChecking for %s()... ", fname );
  if( (fn = dlsym( RTLD_DEFAULT, fname )) == (void *)(0) )
  {
    /* ...bailing out, if not.
     */
    printf( "no\n%s() is not available in the current run time environment\n\n",
	fname
      );
    return 0;
  }
  else
  { /* Next, check that the specified function actually works, with
     * the return buffer initialized with a predefined pattern...
     */
    struct _finddata_t refdata;
    printf( "yes\nAnalysing behaviour of function %s()...", fname );
    if( fn( refname, memset( &refdata, 0xaa, sizeof( refdata ) )) == -1 )
      /*
       * ...again, bailing out if not.
       */
      printf( " failed\n\n" );

    else
    { /* Print a hexdump of the data returned by the working function...
       */
      printf( "\n\nContent of returned[*] file data structure is:--\n\n" );
      hexdump( &refdata, sizeof( refdata ) );

      /* ...followed by a field-by-field analysis, with respect to each
       * of the potentially applicable structural layouts, of the time
       * valued fields within the returned data.
       */
      printf( "\nInterpreting as respectively _finddata32_t and _finddata64_t,"
	  "\nthe embedded file system time stamp values are represented as:--"
	  "\n\n%32s%22s\n%32s%22s\n", "_finddata32_t", "_finddata64_t",
	  ULINE(13), ULINE(20)
       	);
      timdump( "%17s:    0x%08I32x    0x%016I64x\n", &refdata );
      printf( "%17s%15s%22s\n"
	  "\nWhen expressed as elapsed seconds since the beginning of "
	  "\nthe unix epoch, these timestamp values represent:--"
	  "\n\n%32s%22s\n%32s%22s\n", ULINE(17), ULINE(13), ULINE(20),
	  "_finddata32_t", "_finddata64_t", ULINE(13), ULINE(20)
       	);
      timdump( "%17s:%+14I32d%+22I64d\n", &refdata );
      printf( "%17s%15s%22s\n\n", ULINE(17), ULINE(13), ULINE(20) );

      /* Returning non-zero indicates that the specified function did
       * exist, and appeared to work as advertised.
       */
      return 1;
    }
  }
}

int main()
{ /* Begin by creating, and opening, a new file on which to execute
   * the _findfirst() function tests.
   */
  int fd;
  char fname[] = "./chkfind.XXXXXX";
  if( (fd = tmpfile_create( fname )) >= 0 )
  {
    /* Evaluate each of the potentially available functions, in turn...
     */
    if( chkfind( "_findfirst", fname )
      | chkfind( "_findfirst32", fname )
      | chkfind( "_findfirst64", fname )
      ) printf(
	/*
	 * ...adding a footnote to the resultant output, when at least
	 * one of them appears to work as advertised.
	 */
	"[*] In each case, the return data structure is filled with a\n"
	"    repeating pattern of 0xaa bytes, before invoking the function\n"
	"    which is to be analysed.  Any byte sequences which continue to\n"
	"    exhibit this pattern, after the function has returned, may be\n"
        "    assumed to have remained untouched during the function call.\n"
      );

    /* Closing the temporary file, when we are done with it, causes the
     * system to delete it automatically.
     */
    close( fd );
  }
  /* Always terminate indicating successful completion.
   */
  return 0;
}

/* $RCSfile$: end of file */
