// t-psbb.cpp  -*- C++ -*-
//
// Test the effect of the groff .psbb request handling code.
//
// Written by Keith Marshall <keith@users.osdn.me>
// Copyright (C) 2017, Free Software Foundation, Inc.
//
// This file is part of groff.
//
// groff is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// groff is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
//
#define GROFF_PSBB_TEST_CODE
#define GROFF_INPUT_CPP_EMULATION

#ifdef GROFF_INPUT_CPP_EMULATION
// Always defined, within this test module; this serves to demarcate
// code which emulates (not necessarily faithfully) features which are
// required by groff's .psbb request handling code, (which may be found
// in src/roff/troff/input.cpp).
//
#include "psbb.h"
#include "errarg.h"
#include "error.h"

#include <cstring>
#include <cerrno>

const char *program_name = "psbb";
const char *current_filename = "t-psbb";
const char *current_source_filename = "t-psbb.cpp";

int current_lineno = 0;

class symbol
{ // Minimal replacement for groff's class implementation, sufficient
  // to satisfy the usage pattern of the ps_bbox_request() function; it
  // can furnish exactly one token, corresponding to each argv element
  // supplied by the test program, subsequently reporting both end of
  // line and end of file, until reset by a new argv.
  public:
    symbol():m_contents((const char *)(NULL)){}
    const char *contents(){ return m_contents; }
    void next(){ m_contents = (const char *)(NULL); }
    void set_contents(const char *contents){ m_contents = contents; }
    int is_null(){ return (m_contents == (const char *)(NULL)); }
    int newline(){ return 1; }
    int eof(){ return 1; }

  private:
    const char *m_contents;
};

// Simplified implementation of get_long_name(); always return this
// fixed symbol class instance.
//
symbol tok;
symbol get_long_name(int unused){ return tok; }

// Substitute a no-op for skip_line()
//
static void skip_line(void){}

class search_path
{ // Minimal replacement for groff's class implementation, sufficient
  // to emulate its open_file_cautious() method.
  public:
    FILE *open_file_cautious( const char *filename, int opt, const char *mode )
    { return fopen( filename, mode ); }
};
#define FOPEN_RB  "rb"

// Not exactly as groff implements it, but sufficient for our needs,
// without requiring anything more than a default constructor for the
// search_path class.
//
search_path include_search_path;

#endif
// Following the GROFF_INPUT_CPP_EMULATION block, we reproduce content
// from src/roff/troff/input.cpp itself, (as we intend that it would be
// ultimately implemented).

// .psbb
//
// Extract bounding box limits from PostScript file, and assign
// them to the following four gtroff registers:--
//
static int llx_reg_contents = 0;
static int lly_reg_contents = 0;
static int urx_reg_contents = 0;
static int ury_reg_contents = 0;

// psbb_assign_registers()
//
// An extern "C" callback function, invoked via our yacc parser,
// to perform initialization and/or register assignment.
//
void psbb_assign_registers(int llx, int lly, int urx, int ury)
{
  llx_reg_contents = llx;
  lly_reg_contents = lly;
  urx_reg_contents = urx;
  ury_reg_contents = ury;
}

// psbb_open_file_for_parse()
//
// A further extern "C" callback function, called by our yacc parser
// start-up routine, psbb_get_bounding_box(), to attach "yyin" to the
// specified file, in preparation for lexical analysis.
//
FILE *psbb_open_file_for_parse(const char *filename)
{ FILE *fp = include_search_path.open_file_cautious(filename, 0, FOPEN_RB);
  if (fp == NULL) error("cannot open '%1': %2", filename, strerror(errno));
  return fp;
}

// ps_bbox_request()
//
// Handle the .psbb request; this is, effectively, a verbatim copy of
// code, as it should ultimately appear, in src/roff/troff/input.cpp
//
void ps_bbox_request()
{ // Parse input line, to extract file name.
  //
  symbol nm = get_long_name(1);
  if (nm.is_null())
    // No file name specified: ignore the entire request.
    //
    skip_line();
  else {
    // File name acquired: swallow the rest of the line.
    //
    while (!tok.newline() && !tok.eof())
      tok.next();
    errno = 0;

    // Initialize, then update {llx,lly,urx,ury}_reg_contents.
    //
    psbb_assign_registers(0, 0, 0, 0);
    psbb_get_bounding_box(nm.contents());

    // All done for .psbb; move on, to continue
    // input stream processing.
    //
    tok.next();
  }
}

#ifdef GROFF_PSBB_TEST_CODE
// Again, always defined; this block implements the test procedure,
// simulating a groff input stream in which each element of argv is
// interpreted as if parsed as file name arguments to a succession
// of .psbb requests, subsequently printing the bounding box range
// co-ordinates extracted from each named file.

EXTERN_C int psbb_parser_status_check;
static int ps_bbox_request_status(const char *argv)
{
  // Push a single argv element into the emulated groff input
  // stream, then invoke actual src/roff/troff/input.cpp code,
  // as if this argv has been read as ".psbb argv"; return the
  // final internal status code from the underlying parser.
  //
  tok.set_contents(argv); ps_bbox_request();
  return psbb_parser_status_check;
}

int main(int argc, char **argv)
{
  // Require at least one command argument...
  //
  if (argc < 2)
  { error("usage: psbb filename ...\n");
    return 2;
  }
  // ...then push each, in turn, into the simulated groff input
  // stream, and interpret it as if read as an argument to .psbb;
  // successfully processed, report the bounding box result.
  //
  while (--argc > 0)
  { current_lineno = __LINE__; if (ps_bbox_request_status(*++argv) == 0)
      printf("%s: bounding box = (%d,%d)..(%d,%d)\n", *argv,
	  llx_reg_contents, lly_reg_contents, urx_reg_contents, ury_reg_contents
	);
  }
  return 0;
}

#endif /* GROFF_PSBB_TEST_CODE */
