/*
NPlot - A charting library for .NET

GraphicsPdfContentByte.cs
Copyright (C) 2003
Matt Howlett

Redistribution and use of NPlot or parts there-of in source and
binary forms, with or without modification, are permitted provided
that the following conditions are met:

1. Re-distributions in source form must retain at the head of each
   source file the above copyright notice, this list of conditions
   and the following disclaimer.

2. Any product ("the product") that makes use NPlot or parts 
   there-of must either:
  
    (a) allow any user of the product to obtain a complete machine-
        readable copy of the corresponding source code for the 
        product and the version of NPlot used for a charge no more
        than your cost of physically performing source distribution,
	    on a medium customarily used for software interchange, or:

    (b) reproduce the following text in the documentation, about 
        box or other materials intended to be read by human users
        of the product that is provided to every human user of the
        product: 
   
              "This product includes software developed as 
              part of the NPlot library project available 
              from: http://www.nplot.com/" 

        The words "This product" may optionally be replace with 
        the actual name of the product.

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/
/*
 * patch for writing to PDF using iTextSharp.
 * Copyright (C) 2006-2007 mocchi
 */

using System;
using iTextSharp.text;
using iTextSharp.text.pdf;

namespace NPlot
{
	/// <summary>
	/// GraphicsPdfContentByte ̊Tv̐łB
	/// </summary>
	public class GraphicsPdfContentByte : IGraphics
	{
		PdfContentByte cb;
		BaseFont bf;
		System.Drawing.Drawing2D.Matrix trans, trans_;
		System.Drawing.RectangleF region_;

		void SetPen(System.Drawing.Pen pen){
			System.Drawing.Color c = pen.Color;
			cb.SetRGBColorStroke(c.R, c.G, c.B);
			cb.SetLineWidth(pen.Width);
			int no = iTextSharp.text.pdf.PdfContentByte.LINE_CAP_BUTT;
			switch(pen.EndCap){
				case System.Drawing.Drawing2D.LineCap.Round:
					no = iTextSharp.text.pdf.PdfContentByte.LINE_CAP_ROUND;
					break;
				case System.Drawing.Drawing2D.LineCap.Square:
					no = iTextSharp.text.pdf.PdfContentByte.LINE_CAP_PROJECTING_SQUARE;
					break;
			}
			cb.SetLineCap(no);
			no = iTextSharp.text.pdf.PdfContentByte.LINE_JOIN_MITER;
			switch(pen.LineJoin){
				case System.Drawing.Drawing2D.LineJoin.Miter:
				case System.Drawing.Drawing2D.LineJoin.MiterClipped:
					no = iTextSharp.text.pdf.PdfContentByte.LINE_JOIN_MITER;
					break;
				case System.Drawing.Drawing2D.LineJoin.Round:
					no = iTextSharp.text.pdf.PdfContentByte.LINE_JOIN_ROUND;
					break;
				case System.Drawing.Drawing2D.LineJoin.Bevel:
					no = iTextSharp.text.pdf.PdfContentByte.LINE_JOIN_BEVEL;
					break;
			}
			cb.SetLineJoin(no);
			if (pen.DashStyle != System.Drawing.Drawing2D.DashStyle.Solid){
				float[] dp = (float[])pen.DashPattern.Clone();
				float dof = pen.DashOffset * pen.Width;
				for(int i = 0; i < dp.Length; ++i){
					dp[i] *= pen.Width;
				}
				cb.SetLineDash(dp, dof);
			}
		}

		void SetBrush(System.Drawing.Brush brush){
			if (brush is System.Drawing.SolidBrush){
				System.Drawing.SolidBrush sbrush = (System.Drawing.SolidBrush)brush;
				System.Drawing.Color c = sbrush.Color;
				cb.SetRGBColorFill(c.R, c.G, c.B);
			}
		}

		void Begin(){
			cb.SaveState();
			cb.Transform(trans);
			cb.Transform(trans_);
			if (!region_.IsEmpty){
				float x = region_.X, y = region_.Y, width = region_.Width, height = region_.Height;
				cb.Rectangle(x, y, width, height);
				cb.Clip();
			}
			cb.NewPath();
		}
		void Finish(){
			cb.RestoreState();
		}


		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="cb"></param>
		/// <param name="bf"></param>
		public GraphicsPdfContentByte(PdfContentByte cb, BaseFont bf)
		{
			this.cb = cb;
			this.bf = bf;
			ResetTransform();
			ResetClipRectangle();
		}

		/// <summary>
		/// Dispose the object.
		/// </summary>
		public void Dispose(){
		}

		#region IGraphics member

		/// <summary>
		/// Draw rectangle to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="x"></param>
		/// <param name="y"></param>
		/// <param name="width"></param>
		/// <param name="height"></param>
		public void DrawRectangle(System.Drawing.Pen pen, int x, int y, int width, int height) {
			Begin();
			SetPen(pen);
			cb.MoveTo(x, (y+height));
			cb.LineTo(x, y);
			cb.LineTo(x+width, y);
			cb.LineTo(x+width, (y+height));
			cb.ClosePath();
			cb.Stroke();
			Finish();
		}

		/// <summary>
		/// Draw rectangle to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="x"></param>
		/// <param name="y"></param>
		/// <param name="width"></param>
		/// <param name="height"></param>
		public void DrawRectangle(System.Drawing.Pen pen, float x, float y, float width, float height) {
			Begin();
			SetPen(pen);
			cb.MoveTo(x, (y+height));
			cb.LineTo(x, y);
			cb.LineTo(x+width, y);
			cb.LineTo(x+width, (y+height));
			cb.ClosePath();
			cb.Stroke();
			Finish();
		}

		/// <summary>
		/// Draw rectangle to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="rect"></param>
		public void DrawRectangle(System.Drawing.Pen pen, System.Drawing.Rectangle rect) {
			DrawRectangle(pen, rect.X, rect.Y, rect.Width, rect.Height);
		}

		/// <summary>
		/// Fill rectangle to PdfContentByte.
		/// </summary>
		/// <param name="brush"></param>
		/// <param name="x"></param>
		/// <param name="y"></param>
		/// <param name="width"></param>
		/// <param name="height"></param>
		public void FillRectangle(System.Drawing.Brush brush, int x, int y, int width, int height) {
			Begin();
			SetBrush(brush);
			cb.MoveTo(x, (y+height));
			cb.LineTo(x, y);
			cb.LineTo(x+width, y);
			cb.LineTo(x+width, (y+height));
			cb.ClosePath();
			cb.Fill();
			Finish();
		}

		/// <summary>
		/// Fill rectangle to PdfContentByte.
		/// </summary>
		/// <param name="brush"></param>
		/// <param name="x"></param>
		/// <param name="y"></param>
		/// <param name="width"></param>
		/// <param name="height"></param>
		public void FillRectangle(System.Drawing.Brush brush, float x, float y, float width, float height) {
			Begin();
			SetBrush(brush);
			cb.MoveTo(x, (y+height));
			cb.LineTo(x, y);
			cb.LineTo(x+width, y);
			cb.LineTo(x+width, (y+height));
			cb.ClosePath();
			cb.Fill();
			Finish();
		}

		/// <summary>
		/// Fill rectangle to PdfContentByte.
		/// </summary>
		/// <param name="brush"></param>
		/// <param name="rect"></param>
		public void FillRectangle(System.Drawing.Brush brush, System.Drawing.Rectangle rect) {
			FillRectangle(brush, rect.X, rect.Y, rect.Width, rect.Height);
		}

		/// <summary>
		/// Draw line to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="p1"></param>
		/// <param name="p2"></param>
		public void DrawLine(System.Drawing.Pen pen, System.Drawing.Point p1, System.Drawing.Point p2) {
			DrawLine(pen, p1.X, p1.Y, p2.X, p2.Y);
		}

		/// <summary>
		/// Draw line to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="x1"></param>
		/// <param name="y1"></param>
		/// <param name="x2"></param>
		/// <param name="y2"></param>
		public void DrawLine(System.Drawing.Pen pen, int x1, int y1, int x2, int y2) {
			Begin();
			SetPen(pen);
			cb.MoveTo(x1,y1);
			cb.LineTo(x2,y2);
			cb.Stroke();
			Finish();
		}

		/// <summary>
		/// Draw line to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="x1"></param>
		/// <param name="y1"></param>
		/// <param name="x2"></param>
		/// <param name="y2"></param>
		public void DrawLine(System.Drawing.Pen pen, float x1, float y1, float x2, float y2) {
			Begin();
			SetPen(pen);
			cb.MoveTo(x1,y1);
			cb.LineTo(x2,y2);
			cb.Stroke();
			Finish();
		}

		/// <summary>
		/// Draw polygon to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="pts"></param>
		public void DrawPolygon(System.Drawing.Pen pen, System.Drawing.Point[] pts) {
			if (pts.Length == 0) return;
			Begin();
			SetPen(pen);
			cb.MoveTo(pts[0].X,pts[0].Y);
			for (int i = 1; i < pts.Length; ++i){
				cb.LineTo(pts[i].X,pts[i].Y);
			}
			cb.ClosePath();
			cb.Stroke();
			Finish();
		}

		/// <summary>
		/// Fill polygon to PdfContentByte.
		/// </summary>
		/// <param name="brush"></param>
		/// <param name="pts"></param>
		public void FillPolygon(System.Drawing.Brush brush, System.Drawing.Point[] pts) {
			if (pts.Length == 0) return;
			Begin();
			SetBrush(brush);
			cb.MoveTo(pts[0].X,pts[0].Y);
			for (int i = 1; i < pts.Length; ++i){
				cb.LineTo(pts[i].X,pts[i].Y);
			}
			cb.ClosePath();
			cb.Fill();
			Finish();
		}

		/// <summary>
		/// Fill polygon to PdfContentByte.
		/// </summary>
		/// <param name="brush"></param>
		/// <param name="pts"></param>
		public void FillPolygon(System.Drawing.Brush brush, System.Drawing.PointF[] pts) {
			if (pts.Length == 0) return;
			Begin();
			SetBrush(brush);
			cb.MoveTo(pts[0].X,pts[0].Y);
			for (int i = 1; i < pts.Length; ++i){
				cb.LineTo(pts[i].X,pts[i].Y);
			}
			cb.ClosePath();
			cb.Fill();
			Finish();
		}

		/// <summary>
		/// Get and set clipping rectangle.
		/// </summary>
		public System.Drawing.RectangleF ClipRectangle {
			get {
				return region_;
			}
			set {
				region_ = value;
			}
		}

		/// <summary>
		/// Reset clipping rectangle.
		/// </summary>
		public void ResetClipRectangle() {
			region_ = System.Drawing.Rectangle.Empty;
		}

		/// <summary>
		/// Invalid for PDF.
		/// </summary>
		public System.Drawing.Drawing2D.SmoothingMode SmoothingMode {
			get {
				return new System.Drawing.Drawing2D.SmoothingMode ();
			}
			set {
			}
		}

		/// <summary>
		/// Measure string width and height from PdfContentByte.
		/// </summary>
		/// <param name="text"></param>
		/// <param name="font"></param>
		public System.Drawing.SizeF MeasureString(string text, System.Drawing.Font font) {
			return new System.Drawing.SizeF (bf.GetWidthPoint(text, font.SizeInPoints), font.Height);
		}

		/// <summary>
		/// Draw string to PdfContentByte.
		/// </summary>
		/// <param name="text"></param>
		/// <param name="font"></param>
		/// <param name="brush"></param>
		/// <param name="x"></param>
		/// <param name="y"></param>
		public void DrawString(string text, System.Drawing.Font font, System.Drawing.Brush brush, int x, int y) {
			DrawString(text, font, brush, new System.Drawing.PointF(x, y));
		}

		/// <summary>
		/// Draw string to PdfContentByte.
		/// </summary>
		/// <param name="text"></param>
		/// <param name="font"></param>
		/// <param name="brush"></param>
		/// <param name="ptf"></param>
		public void DrawString(string text, System.Drawing.Font font, System.Drawing.Brush brush, System.Drawing.PointF ptf) {
			Begin();
			SetBrush(brush);
			cb.BeginText();
			cb.SetFontAndSize(bf, font.SizeInPoints);
			float asc = bf.GetAscentPoint(text, font.SizeInPoints), desc = bf.GetDescentPoint(text, font.SizeInPoints);
			float bh = font.Height - (asc - desc);
			cb.SetTextMatrix(1.0f, 0.0f, 0.0f, -1.0f, ptf.X, ptf.Y+asc+bh/2.0f);
			cb.ShowText(text);
			cb.EndText();
			Finish();
		}

		/// <summary>
		/// Draw string to PdfContentByte.
		/// </summary>
		/// <param name="text"></param>
		/// <param name="font"></param>
		/// <param name="brush"></param>
		/// <param name="ptf"></param>
		/// <param name="sf"></param>
		public void DrawString(string text, System.Drawing.Font font, System.Drawing.Brush brush, System.Drawing.PointF ptf, System.Drawing.StringFormat sf) {
			Begin();
			SetBrush(brush);
			cb.BeginText();
			cb.SetFontAndSize(bf, font.SizeInPoints);
			float asc = bf.GetAscentPoint(text, font.SizeInPoints), desc = bf.GetDescentPoint(text, font.SizeInPoints);
			float bh = font.Height - (asc - desc);
			cb.SetTextMatrix(1.0f, 0.0f, 0.0f, -1.0f, ptf.X, ptf.Y+asc+bh/2.0f);
			cb.ShowText(text);
			cb.EndText();
			Finish();
		}

		/// <summary>
		/// Draw string to PdfContentByte.
		/// </summary>
		/// <param name="text"></param>
		/// <param name="font"></param>
		/// <param name="brush"></param>
		/// <param name="rcf"></param>
		/// <param name="sf"></param>
		public void DrawString(string text, System.Drawing.Font font, System.Drawing.Brush brush, System.Drawing.RectangleF rcf, System.Drawing.StringFormat sf) {
			DrawString(text, font, brush, new System.Drawing.PointF(rcf.X, rcf.Y));
		}

		/// <summary>
		/// Draw image to PdfContentByte.
		/// </summary>
		/// <param name="img"></param>
		/// <param name="rect"></param>
		public void DrawImage(System.Drawing.Image img, System.Drawing.Rectangle rect) {
			Begin();
			Image image = Image.GetInstance(img, img.RawFormat);
			image.SetAbsolutePosition(rect.X, rect.Y);
			image.ScaleToFit(rect.Width, rect.Height);
			cb.AddImage(image);
			Finish();
		}

		/// <summary>
		/// Draw ellipse to PdfContentByte.
		/// </summary>
		/// <param name="pen"></param>
		/// <param name="x"></param>
		/// <param name="y"></param>
		/// <param name="width"></param>
		/// <param name="height"></param>
		public void DrawEllipse(System.Drawing.Pen pen, int x, int y, int width, int height) {
			Begin();
			SetPen(pen);
			cb.Ellipse(x, y, x+width, (y+height));
			cb.Stroke();
			Finish();
		}

		/// <summary>
		/// Fill ellipse to PdfContentByte.
		/// </summary>
		/// <param name="brush"></param>
		/// <param name="x"></param>
		/// <param name="y"></param>
		/// <param name="width"></param>
		/// <param name="height"></param>
		public void FillEllipse(System.Drawing.Brush brush, int x, int y, int width, int height) {
			Begin();
			SetBrush(brush);
			cb.Ellipse(x, y, x+width, (y+height));
			cb.Fill();
			Finish();
		}

		/// <summary>
		/// Get or set transform matrix.
		/// </summary>
		public System.Drawing.Drawing2D.Matrix Transform {
			get {
				return trans_.Clone();
			}
			set {
				trans_ = value;
			}
		}

		/// <summary>
		/// Reset transform matrix.
		/// </summary>
		public void ResetTransform() {
			trans = new System.Drawing.Drawing2D.Matrix();
			trans.Scale(1.0f, -1.0f);
			trans.Translate(0,-cb.PdfDocument.PageSize.Height);
			trans_ = new System.Drawing.Drawing2D.Matrix();
		}

		/// <summary>
		/// Translate transform matrix.
		/// </summary>
		/// <param name="x"></param>
		/// <param name="y"></param>
		public void TranslateTransform(float x, float y) {
			trans_.Translate(x, y);
		}

		/// <summary>
		/// Rotate transform matrix.
		/// </summary>
		/// <param name="theta">angle (degree)</param>
		public void RotateTransform(float theta) {
			trans_.Rotate(theta);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="sx"></param>
		/// <param name="sy"></param>
		public void ScaleTransform(float sx, float sy) {
			trans_.Scale(sx, sy);
		}

		#endregion
	}
}
