/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.opinionbox.util;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.om.security.UserIdPrincipal;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.turbine.services.InstantiationException;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBox;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBoxMap;
import com.aimluck.eip.cayenne.om.security.TurbineGroup;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.cayenne.om.security.TurbineUserGroupRole;
import com.aimluck.eip.common.ALBaseUser;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.mail.util.ALMailUtils;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.orgutils.ALOrgUtilsService;
import com.aimluck.eip.user.beans.UserGroupLiteBean;
import com.aimluck.eip.user.beans.UserLiteBean;
import com.aimluck.eip.util.ALCellularUtils;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 報告書のユーティリティクラスです。 <BR>
 *
 */
public class OpinionBoxUtils {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(OpinionBoxUtils.class.getName());

  /** 未読 */
  public static final String DB_STATUS_UNREAD = "U";

  /** 既読 */
  public static final String DB_STATUS_READ = "R";

  public static final String DATE_TIME_FORMAT =
    ALDateTimeField.DEFAULT_DATE_TIME_FORMAT;

  public static final String OPINIONBOX_PORTLET_NAME = "OpinionBox";

  /** データベースに登録されたファイルを表す識別子 */
  public static final String PREFIX_DBFILE = "DBF";

  /** パラメータリセットの識別子 */
  private static final String RESET_FLAG = "reset_params";

  /** 検索キーワード変数の識別子 */
  public static final String TARGET_KEYWORD = "keyword";

  /**
   * トピックオブジェクトモデルを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @param isJoin
   *          カテゴリテーブルをJOINするかどうか
   * @return
   */
  public static OriTOpinionBox getOriTOpinionBoxParentReply(RunData rundata,
      Context context, boolean isJoin) throws ALPageNotFoundException,
      ALDBErrorException {
    String opinionid =
      ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
    try {
      if (opinionid == null || Integer.valueOf(opinionid) == null) {
        // トピック ID が空の場合
        logger.debug("[OriTOpinionBox] Empty ID...");
        throw new ALPageNotFoundException();
      }

      int userid = ALEipUtils.getUserId(rundata);

      SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);
      Expression exp1 =
        ExpressionFactory.matchDbExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, Integer
          .valueOf(opinionid));
      query.setQualifier(exp1);
      query.distinct(true);

      List<OriTOpinionBox> opinionboxs = query.fetchList();
      if (opinionboxs == null || opinionboxs.size() == 0) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[OpinionBoxTopic] Not found ID...");
        throw new ALPageNotFoundException();
      }

      OriTOpinionBox opinionbox = opinionboxs.get(0);

      // アクセス権限チェック

      return opinionbox;
    } catch (ALPageNotFoundException pageNotFound) {
      // logger.error(pageNotFound);
      throw pageNotFound;
    } catch (Exception ex) {
      logger.error("[OpinionBoxUtils]", ex);
      throw new ALDBErrorException();

    }
  }

  /**
   * 返信記事オブジェクトモデルを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @param isSuperUser
   *          カテゴリテーブルをJOINするかどうか
   * @return
   */
  public static OriTOpinionBox getOriTOpinionBoxParentReply(RunData rundata,
      Context context, String opinionid, boolean isSuperUser)
      throws ALPageNotFoundException, ALDBErrorException {
    try {
      if (opinionid == null || Integer.valueOf(opinionid) == null) {
        // トピック ID が空の場合
        logger.debug("[OpinionBoxTopic] Empty ID...");
        throw new ALPageNotFoundException();
      }

      SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);
      Expression exp1 =
        ExpressionFactory.matchDbExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, Integer
          .valueOf(opinionid));
      query.setQualifier(exp1);

      if (!isSuperUser) {
        Expression exp2 =
          ExpressionFactory.matchExp(OriTOpinionBox.USER_ID_PROPERTY, Integer
            .valueOf(ALEipUtils.getUserId(rundata)));
        query.andQualifier(exp2);
      }

      List<OriTOpinionBox> opinionboxs = query.fetchList();
      if (opinionboxs == null || opinionboxs.size() == 0) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[OriTOpinionBox] Not found ID...");
        throw new ALPageNotFoundException();
      }
      return opinionboxs.get(0);
    } catch (Exception ex) {
      logger.error("[EipUtils]", ex);
      throw new ALDBErrorException();

    }
  }

  /**
   * OpinionBox オブジェクトモデルを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @param mode_update
   * @return
   */
  public static OriTOpinionBox getOriTOpinionBox(RunData rundata, Context context)
      throws ALDBErrorException {
    String requestid =
      ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
    try {
      if (requestid == null || Integer.valueOf(requestid) == null) {
        // Request IDが空の場合
        logger.debug("[OpinionBoxUtils] Empty ID...");
        return null;
      }

      SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);
      Expression exp1 =
        ExpressionFactory.matchDbExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, requestid);
      query.setQualifier(exp1);

      List<OriTOpinionBox> requests = query.fetchList();

      if (requests == null || requests.size() == 0) {
        // 指定した OpinionBox IDのレコードが見つからない場合
        logger.debug("[OpinionBoxUtils] Not found ID...");
        throw new ALPageNotFoundException();
      }

      return requests.get(0);
    } catch (ALPageNotFoundException ex) {
      ALEipUtils.redirectPageNotFound(rundata);
      return null;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      throw new ALDBErrorException();
    }
  }

  /**
   * OpinionBox オブジェクトモデルを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @param mode_update
   * @return
   */
  public static List<OriTOpinionBox> getOriTOpinionBox(OriTOpinionBox opinionbox)
      throws ALPageNotFoundException {
    try {

      SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);

      Expression exp =
        ExpressionFactory.matchExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, opinionbox);

      query.setQualifier(exp);

      return query.fetchList();
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * マップオブジェクトモデルを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */
  public static List<OriTOpinionBoxMap> getOriTOpinionBoxMap(OriTOpinionBox opinionbox) {
    try {
      SelectQuery<OriTOpinionBoxMap> query = Database.query(OriTOpinionBoxMap.class);
      Expression exp =
        ExpressionFactory.matchDbExp(OriTOpinionBoxMap.ORI_TOPINION_BOX_PROPERTY
          + "."
          + OriTOpinionBox.OPINION_ID_PK_COLUMN, opinionbox.getOpinionId());
      query.setQualifier(exp);

      List<OriTOpinionBoxMap> maps = query.fetchList();

      if (maps == null || maps.size() == 0) {
        // 指定した OpinionBox IDのレコードが見つからない場合
        logger.debug("[OpinionBoxSelectData] Not found ID...");
        return null;
      }
      return maps;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  public static List<UserLiteBean> getAuthorityUsers(RunData rundata,
      String groupname, boolean includeLoginuser) {

    try {
      // アクセス権限
      ALAccessControlFactoryService aclservice =
        (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
          .getInstance())
          .getService(ALAccessControlFactoryService.SERVICE_NAME);
      ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

      List<TurbineUser> ulist =
        aclhandler.getAuthorityUsersFromGroup(
          rundata,
          ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_SELF,
          groupname,
          includeLoginuser);

      List<UserLiteBean> list = new ArrayList<UserLiteBean>();

      UserLiteBean user;
      // ユーザデータを作成し、返却リストへ格納
      for (TurbineUser tuser : ulist) {
        user = new UserLiteBean();
        user.initField();
        user.setUserId(tuser.getUserId());
        user.setName(tuser.getLoginName());
        user.setAliasName(tuser.getFirstName(), tuser.getLastName());
        list.add(user);
      }
      return list;
    } catch (InstantiationException e) {
      return null;
    }

  }

  /**
   * 表示切り替えのリセットフラグがあるかを返す．
   *
   * @param rundata
   * @param context
   * @return
   */
  public static boolean hasResetFlag(RunData rundata, Context context) {
    String resetflag = rundata.getParameters().getString(RESET_FLAG);
    return resetflag != null;
  }

  public static void clearOpinionBoxSession(RunData rundata, Context context) {
    List<String> list = new ArrayList<String>();
    list.add("entityid");
    list.add("submenu");
    list.add("com.aimluck.eip.opinionbox.OpinionBoxSelectDatasort");
    list.add("com.aimluck.eip.opinionbox.OpinionBoxSelectDatasorttype");
    list.add("com.aimluck.eip.opinionbox.OpinionBoxSelectDatafiltertype");
    list.add("com.aimluck.eip.opinionbox.OpinionBoxSelectDatafilter");
    ALEipUtils.removeTemp(rundata, context, list);
  }

  public static int getViewId(RunData rundata, Context context, int uid)
      throws ALDBErrorException {
    int view_uid = -1;
    OriTOpinionBox record = OpinionBoxUtils.getOriTOpinionBox(rundata, context);
    if (record != null) {
      view_uid = record.getUserId();
    } else {
      if (rundata.getParameters().containsKey("view_uid")) {
        view_uid =
          Integer.parseInt(rundata.getParameters().getString("view_uid"));
      } else {
        view_uid = uid;
      }
    }
    ALEipUtils.setTemp(rundata, context, "view_uid", String.valueOf(view_uid));
    return view_uid;
  }

  /**
   * 子のレポート検索のクエリを返します
   *
   * @param requestid
   *          レポートを検索するリクエストのid
   * @return query
   */
  public static SelectQuery<OriTOpinionBox> getSelectQueryForCoOpinionBoxs(int requestid) {
    SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);
    Expression exp =
      ExpressionFactory.matchDbExp(OriTOpinionBox.PARENT_ID_PROPERTY, Integer
        .valueOf(requestid));
    query.setQualifier(exp);
    return query;
  }

  /**
   * Date のオブジェクトを指定した形式の文字列に変換する．
   *
   * @param date
   * @param dateFormat
   * @return
   */
  public static String translateDate(Date date, String dateFormat) {
    if (date == null) {
      return "Unknown";
    }

    // 日付を表示形式に変換
    SimpleDateFormat sdf = new SimpleDateFormat(dateFormat);
    sdf.setTimeZone(TimeZone.getDefault());
    return sdf.format(date);
  }

  /**
   * 返信記事オブジェクトモデルを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @param isSuperUser
   *          カテゴリテーブルをJOINするかどうか
   * @return
   */
  public static OriTOpinionBox getOriTOpinionBoxReply(RunData rundata, Context context,
      String opinionid, boolean isSuperUser) throws ALPageNotFoundException,
      ALDBErrorException {
    try {
      if (opinionid == null || Integer.valueOf(opinionid) == null) {
        // トピック ID が空の場合
        logger.debug("[OpinionBoxTopic] Empty ID...");
        throw new ALPageNotFoundException();
      }

      SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);
      Expression exp1 =
        ExpressionFactory.matchDbExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, Integer
          .valueOf(opinionid));
      query.setQualifier(exp1);

      if (!isSuperUser) {
        Expression exp2 =
          ExpressionFactory.matchExp(OriTOpinionBox.USER_ID_PROPERTY, Integer
            .valueOf(ALEipUtils.getUserId(rundata)));
        query.andQualifier(exp2);
      }

      List<OriTOpinionBox> opinionboxs = query.fetchList();
      if (opinionboxs == null || opinionboxs.size() == 0) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[OpinionBox] Not found ID...");
        throw new ALPageNotFoundException();
      }
      return opinionboxs.get(0);
    } catch (Exception ex) {
      logger.error("[OpinionBoxUtils]", ex);
      throw new ALDBErrorException();

    }
  }

  /**
   * パソコンへ送信するメールの内容を作成する．
   *
   * @return
   */
  public static String createMsgForPc(RunData rundata, OriTOpinionBox opinionbox,
      List<ALEipUser> mapList, Boolean type) {
    boolean enableAsp = JetspeedResources.getBoolean("aipo.asp", false);
    ALEipUser loginUser = null;
    ALBaseUser user = null;

    try {
      loginUser = ALEipUtils.getALEipUser(rundata);
      user =
        (ALBaseUser) JetspeedSecurity.getUser(new UserIdPrincipal(loginUser
          .getUserId()
          .toString()));
    } catch (Exception e) {
      return "";
    }
    String CR = System.getProperty("line.separator");
    StringBuffer body = new StringBuffer("");
    body.append(loginUser.getAliasName().toString());
    if (!"".equals(user.getEmail())) {
      body.append("(").append(user.getEmail()).append(")");
    }
    body
      .append("さんが目安箱")
      .append(type ? "を追加しました。" : "を編集しました。")
      .append(CR)
      .append(CR);
    body
      .append("[タイトル]")
      .append(CR)
      .append(opinionbox.getOpinionBoxName().toString())
      .append(CR);
    body.append("[日時]").append(CR).append(
      translateDate(opinionbox.getCreateDate(), "yyyy年M月d日H時m分")).append(CR);

    if (opinionbox.getNote().toString().length() > 0) {
      body
        .append("[内容]")
        .append(CR)
        .append(opinionbox.getNote().toString())
        .append(CR);
    }

    if (mapList != null) {
      int size = mapList.size();
      int i;
      body.append("[通知先]").append(CR);
      for (i = 0; i < size; i++) {
        if (i != 0) {
          body.append(", ");
        }
        ALEipUser member = mapList.get(i);
        body.append(member.getAliasName());
      }
      body.append(CR);
    }
    body.append(CR);
    body
      .append("[")
      .append(ALOrgUtilsService.getAlias())
      .append("へのアクセス]")
      .append(CR);
    if (enableAsp) {
      body.append("　").append(ALMailUtils.getGlobalurl()).append(CR);
    } else {
      body.append("・社外").append(CR);
      body.append("　").append(ALMailUtils.getGlobalurl()).append(CR);
      body.append("・社内").append(CR);
      body.append("　").append(ALMailUtils.getLocalurl()).append(CR).append(CR);
    }

    body.append("---------------------").append(CR);
    body.append(ALOrgUtilsService.getAlias()).append(CR);

    return body.toString();
  }

  /**
   * 携帯電話へ送信するメールの内容を作成する．
   *
   * @return
   */
  public static String createMsgForCellPhone(RunData rundata,
      OriTOpinionBox opinionbox, List<ALEipUser> mapList,
      Boolean type, int destUserID) {
    ALEipUser loginUser = null;
    ALBaseUser user = null;
    try {
      loginUser = ALEipUtils.getALEipUser(rundata);
      user =
        (ALBaseUser) JetspeedSecurity.getUser(new UserIdPrincipal(loginUser
          .getUserId()
          .toString()));
    } catch (Exception e) {
      return "";
    }
    String CR = System.getProperty("line.separator");
    StringBuffer body = new StringBuffer("");
    body.append(loginUser.getAliasName().toString());
    if (!"".equals(user.getEmail())) {
      body.append("(").append(user.getEmail()).append(")");
    }
    body
      .append("さんが目安箱")
      .append(type ? "を追加しました。" : "を編集しました。")
      .append(CR)
      .append(CR);
    body
      .append("[タイトル]")
      .append(CR)
      .append(opinionbox.getOpinionBoxName().toString())
      .append(CR);
    body.append("[日時]").append(CR).append(
      translateDate(opinionbox.getCreateDate(), "yyyy年M月d日H時m分")).append(CR);

    if (mapList != null) {
      int size = mapList.size();
      int i;
      body.append("[通知先]").append(CR);
      for (i = 0; i < size; i++) {
        if (i != 0) {
          body.append(", ");
        }
        ALEipUser member = mapList.get(i);
        body.append(member.getAliasName());
      }
      body.append(CR);
    }
    body.append(CR);

    ALEipUser destUser;
    try {
      destUser = ALEipUtils.getALEipUser(destUserID);
    } catch (ALDBErrorException ex) {
      logger.error("Exception", ex);
      return "";
    }
    body
      .append("[")
      .append(ALOrgUtilsService.getAlias())
      .append("へのアクセス]")
      .append(CR);
    body.append("　").append(ALMailUtils.getGlobalurl()).append("?key=").append(
      ALCellularUtils.getCellularKey(destUser)).append(CR);
    body.append("---------------------").append(CR);
    body.append(ALOrgUtilsService.getAlias()).append(CR);
    return body.toString();
  }

  /**
   * パソコンへ送信するメールの内容を作成する（返信用）．
   *
   * @return
   */
  public static String createReplyMsgForPc(RunData rundata, OriTOpinionBox opinionbox,
      OriTOpinionBox opinionboxparentopinionbox) {
    boolean enableAsp = JetspeedResources.getBoolean("aipo.asp", false);
    ALEipUser loginUser = null;
    ALBaseUser user = null;

    try {
      loginUser = ALEipUtils.getALEipUser(rundata);
      user =
        (ALBaseUser) JetspeedSecurity.getUser(new UserIdPrincipal(loginUser
          .getUserId()
          .toString()));
    } catch (Exception e) {
      return "";
    }
    String CR = System.getProperty("line.separator");
    StringBuffer body = new StringBuffer("");
    body.append(loginUser.getAliasName().toString());
    if (!"".equals(user.getEmail())) {
      body.append("(").append(user.getEmail()).append(")");
    }
    body.append("さんが目安箱").append("に返信しました。").append(CR).append(CR);
    body.append("[タイトル]").append(CR).append(
      opinionboxparentopinionbox.getOpinionBoxName().toString()).append(CR);
    body.append("[返信日時]").append(CR).append(
      translateDate(opinionbox.getCreateDate(), "yyyy年M月d日H時m分")).append(CR);

    if (opinionbox.getNote().toString().length() > 0) {
      body
        .append("[返信内容]")
        .append(CR)
        .append(opinionbox.getNote().toString())
        .append(CR);
    }
    body.append(CR);
    body
      .append("[")
      .append(ALOrgUtilsService.getAlias())
      .append("へのアクセス]")
      .append(CR);
    if (enableAsp) {
      body.append("　").append(ALMailUtils.getGlobalurl()).append(CR);
    } else {
      body.append("・社外").append(CR);
      body.append("　").append(ALMailUtils.getGlobalurl()).append(CR);
      body.append("・社内").append(CR);
      body.append("　").append(ALMailUtils.getLocalurl()).append(CR).append(CR);
    }

    body.append("---------------------").append(CR);
    body.append(ALOrgUtilsService.getAlias()).append(CR);

    return body.toString();
  }

  /**
   * 携帯電話へ送信するメールの内容を作成する（返信用）．
   *
   * @return
   */
  public static String createReplyMsgForCellPhone(RunData rundata,
      OriTOpinionBox opinionbox, OriTOpinionBox opinionboxparentopinionbox, int destUserID) {
    ALEipUser loginUser = null;
    ALBaseUser user = null;
    try {
      loginUser = ALEipUtils.getALEipUser(rundata);
      user =
        (ALBaseUser) JetspeedSecurity.getUser(new UserIdPrincipal(loginUser
          .getUserId()
          .toString()));
    } catch (Exception e) {
      return "";
    }
    String CR = System.getProperty("line.separator");
    StringBuffer body = new StringBuffer("");
    body.append(loginUser.getAliasName().toString());
    if (!"".equals(user.getEmail())) {
      body.append("(").append(user.getEmail()).append(")");
    }
    body.append("さんが報告書").append("に返信しました。").append(CR).append(CR);
    body.append("[タイトル]").append(CR).append(
      opinionboxparentopinionbox.getOpinionBoxName().toString()).append(CR);
    body.append("[返信日時]").append(CR).append(
      translateDate(opinionbox.getCreateDate(), "yyyy年M月d日H時m分")).append(CR);
    body.append(CR);

    ALEipUser destUser;
    try {
      destUser = ALEipUtils.getALEipUser(destUserID);
    } catch (ALDBErrorException ex) {
      logger.error("Exception", ex);
      return "";
    }
    body
      .append("[")
      .append(ALOrgUtilsService.getAlias())
      .append("へのアクセス]")
      .append(CR);
    body.append("　").append(ALMailUtils.getGlobalurl()).append("?key=").append(
      ALCellularUtils.getCellularKey(destUser)).append(CR);
    body.append("---------------------").append(CR);
    body.append(ALOrgUtilsService.getAlias()).append(CR);
    return body.toString();
  }

  /**
   * フィルターを初期化する．
   *
   * @param rundata
   * @param context
   * @param className
   */
  public static void resetFilter(RunData rundata, Context context,
      String className) {
    ALEipUtils.setTemp(rundata, context, TARGET_KEYWORD, "");
  }

  /**
   * 表示切り替えで指定した検索キーワードを取得する．
   *
   * @param rundata
   * @param context
   * @return
   */
  public static String getTargetKeyword(RunData rundata, Context context) {
    String target_keyword = null;
    String keywordParam = rundata.getParameters().getString(TARGET_KEYWORD);
    target_keyword = ALEipUtils.getTemp(rundata, context, TARGET_KEYWORD);

    if (keywordParam == null && (target_keyword == null)) {
      ALEipUtils.setTemp(rundata, context, TARGET_KEYWORD, "");
      target_keyword = "";
    } else if (keywordParam != null) {
      ALEipUtils.setTemp(rundata, context, TARGET_KEYWORD, keywordParam.trim());
      target_keyword = keywordParam;
    }
    return target_keyword;
  }

  /**
   * 投稿の通知メンバーに入っているか.
   *
   * @param rundata
   * @param context
   * @return
   */
  public static boolean isSelf(RunData rundata, Context context) {
    boolean isSelf = false;
    if (rundata.getParameters().getStringKey("entityid") != null) {
      SelectQuery<OriTOpinionBoxMap> q = Database.query(OriTOpinionBoxMap.class);
      Expression exp1 =
        ExpressionFactory.matchExp(OriTOpinionBoxMap.OPINION_ID_PROPERTY, rundata
          .getParameters()
          .getStringKey("entityid")
          .toString());
      q.andQualifier(exp1);
      List<OriTOpinionBoxMap> queryList = q.fetchList();
      for (OriTOpinionBoxMap repo : queryList) {
        if (repo.getUserId() == ALEipUtils.getUserId(rundata)) {
          isSelf = true;
        }
      }
    }

    return isSelf;
  }

  public static List<OriTOpinionBox> getChildOpinionBoxs(Integer opinionId) {
    SelectQuery<OriTOpinionBox> rquery = Database.query(OriTOpinionBox.class);
    rquery.andQualifier(ExpressionFactory.matchExp(
      OriTOpinionBox.PARENT_ID_PROPERTY,
      opinionId));
    return rquery.fetchList();
  }

  /**
   * 指定されたグループに所属するユーザーを取得します。<br/>
   * DISABLEDがNのユーザー（即ち無効化されたユーザー）は取得しないことに注意してください。
   *
   * @param groupname
   *          グループ名
   * @return ALEipUser の List
   */
  public static List<ALEipUser> getSuperiorUsers(List<UserGroupLiteBean> groupList) {
    List<ALEipUser> list = new ArrayList<ALEipUser>();

    // SQLの作成
    StringBuffer statement = new StringBuffer();

	statement.append("SELECT DISTINCT ");
	statement.append("   B.USER_ID ");
	statement.append("  , B.LOGIN_NAME ");
	statement.append("  , B.FIRST_NAME ");
	statement.append("  , B.LAST_NAME ");
	statement.append("  , D.POSITION  ");
	statement.append("FROM ");
	statement.append("  TURBINE_USER_GROUP_ROLE AS A  ");
	statement.append("  LEFT JOIN TURBINE_USER AS B  ");
	statement.append("    ON A.USER_ID = B.USER_ID  ");
	statement.append("  LEFT JOIN TURBINE_GROUP AS C  ");
	statement.append("    ON A.GROUP_ID = C.GROUP_ID  ");
	statement.append("  LEFT JOIN EIP_M_USER_POSITION AS D  ");
	statement.append("    ON A.USER_ID = D.USER_ID  ");
	statement.append("WHERE ");
	statement.append("  B.USER_ID > 3  ");
	statement.append("  AND B.DISABLED = 'F'  ");
	statement.append("  AND C.GROUP_ALIAS_NAME IN (  ");
	statement.append("    SELECT ");
	statement.append("      POST.POST_NAME  ");
	statement.append("    FROM ");
	statement.append("      EIP_M_POST POST  ");
	statement.append("    WHERE ");
	statement.append("      POST.HEAD_OFFICE_FLG = '1' ");
	statement.append("  )  ");

	if (groupList.size() > 0) {

	statement.append("UNION  ");
	statement.append("SELECT DISTINCT ");
	statement.append("  B.USER_ID ");
	statement.append("  , B.LOGIN_NAME ");
	statement.append("  , B.FIRST_NAME ");
	statement.append("  , B.LAST_NAME ");
	statement.append("  , D.POSITION  ");
	statement.append("FROM ");
	statement.append("  TURBINE_USER_GROUP_ROLE AS A  ");
	statement.append("  LEFT JOIN TURBINE_USER AS B  ");
	statement.append("    ON A.USER_ID = B.USER_ID  ");
	statement.append("  LEFT JOIN TURBINE_GROUP AS C  ");
	statement.append("    ON A.GROUP_ID = C.GROUP_ID  ");
	statement.append("  LEFT JOIN EIP_M_USER_POSITION AS D  ");
	statement.append("    ON A.USER_ID = D.USER_ID  ");
	statement.append("WHERE ");
	statement.append("  B.USER_ID > 3  ");
	statement.append("  AND B.DISABLED = 'F'  ");
	statement.append("  AND C.GROUP_ALIAS_NAME IN ( " );

	for (int i = 0; i < groupList.size(); i++) {
		if (i == 0) {
		statement.append("'" + groupList.get(i).getName() +"'");
		} else {
			statement.append(",'" + groupList.get(i).getName() +"'");
		}
	}

	statement.append(" ) ");
	statement.append("  AND B.POSITION_ID IN (  ");
	statement.append("    SELECT ");
	statement.append("      POSITION_ID  ");
	statement.append("    FROM ");
	statement.append("      EIP_M_POSITION  ");
	statement.append("    WHERE ");
	statement.append("      SUPERIOR_FLG = '1' ");
	statement.append("  )  ");
	}


    String query = statement.toString();

    try {
      List<TurbineUser> list2 =
        Database
          .sql(TurbineUser.class, query)
          //.param("groupName", groupname)
          .fetchList();

      ALEipUser user;
      for (TurbineUser tuser : list2) {
        user = new ALEipUser();
        user.initField();
        user.setUserId(tuser.getUserId());
        user.setName(tuser.getLoginName());
        user.setAliasName(tuser.getFirstName(), tuser.getLastName());
        list.add(user);
      }
    } catch (Throwable t) {
      logger.error("[OpinionBoxUtils]", t);
    }

    return list;
  }


  /**
   * ユーザーの所属する部署の一覧を取得します。
   *
   * @param uid
   *          ユーザーID
   * @return 所属する部署リスト
   */
  public static List<UserGroupLiteBean> getPostBeanList(int uid) {
    SelectQuery<TurbineUserGroupRole> query =
      Database.query(TurbineUserGroupRole.class);
    Expression exp1 =
      ExpressionFactory.matchExp(
        TurbineUserGroupRole.TURBINE_USER_PROPERTY,
        Integer.valueOf(uid));
    Expression exp2 =
      ExpressionFactory.greaterExp(
        TurbineUserGroupRole.TURBINE_GROUP_PROPERTY,
        Integer.valueOf(3));
    Expression exp3 =
      ExpressionFactory.matchExp(TurbineUserGroupRole.TURBINE_GROUP_PROPERTY
        + "."
        + TurbineGroup.OWNER_ID_PROPERTY, Integer.valueOf(1));
    query.setQualifier(exp1);
    query.andQualifier(exp2);
    query.andQualifier(exp3);
    List<TurbineUserGroupRole> list = query.fetchList();

    if (list == null || list.size() < 0) {
      return null;
    }

    List<UserGroupLiteBean> resultList = new ArrayList<UserGroupLiteBean>();

    TurbineGroup group = null;
    UserGroupLiteBean bean = null;
    for (TurbineUserGroupRole ugr : list) {
      group = ugr.getTurbineGroup();
      bean = new UserGroupLiteBean();
      bean.initField();
      bean.setGroupId(group.getName());
      bean.setName(group.getGroupAliasName());
      resultList.add(bean);
    }

    return resultList;
  }



}
