/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.opinionbox;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBox;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBoxMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.opinionbox.util.OpinionBoxUtils;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 報告書検索データを管理するクラスです。 <BR>
 *
 */
public class OpinionBoxSelectData extends
    ALAbstractSelectData<OriTOpinionBox, OriTOpinionBox> implements ALData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(OpinionBoxSelectData.class.getName());

  /** サブメニュー（送信） */
  public static final String SUBMENU_CREATED = "created";

  /** サブメニュー（受信） */
  public static final String SUBMENU_REQUESTED = "requested";

  /** サブメニュー（全て） */
  public static final String SUBMENU_ALL = "all";

  /** 親レポートオブジェクト */
  private Object parentOpinionBox;

  /** 子レポートオブジェクト */
  private List<OpinionBoxResultData> coOpinionBoxList;

  /** 現在選択されているサブメニュー */
  private String currentSubMenu;

  /** 返信フォーム表示の有無（トピック詳細表示） */
  private boolean showReplyForm = false;

  private ALEipUser login_user;

  /** 他ユーザーの報告書の閲覧権限 */
  private boolean hasAuthorityOther;

  /** 検索ワード */
  private ALStringField target_keyword;

  /** 現在のユーザ **/
  private int uid;

  /** 報告書作成ユーザ **/
  private int view_uid;

  /** アクセス権限の機能名 */
  private String aclPortletFeature = null;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    uid = ALEipUtils.getUserId(rundata);

    // if (OpinionBoxUtils.hasResetFlag(rundata, context)) {
    // OpinionBoxUtils.clearOpinionBoxSession(rundata, context);
    // }
    login_user = ALEipUtils.getALEipUser(rundata);

    String subMenuParam = rundata.getParameters().getString("submenu");
    currentSubMenu = ALEipUtils.getTemp(rundata, context, "submenu");
    if (subMenuParam == null && currentSubMenu == null) {
      ALEipUtils.setTemp(rundata, context, "submenu", SUBMENU_REQUESTED);
      currentSubMenu = SUBMENU_REQUESTED;
    } else if (subMenuParam != null) {
      ALEipUtils.setTemp(rundata, context, "submenu", subMenuParam);
      currentSubMenu = subMenuParam;
    }

    String sort = ALEipUtils.getTemp(rundata, context, LIST_SORT_STR);
    String sorttype = ALEipUtils.getTemp(rundata, context, LIST_SORT_TYPE_STR);

    if (sort == null || sort.equals("")) {
      ALEipUtils.setTemp(rundata, context, LIST_SORT_STR, "create_date");
    }

    if ("create_date".equals(ALEipUtils
      .getTemp(rundata, context, LIST_SORT_STR))
      && (sorttype == null || "".equals(sorttype))) {
      ALEipUtils.setTemp(
        rundata,
        context,
        LIST_SORT_TYPE_STR,
        ALEipConstants.LIST_SORT_TYPE_DESC);
    }

    // 報告書作成ユーザ
    if (rundata.getParameters().getStringKey("clientid") != null) {
      view_uid =
        Integer.parseInt(rundata
          .getParameters()
          .getStringKey("clientid")
          .toString());
    }

    // 報告書通知先に入っているか
    boolean isSelf = OpinionBoxUtils.isSelf(rundata, context);

    // アクセス権限
    if ((!ALEipConstants.MODE_DETAIL.equals(action.getMode()) && (!SUBMENU_ALL
      .equals(currentSubMenu)))
      || isSelf
      || uid == view_uid) {
      aclPortletFeature = ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_SELF;
    } else {
      aclPortletFeature =
        ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_OTHER;
    }

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();
    hasAuthorityOther =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_OTHER,
        ALAccessControlConstants.VALUE_ACL_LIST);

    // hasAuthorityOther = true;
    showReplyForm = true;
    target_keyword = new ALStringField();

    super.init(action, rundata, context);

  }

  /**
   * 一覧データを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */
  @Override
  public ResultList<OriTOpinionBox> selectList(RunData rundata, Context context) {
    try {
      if (OpinionBoxUtils.hasResetFlag(rundata, context)) {
        OpinionBoxUtils.resetFilter(rundata, context, this.getClass().getName());
        target_keyword.setValue("");
      } else {
        target_keyword.setValue(OpinionBoxUtils.getTargetKeyword(rundata, context));
      }
      SelectQuery<OriTOpinionBox> query = getSelectQuery(rundata, context);
      buildSelectQueryForFilter(query, rundata, context);
      buildSelectQueryForListView(query);
      buildSelectQueryForListViewSort(query, rundata, context);

      ResultList<OriTOpinionBox> list = query.getResultList();
      return list;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery<OriTOpinionBox> getSelectQuery(RunData rundata,
      Context context) {

    SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);

    Integer login_user_id =
      Integer.valueOf((int) login_user.getUserId().getValue());
    if ((target_keyword != null) && (!target_keyword.getValue().equals(""))) {
      ALEipUtils.setTemp(rundata, context, LIST_SEARCH_STR, target_keyword
        .getValue());
    } else {
      ALEipUtils.removeTemp(rundata, context, LIST_SEARCH_STR);
    }

    if (ALEipUtils.getTemp(rundata, context, "OpinionBox_Maximize") == "false") {
      // 通常画面
      // 受信したもので未読
      SelectQuery<OriTOpinionBoxMap> q = Database.query(OriTOpinionBoxMap.class);
      Expression exp1 =
        ExpressionFactory.matchExp(
          OriTOpinionBoxMap.USER_ID_PROPERTY,
          login_user_id);
      q.andQualifier(exp1);
      Expression exp2 =
        ExpressionFactory.matchExp(
          OriTOpinionBoxMap.STATUS_PROPERTY,
          OpinionBoxUtils.DB_STATUS_UNREAD);
      q.andQualifier(exp2);
      List<OriTOpinionBoxMap> queryList = q.fetchList();

      List<Integer> resultid = new ArrayList<Integer>();
      for (OriTOpinionBoxMap item : queryList) {
        if (item.getOpinionId() != 0 && !resultid.contains(item.getOpinionId())) {
          resultid.add(item.getOpinionId());
        } else if (!resultid.contains(item.getOpinionId())) {
          resultid.add(item.getOpinionId());
        }
      }
      if (resultid.size() == 0) {
        // 検索結果がないことを示すために-1を代入
        resultid.add(-1);
      }
      Expression ex =
        ExpressionFactory.inDbExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, resultid);
      query.andQualifier(ex);

    } else if (SUBMENU_CREATED.equals(currentSubMenu)) {
      // 送信
      Expression exp1 =
        ExpressionFactory.matchExp(OriTOpinionBox.USER_ID_PROPERTY, login_user_id);
      query.andQualifier(exp1);

    } else if (SUBMENU_REQUESTED.equals(currentSubMenu)) {
      // 受信
      SelectQuery<OriTOpinionBoxMap> q = Database.query(OriTOpinionBoxMap.class);
      Expression exp1 =
        ExpressionFactory.matchExp(
          OriTOpinionBoxMap.USER_ID_PROPERTY,
          login_user_id);
      q.andQualifier(exp1);
      List<OriTOpinionBoxMap> queryList = q.fetchList();

      List<Integer> resultid = new ArrayList<Integer>();
      for (OriTOpinionBoxMap item : queryList) {
        if (item.getOpinionId() != 0 && !resultid.contains(item.getOpinionId())) {
          resultid.add(item.getOpinionId());
        } else if (!resultid.contains(item.getOpinionId())) {
          resultid.add(item.getOpinionId());
        }
      }
      if (resultid.size() == 0) {
        // 検索結果がないことを示すために-1を代入
        resultid.add(-1);
      }
      Expression exp3 =
        ExpressionFactory.inDbExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, resultid);
      query.andQualifier(exp3);
    } else if (SUBMENU_ALL.equals(currentSubMenu)) {
      // 全て
    }

    // 検索

    String search = ALEipUtils.getTemp(rundata, context, LIST_SEARCH_STR);

    if (search != null && !search.equals("")) {
      current_search = search;
      Expression ex1 =
        ExpressionFactory.likeExp(OriTOpinionBox.OPINION_BOX_NAME_PROPERTY, "%"
          + search
          + "%");
      Expression ex2 =
        ExpressionFactory.likeExp(OriTOpinionBox.NOTE_PROPERTY, "%" + search + "%");
      SelectQuery<OriTOpinionBox> q = Database.query(OriTOpinionBox.class);
      q.andQualifier(ex1.orExp(ex2));
      List<OriTOpinionBox> queryList = q.fetchList();
      List<Integer> resultid = new ArrayList<Integer>();
      for (OriTOpinionBox item : queryList) {
        if (item.getParentId() != 0 && !resultid.contains(item.getParentId())) {
          resultid.add(item.getParentId());
        } else if (!resultid.contains(item.getOpinionId())) {
          resultid.add(item.getOpinionId());
        }
      }
      if (resultid.size() == 0) {
        // 検索結果がないことを示すために-1を代入
        resultid.add(-1);
      }
      Expression ex3 =
        ExpressionFactory.inDbExp(OriTOpinionBox.OPINION_ID_PK_COLUMN, resultid);
      query.andQualifier(ex3);
    }

    // replyを除く
    Expression ex =
      ExpressionFactory.noMatchExp(OriTOpinionBox.OPINION_BOX_NAME_PROPERTY, "");
    query.andQualifier(ex);
    return query;

  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   *
   * @param obj
   * @return
   */
  @Override
  protected Object getResultData(OriTOpinionBox record) {
    try {
      OpinionBoxResultData rd = new OpinionBoxResultData();
      rd.initField();
      rd.setOpinionId(record.getOpinionId().intValue());
      rd.setTokumeiFlg(record.getTokumeiFlg());
      rd.setOpinionBoxName(record.getOpinionBoxName());
      rd.setCreateDate(record.getCreateDate());
      ALEipUser client = ALEipUtils.getALEipUser(record.getUserId().intValue());
      rd.setClientName(client.getAliasName().getValue());
      rd.setClientId(client.getUserId().getValue());
      // 自身の報告書かを設定する
      Integer login_user_id =
        Integer.valueOf((int) login_user.getUserId().getValue());
      rd.setIsSelfOpinionBox(record.getUserId().intValue() == login_user_id
        .intValue());
      return rd;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 詳細データを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */
  @Override
  public OriTOpinionBox selectDetail(RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    OriTOpinionBox request = OpinionBoxUtils.getOriTOpinionBox(rundata, context);

    return request;
  }

  /**
   * 詳細表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FASLE 失敗
   */
  @Override
  public boolean doViewDetail(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);
      doCheckAclPermission(
        rundata,
        context,
        ALAccessControlConstants.VALUE_ACL_DETAIL);
      action.setMode(ALEipConstants.MODE_DETAIL);
      List<OriTOpinionBox> aList = selectDetailList(rundata, context);
      if (aList != null) {
        coOpinionBoxList = new ArrayList<OpinionBoxResultData>();
        int size = aList.size();
        for (int i = 0; i < size; i++) {
          coOpinionBoxList
            .add((OpinionBoxResultData) getResultDataDetail(aList.get(i)));
        }
      }

      action.setResultData(this);
      action.putData(rundata, context);
      return true;
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   * 詳細データを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */
  public List<OriTOpinionBox> selectDetailList(RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    String opinionid =
      ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);

    if (opinionid == null || Integer.valueOf(opinionid) == null) {
      // トピック ID が空の場合
      logger.debug("[OpinionBoxTopic] Empty ID...");
      throw new ALPageNotFoundException();
    }

    String coopinionboxsort =
      ALEipUtils
        .getPortlet(rundata, context)
        .getPortletConfig()
        .getInitParameter("p2b-sort");

    try {
      parentOpinionBox =
        getResultDataDetail(OpinionBoxUtils.getOriTOpinionBoxParentReply(
          rundata,
          context,
          false));

      SelectQuery<OriTOpinionBox> query =
        getSelectQueryForCoopinionbox(rundata, context, opinionid, coopinionboxsort);
      /** 詳細画面は全件表示する */
      // buildSelectQueryForListView(query);
      if ("response_new".equals(coopinionboxsort)) {
        query.orderDesending(OriTOpinionBox.CREATE_DATE_PROPERTY);
      } else {
        query.orderAscending(OriTOpinionBox.CREATE_DATE_PROPERTY);
      }

      List<OriTOpinionBox> resultList = query.fetchList();

      // 表示するカラムのみデータベースから取得する．
      return resultList;
    } catch (ALPageNotFoundException pageNotFound) {
      // logger.error("[OpinionBoxSelectData]", pageNotFound);
      throw pageNotFound;
    } catch (Exception ex) {
      logger.error("[OpinionBoxSelectData]", ex);
      throw new ALDBErrorException();
    }
  }

  private SelectQuery<OriTOpinionBox> getSelectQueryForCoopinionbox(RunData rundata,
      Context context, String opinionid, String coopinionboxsort) {
    SelectQuery<OriTOpinionBox> query = Database.query(OriTOpinionBox.class);
    Expression exp =
      ExpressionFactory.matchExp(OriTOpinionBox.PARENT_ID_PROPERTY, Integer
        .valueOf(opinionid));
    query.setQualifier(exp);
    query.distinct(true);
    return query;
  }

  /**
   * ResultData に値を格納して返します。（詳細データ） <BR>
   *
   * @param obj
   * @return
   */
  @Override
  protected Object getResultDataDetail(OriTOpinionBox obj)
      throws ALPageNotFoundException, ALDBErrorException {

    try {

      OriTOpinionBox record = obj;
      OpinionBoxDetailResultData rd = new OpinionBoxDetailResultData();
      rd.initField();
      rd.setUserId(record.getUserId().longValue());
      rd.setOpinionBoxName(record.getOpinionBoxName());
      rd.setOpinionId(record.getOpinionId().longValue());
      rd.setTokumeiFlg(record.getTokumeiFlg());
      rd.setNote(record.getNote());
      ALEipUser client = ALEipUtils.getALEipUser(record.getUserId().intValue());
      rd.setClientName(client.getAliasName().getValue());
      // 自身の報告書かを設定する
      Integer login_user_id =
        Integer.valueOf((int) login_user.getUserId().getValue());
      rd.setIsSelfOpinionBox(record.getUserId().intValue() == login_user_id);

      List<Integer> users = new ArrayList<Integer>();
      OriTOpinionBoxMap map = null;
      List<OriTOpinionBoxMap> tmp_maps = OpinionBoxUtils.getOriTOpinionBoxMap(record);
      HashMap<Integer, String> statusList = new HashMap<Integer, String>();

      if (record.getParentId().intValue() == 0) {
        int size = tmp_maps.size();
        for (int i = 0; i < size; i++) {
          map = tmp_maps.get(i);
          users.add(map.getUserId());
          if (map.getUserId().intValue() == login_user_id) {
            // 既読に変更する
            map.setStatus(OpinionBoxUtils.DB_STATUS_READ);
            Database.commit();
          }
          statusList.put(map.getUserId(), map.getStatus());
        }
        rd.setStatusList(statusList);
        SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
        Expression exp =
          ExpressionFactory.inDbExp(TurbineUser.USER_ID_PK_COLUMN, users);
        query.setQualifier(exp);
        rd.setMapList(ALEipUtils.getUsersFromSelectQuery(query));

      }


      rd.setCreateDate(record.getCreateDate());
      rd.setUpdateDate(record.getUpdateDate());
      // rd.setCreateDate(OpinionBoxUtils.translateDate(
      // record.getCreateDate(),
      // "yyyy年M月d日H時m分"));
      // rd.setUpdateDate(OpinionBoxUtils.translateDate(
      // record.getUpdateDate(),
      // "yyyy年M月d日H時m分"));

      return rd;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * @return
   *
   */
  @Override
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();
    map.putValue("opinionbox_name", OriTOpinionBox.OPINION_BOX_NAME_PROPERTY);
    map.putValue("create_date", OriTOpinionBox.CREATE_DATE_PROPERTY);
    map.putValue("user_id", OriTOpinionBox.USER_ID_PROPERTY);
    map.putValue("parent_id", OriTOpinionBox.PARENT_ID_PROPERTY);
    return map;
  }

  /**
   * 現在選択されているサブメニューを取得します。 <BR>
   *
   * @return
   */
  public String getCurrentSubMenu() {
    return this.currentSubMenu;
  }

  public ALEipUser getLoginUser() {
    return login_user;
  }

  public boolean showReplyForm() {
    return showReplyForm;
  }

  public List<OpinionBoxResultData> getCoOpinionBoxList() {
    return coOpinionBoxList;
  }

  public Object getParentOpinionBox() {
    return parentOpinionBox;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    // return ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_SELF;
    return aclPortletFeature;
  }

  public boolean hasAuthorityOther() {
    return hasAuthorityOther;
  }

  /**
   * @return target_keyword
   */
  public ALStringField getTargetKeyword() {
    return target_keyword;
  }
}
