/*
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.lunchorder;

import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTLunchOrder;
import com.aimluck.eip.cayenne.om.portlet.AvzTLunchOrderItem;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.lunchorder.util.LunchOrderUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 注文表データを管理するクラスです。 <BR>
 * 
 */
public class LunchOrderInputFormSelectData extends
    ALAbstractSelectData<AvzTLunchOrderItem, Object> implements ALData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(LunchOrderInputFormSelectData.class
      .getName());

  /** 選択された注文先弁当屋の注文制約ステイタス * */
  private String status;

  /** 選択された注文先弁当屋の総注文数 * */
  private int totalSum;

  /** 選択された注文先弁当屋の総注金額 * */
  private int totalPrice;

  /** 選択された注文先弁当屋名 * */
  private ALStringField storeName;

  /** 選択された注文先弁当屋の最低金額or最低注文数の内容表示 * */
  private String minComment;

  /** 選択された注文先弁当屋のコメント * */
  private ALStringField storeComment;

  /** <code>MIN_NUM</code> 最低注文条件変数の識別子 */
  public static final String MIN_NUM = "N";

  /** <code>MIN_PRICE</code> 最低注文条件変数の識別子 */
  public static final String MIN_PRICE = "P";

  /** <code>NONE_CONDITION</code> 最低注文条件変数の識別子 */
  public static final String NONE_CONDITION = "F";

  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractSelectData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
  }

  @Override
  public void initField() {
    // 弁当屋名
    storeName = new ALStringField();
    // コメント
    storeComment = new ALStringField();

  }

  /**
   * 注文表データを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   * @see com.aimluck.eip.common.ALAbstractListData#selectData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  public ResultList<AvzTLunchOrderItem> selectList(RunData rundata,
      Context context) {
    try {
      getStoreDetail(Integer.parseInt(ALEipUtils.getTemp(
        rundata,
        context,
        ALEipConstants.ENTITY_ID)));// 弁当屋詳細情報取得

      String storeMapIdKey = "com.aimluck.eip.lunchorder.storeMapIdKey";// セッションに保存する際にしようするKey名

      ALEipUtils.setTemp(rundata, context, storeMapIdKey, ALEipUtils.getTemp(
        rundata,
        context,
        ALEipConstants.ENTITY_ID));// 選択された注文先の店Idをセッションに保存

      SelectQuery<AvzTLunchOrderItem> query = getSelectQuery(rundata, context);
      buildSelectQueryForListView(query);
      query.orderAscending("itemTitle");
      ResultList<AvzTLunchOrderItem> list = query.getResultList();

      return list;

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }

  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery<AvzTLunchOrderItem> getSelectQuery(RunData rundata,
      Context context) {

    Expression exp01 =
      ExpressionFactory.matchDbExp(// 弁当屋ごとの条件(選択された弁当屋に絞る)
        AvzTLunchOrderItem.TO_AVZ_TLUNCH_ORDER_PROPERTY
          + "."
          + AvzTLunchOrder.ORDER_MAP_ID_PK_COLUMN,
        (ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID)));

    SelectQuery<AvzTLunchOrderItem> query =
      Database.query(AvzTLunchOrderItem.class, exp01);

    return query;
  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   * 
   * @param obj
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getListData(java.lang.Object)
   */
  @Override
  protected Object getResultData(AvzTLunchOrderItem obj) {
    try {

      AvzTLunchOrderItem itemRecord = obj;

      LunchOrderInputFormResultData rd = new LunchOrderInputFormResultData();// インスタンス生成

      rd.initField(); // オブジェクトのフィールドを初期化
      rd.setItemId(itemRecord.getItemId());// IDセット
      rd.setItemName(itemRecord.getItemTitle());// 弁当名セット
      rd.setItemPrice(itemRecord.getPrice());// 価格セット
      rd.setComment(itemRecord.getComment());// コメントセット

      return rd;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  public void getStoreDetail(int storeMapId) {
    Expression exp01 = ExpressionFactory.matchDbExp(// 弁当屋ごとの条件(選択された弁当屋に絞る)
      AvzTLunchOrder.ORDER_MAP_ID_PK_COLUMN,
      storeMapId);

    SelectQuery<AvzTLunchOrder> query =
      Database.query(AvzTLunchOrder.class, exp01);

    ResultList<AvzTLunchOrder> list = query.getResultList();

    AvzTLunchOrder avzTLunchOrder = list.get(0);

    status = LunchOrderUtils.getStatus(storeMapId);// 注文表で表示する属性を取得
    totalSum = LunchOrderUtils.getTotalSum(storeMapId);// 選択された注文先弁当屋の総注文数を取得
    totalPrice = LunchOrderUtils.getTotalPrice(storeMapId);// 合計金額を変数に入れる
    storeComment.setValue(avzTLunchOrder.getComment());// 選択された注文先弁当屋のコメントを取得

    if (avzTLunchOrder.getMinStatus().equals(MIN_NUM)) {// 最低注文数OR最低注文金額を取得。どちらも無い場合は””を設定。

      int num = avzTLunchOrder.getMinNum();
      minComment = "最低注文数は「" + num + "個」です";

    } else if (avzTLunchOrder.getMinStatus().equals(MIN_PRICE)) {

      int price = avzTLunchOrder.getMinPlice();
      minComment = "最低注文金額は「" + price + "円」です";

    } else {
      minComment = "";
    }
    storeName.setValue(avzTLunchOrder.getStoreName());// 店舗名

  }

  /**
   * 詳細データを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#selectDetail(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  public Object selectDetail(RunData rundata, Context context)
      throws ALPageNotFoundException {
    return null;
  }

  /**
   * ResultData に値を格納して返します。（詳細データ） <BR>
   * 
   * @param obj
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getResultDataDetail(java.lang.Object)
   */
  @Override
  protected Object getResultDataDetail(Object obj) {
    return null;
  }

  /**
   * @return
   * @velocityで店舗名取得メソッド
   */
  public ALStringField getStoreName() {
    return storeName;
  }

  /**
   * @return
   * @velocityで状態取得メソッド
   */
  public String getStatus() {
    return status;
  }

  /**
   * @return
   * @velocityで最低条件内容取得メソッド
   */
  public String getMinComment() {
    return minComment;
  }

  /**
   * @return
   * @velocityで合計金額取得メソッド
   */
  public int getTotalPrice() {
    return totalPrice;
  }

  /**
   * @return
   * @velocityで合計注文数取得メソッド
   */
  public int getTotalSum() {
    return totalSum;
  }

  /**
   * @return
   * @velocityで店舗コメント取得メソッド
   */
  public ALStringField getStoreComment() {
    return storeComment;
  }

  /**
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getColumnMap()
   */
  @Override
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();
    return map;
  }
}
