/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeFile;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeTopic;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;
import com.aimluck.eip.fileupload.beans.FileuploadLiteBean;
import com.aimluck.eip.fileupload.util.FileuploadUtils;
import com.aimluck.eip.improveope.util.ImproveopeUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.modules.actions.improveope.ImproveopeAction;
import com.aimluck.eip.modules.screens.ImproveopeTopicDetailScreen;
import com.aimluck.eip.modules.screens.ImproveopeTopicFormJSONScreen;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.services.storage.ALStorageService;
import com.aimluck.eip.util.ALEipUtils;
//import com.aimluck.eip.cayenne.om.portlet.EipTMsgboardFile;
//import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeTopic;

/**
 * 掲示板返信のフォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopeTopicReplyFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopeTopicReplyFormData.class.getName());

  /** メモ */
  private ALStringField note;

  /** 添付ファイル */
  private ALStringField attachment = null;

  /** 添付ファイルリスト */
  private List<FileuploadLiteBean> fileuploadList = null;

  /** 添付フォルダ名 */
  private String folderName = null;

  private int uid;

  private String orgId;

  /** アクセス権限の機能名 */
  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclCategoryList;

  /** 他ユーザーの作成したトピックの編集権限 */
  private boolean hasAclUpdateTopicOthers;

  /** 他ユーザーの作成したトピックの削除権限 */
  private boolean hasAclDeleteTopicOthers;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();

    folderName = rundata.getParameters().getString("folderName");

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclCategoryList =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF,
        ALAccessControlConstants.VALUE_ACL_LIST);

    hasAclDeleteTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_DELETE);

    hasAclUpdateTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_UPDATE);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {
    // メモ
    note = new ALStringField();
    note.setFieldName("内容");
    note.setTrim(false);
    // Attachment
    attachment = new ALStringField();
    attachment.setFieldName("添付ファイル");
    attachment.setTrim(true);

    fileuploadList = new ArrayList<FileuploadLiteBean>();
  }

  /**
   * 掲示板の各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // メモ必須項目
    note.setNotNull(true);
    // メモの文字数制限
    note.limitMaxLength(10000);
  }

  /**
   * トピックのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // メモ
    note.validate(msgList);
    return (msgList.size() == 0);
  }

  /**
   * トピックをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    return false;
  }

  /**
   * 返信記事をデータベースから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    try {
      String topicid = rundata.getParameters().getString("topic_reply_id");

      // オブジェクトモデルを取得
      OriTImproveopeTopic topic;

      if (this.hasAclDeleteTopicOthers) {
        topic =
          ImproveopeUtils.getOriTImproveopeTopicReply(
            rundata,
            context,
            topicid,
            true);
      } else {
        topic =
          ImproveopeUtils.getOriTImproveopeTopicReply(
            rundata,
            context,
            topicid,
            false);
      }
      if (topic == null) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[ImproveopeTopicReplyFormData] Not found ID...");
        throw new ALPageNotFoundException();
      }

      List<String> fpaths = new ArrayList<String>();
      List<?> files = topic.getOriTImproveopeFile();
      if (files != null && files.size() > 0) {
        int fsize = files.size();
        for (int i = 0; i < fsize; i++) {
          fpaths.add(((OriTImproveopeFile) files.get(i)).getFilePath());
        }
      }

      // 返信記事を削除
      // 添付ファイルはカスケードで自動的に削除される．
      Database.delete(topic);
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        topic.getTopicId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        topic.getTopicName());

      if (fpaths.size() > 0) {
        // ローカルファイルに保存されているファイルを削除する．
        int fsize = fpaths.size();
        for (int i = 0; i < fsize; i++) {
          ALStorageService.deleteFile(ImproveopeUtils.getSaveDirPath(orgId, uid)
            + fpaths.get(i));
        }
      }
    } catch (Exception e) {
      Database.rollback();
      logger.error("[ImproveopeTopicReplyFormData]", e);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * トピックをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    try {
      // オブジェクトモデルを取得
    	OriTImproveopeTopic parenttopic =
        ImproveopeUtils.getOriTImproveopeParentTopic(rundata, context, false);
      if (parenttopic == null) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[ImproveopeTopicReplyFormData] Not found ID...");
        throw new ALPageNotFoundException();
      }

      Date updateDate = Calendar.getInstance().getTime();

      // 新規オブジェクトモデル
      OriTImproveopeTopic topic = Database.create(OriTImproveopeTopic.class);

      // トピック名
      topic.setTopicName("");
      // 親トピック ID
      topic.setParentId(parenttopic.getTopicId());
      // ユーザーID
      topic.setOwnerId(Integer.valueOf(uid));
      // メモ
      topic.setNote(note.getValue());
      // 作成者
      topic.setCreateUserId(Integer.valueOf(uid));
      // 更新者
      topic.setUpdateUserId(Integer.valueOf(uid));
      // 作成日
      topic.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      topic.setUpdateDate(updateDate);

      // 親トピックの更新情報を更新する．
      parenttopic.setUpdateUserId(Integer.valueOf(uid));
      parenttopic.setUpdateDate(updateDate);

      // ファイルをデータベースに登録する．
      if (!ImproveopeUtils.insertFileDataDelegate(
        rundata,
        context,
        topic,
        fileuploadList,
        folderName,
        msgList)) {
        return false;
      }

      Database.commit();

      // 投稿コメントポイント付与
      ImproveopeUtils.grantTopicCommentPoint(topic);


      List<ALEipUser> memberList = new ArrayList<ALEipUser>();
      memberList = selectMsgMember(rundata, context);

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        topic.getTopicId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        parenttopic.getTopicName(),
        "insert");

      /* 自分以外の全員に新着ポートレット登録 */
      // アクティビティ
      ALEipUser user = ALEipUtils.getALEipUser(uid);
      // 更新情報
      ImproveopeUtils.createNewCommentActivity(parenttopic, user
        .getName()
        .getValue());

      // あなた宛のお知らせ
      List<String> recipient = new ArrayList<String>();
      for (int i = 0; i < memberList.size(); i++) {
        recipient.add(memberList.get(i).getName().toString());
      }
      ImproveopeUtils.createNewTopicActivity(parenttopic, user
        .getName()
        .toString(), recipient);


      // 添付ファイル保存先のフォルダを削除
      ALStorageService.deleteTmpFolder(uid, folderName);

      // メール送信
//      try {
//        int msgType =
//          ALMailUtils.getSendDestType(ALMailUtils.KEY_MSGTYPE_MSGBOARD);
//        if (msgType > 0) {
//          // パソコンへメールを送信
//          List<ALEipUserAddr> destMemberList =
//            ALMailUtils.getALEipUserAddrs(memberList, ALEipUtils
//              .getUserId(rundata), false);
//          String subject = "[" + ALOrgUtilsService.getAlias() + "]業務提案の掲示板";
//          String orgId = Database.getDomainName();
//
//          List<ALAdminMailMessage> messageList =
//            new ArrayList<ALAdminMailMessage>();
//          for (ALEipUserAddr destMember : destMemberList) {
//            ALAdminMailMessage message = new ALAdminMailMessage(destMember);
//            message.setPcSubject(subject);
//            message.setCellularSubject(subject);
//            message.setPcBody(ImproveopeUtils.createReplyMsgForPc(
//              rundata,
//              topic,
//              parenttopic));
//            message.setCellularBody(ImproveopeUtils.createReplyMsgForPc(
//              rundata,
//              topic,
//              parenttopic));
//            messageList.add(message);
//          }
//          ALMailService.sendAdminMailAsync(new ALAdminMailContext(
//            orgId,
//            ALEipUtils.getUserId(rundata),
//            messageList,
//            ALMailUtils.getSendDestType(ALMailUtils.KEY_MSGTYPE_MSGBOARD)));
//        }
//      } catch (Exception ex) {
//        msgList.add("メールを送信できませんでした。");
//        logger.error("Exception", ex);
//        return false;
//      }

    } catch (Exception e) {
      logger.error("[ImproveopeTopicReplyFormData]", e);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * 自分以外のトピック関係者のuser_idのListを習得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   * @throws ALDBErrorException
   * @throws ALPageNotFoundException
   */
  private List<ALEipUser> selectMsgMember(RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    // 関連トピックをすべて取得する
    OriTImproveopeTopic parenttopic =
      ImproveopeUtils.getOriTImproveopeParentTopic(rundata, context, false);
    SelectQuery<OriTImproveopeTopic> topicQuery =
      Database.query(OriTImproveopeTopic.class);
    Expression topicExp =
      ExpressionFactory.matchExp(
        OriTImproveopeTopic.PARENT_ID_PROPERTY,
        parenttopic.getTopicId());
    topicQuery.setQualifier(topicExp);

    // トピックに関連する全てのユーザーIDを取得する
    List<OriTImproveopeTopic> topicList = topicQuery.fetchList();
    topicList.add(parenttopic);
    List<Integer> userIdList = new ArrayList<Integer>();
    for (OriTImproveopeTopic topic : topicList) {
      Integer userId = topic.getCreateUserId();
      if (!userId.equals(uid) && !userIdList.contains(userId)) {
        userIdList.add(userId);
      }
    }

    // ユーザーIDからユーザー情報を取得する。
    SelectQuery<TurbineUser> userQuery = Database.query(TurbineUser.class);
    Expression userExp =
      ExpressionFactory.inDbExp(TurbineUser.USER_ID_PK_COLUMN, userIdList);
    userQuery.setQualifier(userExp);
    return ALEipUtils.getUsersFromSelectQuery(userQuery);

  }

  /**
   * データベースに格納されているトピックを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    return false;
  }

  /**
   * トピック詳細表示ページからデータを新規登録します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doInsert(ALAction action, RunData rundata, Context context) {
    try {
      if (!doCheckSecurity(rundata, context)) {
        return false;
      }
      init(action, rundata, context);

      doCheckAclPermission(
        rundata,
        context,
        ALAccessControlConstants.VALUE_ACL_INSERT);

      action.setMode(ALEipConstants.MODE_INSERT);
      List<String> msgList = new ArrayList<String>();
      setValidator();
      boolean res =
        (setFormData(rundata, context, msgList) && validate(msgList) && insertFormData(
          rundata,
          context,
          msgList));
      if (!res) {
        action.setMode(ALEipConstants.MODE_NEW_FORM);
        setMode(action.getMode());
      }
      if (action instanceof ImproveopeTopicFormJSONScreen) {
        action.setResultData(this);
        action.addErrorMessages(msgList);
        action.putData(rundata, context);
      } else {
        ImproveopeAction improveopeAction = (ImproveopeAction) action;
        improveopeAction.setResultDataOnTopicDetail(this);
        improveopeAction.addErrorMessagesOnTopicDetail(msgList);
        improveopeAction.putDataOnTopicDetail(rundata, context);
      }
      return res;
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   * トピック詳細表示ページにフォームを表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doViewForm(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);

      // doCheckAclPermission(rundata, context,
      // ALAccessControlConstants.VALUE_ACL_INSERT);

      action.setMode("reply");
      // mode = action.getMode();
      List<String> msgList = new ArrayList<String>();
      boolean res = setFormData(rundata, context, msgList);
      if (action instanceof ImproveopeTopicDetailScreen) {
        ImproveopeTopicDetailScreen improveopeAction =
          (ImproveopeTopicDetailScreen) action;
        improveopeAction.setResultDataOnTopicDetail(this);
        improveopeAction.addErrorMessagesOnTopicDetail(msgList);
        improveopeAction.putDataOnTopicDetail(rundata, context);
      } else {
        ImproveopeAction improveopeAction = (ImproveopeAction) action;
        improveopeAction.setResultDataOnTopicDetail(this);
        improveopeAction.addErrorMessagesOnTopicDetail(msgList);
        improveopeAction.putDataOnTopicDetail(rundata, context);
      }
      return res;
      // } catch (ALPermissionException e) {
      // ALEipUtils.redirectPermissionError(rundata);
      // return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    try {
      fileuploadList = FileuploadUtils.getFileuploadList(rundata);
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    return res;
  }

  public void setAclPortletFeature(String featureName) {
    aclPortletFeature = featureName;
  }

  /**
   * メモを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  public List<FileuploadLiteBean> getAttachmentFileNameList() {
    return fileuploadList;
  }

  public String getFolderName() {
    return folderName;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    if (aclPortletFeature == null || "".equals(aclPortletFeature)) {
      return ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF;
    } else {
      return aclPortletFeature;
    }
  }

  /**
   * 他ユーザのトピックを編集する権限があるかどうかを返します。
   *
   * @return
   */

  public boolean hasAclUpdateTopicOthers() {
    return hasAclUpdateTopicOthers;
  }

  /**
   * 他ユーザのトピックを削除する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclDeleteTopicOthers() {
    return hasAclDeleteTopicOthers;
  }
}
