/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeFile;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeTopic;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.fileupload.beans.FileuploadLiteBean;
import com.aimluck.eip.improveope.util.ImproveopeUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.services.storage.ALStorageService;
import com.aimluck.eip.timeline.util.TimelineUtils;
import com.aimluck.eip.util.ALEipUtils;
//import com.aimluck.eip.cayenne.om.portlet.EipTMsgboardCategory;
//import com.aimluck.eip.cayenne.om.portlet.EipTMsgboardCategoryMap;
//import com.aimluck.eip.cayenne.om.portlet.EipTMsgboardFile;
//import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeTopic;
//import com.aimluck.eip.cayenne.om.portlet.EipTMsgboardCategory;
//import com.aimluck.eip.cayenne.om.portlet.EipTMsgboardCategoryMap;
//import com.aimluck.eip.cayenne.om.portlet.EipTMsgboardFile;
//import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeTopic;

/**
 * 掲示板トピックのフォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopeTopicFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopeTopicFormData.class.getName());

  /** トピック名 */
  private ALStringField topic_name;

  /** カテゴリ ID */
  private ALNumberField category_id;

  /** カテゴリ名 */
  private ALStringField category_name;

  /** メモ */
  private ALStringField note;

  /** カテゴリ一覧 */
//  private List<ImproveopeCategoryResultData> categoryList;

  /** */
  private boolean is_new_category;

  /** 添付ファイルリスト */
  private List<FileuploadLiteBean> fileuploadList = null;

  /** 添付フォルダ名 */
  private String folderName = null;

  private int uid;

//  private EipTMsgboardCategory category;

  private String orgId;

  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclCategoryList;

  /** 他ユーザーの作成したトピックの編集権限 */
  private boolean hasAclUpdateTopicOthers;

  /** 他ユーザーの作成したトピックの削除権限 */
  private boolean hasAclDeleteTopicOthers;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
    is_new_category = rundata.getParameters().getBoolean("is_new_category");

    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();

    folderName = rundata.getParameters().getString("folderName");

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclCategoryList =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF,
        ALAccessControlConstants.VALUE_ACL_LIST);

    hasAclDeleteTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_DELETE);

    hasAclUpdateTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_UPDATE);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {
    // トピック名
    topic_name = new ALStringField();
    topic_name.setFieldName("タイトル");
    topic_name.setTrim(true);
    // カテゴリID
    category_id = new ALNumberField();
    category_id.setFieldName("カテゴリ");
    // カテゴリ
    category_name = new ALStringField();
    category_name.setFieldName("カテゴリ名");
    // メモ
    note = new ALStringField();
    note.setFieldName("内容");
    note.setTrim(false);

    fileuploadList = new ArrayList<FileuploadLiteBean>();

    aclPortletFeature =
      ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF;
  }

  /**
   *
   * @param rundata
   * @param context
   */
//  public void loadCategoryList(RunData rundata, Context context) {
//    categoryList = ImproveopeUtils.loadCategoryList(rundata);
//  }

  /**
   * 掲示板の各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // トピック名必須項目
    topic_name.setNotNull(true);
    // トピック名の文字数制限
    topic_name.limitMaxLength(50);
    // メモ必須項目
    note.setNotNull(true);
    // メモの文字数制限
    note.limitMaxLength(10000);
    //if (is_new_category) {
      // カテゴリ名必須項目
    //   category_name.setNotNull(true);
      // カテゴリ名文字数制限
    //  category_name.limitMaxLength(50);
    //}
  }

  /**
   * トピックのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // トピック名
    topic_name.validate(msgList);
    // メモ
    note.validate(msgList);

    //if (is_new_category) {
      // カテゴリ名
    //  category_name.validate(msgList);
    //}

    return (msgList.size() == 0);

  }

  /**
   * トピックをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      OriTImproveopeTopic topic =
        ImproveopeUtils.getOriTImproveopeParentTopic(rundata, context, false);
      if (topic == null) {
        return false;
      }

      // トピック名
      topic_name.setValue(topic.getTopicName());

      // 内容
      note.setValue(topic.getNote());
      // ファイル
      SelectQuery<OriTImproveopeFile> query =
        Database.query(OriTImproveopeFile.class);
      query.andQualifier(ExpressionFactory.matchDbExp(
    		  OriTImproveopeFile.ORI_TIMPROVEOPE_TOPIC_PROPERTY,
        topic.getTopicId()));
      List<OriTImproveopeFile> improveopeFileList = query.fetchList();
      for (OriTImproveopeFile file : improveopeFileList) {
        FileuploadLiteBean fbean = new FileuploadLiteBean();
        fbean.initField();
        fbean.setFileId(file.getFileId());
        fbean.setFileName(file.getFileName());
        fileuploadList.add(fbean);
      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * トピックをデータベースから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    try {

      // FIX_ME イベントログのために一度IDと名前を取得
      int parentid =
        Integer.parseInt(ALEipUtils.getTemp(
          rundata,
          context,
          ALEipConstants.ENTITY_ID));

      OriTImproveopeTopic parent =
        Database.get(OriTImproveopeTopic.class, (long) parentid);

      // オブジェクトモデルを取得
      List<OriTImproveopeTopic> list;

      if (this.hasAclDeleteTopicOthers) {
        list =
          ImproveopeUtils.getOriTImproveopeTopicListToDeleteTopic(
            rundata,
            context,
            true);
      } else {
        list =
          ImproveopeUtils.getOriTImproveopeTopicListToDeleteTopic(
            rundata,
            context,
            false);
      }

      if (list == null) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[ImproveopeTopicFormData] Not found List...");
        throw new ALPageNotFoundException();
      }

      List<Integer> topicIdList = new ArrayList<Integer>();
      OriTImproveopeTopic topic;
      int size = list.size();
      for (int i = 0; i < size; i++) {
        topic = list.get(i);
        topicIdList.add(topic.getTopicId());
      }

      // トピックを削除
      SelectQuery<OriTImproveopeTopic> query =
        Database.query(OriTImproveopeTopic.class);
      Expression exp =
        ExpressionFactory.inDbExp(
          OriTImproveopeTopic.TOPIC_ID_PK_COLUMN,
          topicIdList);
      query.setQualifier(exp);

      List<OriTImproveopeTopic> topics = query.fetchList();

      List<String> fpaths = new ArrayList<String>();
      if (topics.size() > 0) {
        int tsize = topics.size();
        for (int i = 0; i < tsize; i++) {
          List<?> files = topics.get(i).getOriTImproveopeFile();
          if (files != null && files.size() > 0) {
            int fsize = files.size();
            for (int j = 0; j < fsize; j++) {
              fpaths.add(((OriTImproveopeFile) files.get(j)).getFilePath());
            }
          }
        }
      }

      Database.deleteAll(topics);
      Database.commit();

      TimelineUtils.deleteTimelineActivity(rundata, context, "Improveope", parent
        .getTopicId()
        .toString());

      if (fpaths.size() > 0) {
        // ローカルファイルに保存されているファイルを削除する．
        int fsize = fpaths.size();
        for (int i = 0; i < fsize; i++) {
          ALStorageService.deleteFile(ImproveopeUtils.getSaveDirPath(orgId, uid)
            + fpaths.get(i));
        }
      }

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        parent.getTopicId(),
        ALEventlogConstants.PORTLET_TYPE_MSGBOARD_TOPIC,
        parent.getTopicName());

    } catch (Exception e) {
      Database.rollback();
      logger.error("[ImproveopeCategorySelectData]", e);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * トピックをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 新規オブジェクトモデル
      OriTImproveopeTopic topic = Database.create(OriTImproveopeTopic.class);
      // トピック名
      topic.setTopicName(topic_name.getValue());
      // 親トピックID
      topic.setParentId(Integer.valueOf(0));

      // ユーザーID
      topic.setOwnerId(Integer.valueOf(uid));
      // メモ
      topic.setNote(note.getValue());
      // 作成者
      topic.setCreateUserId(Integer.valueOf(uid));
      // 更新者
      topic.setUpdateUserId(Integer.valueOf(uid));
      // 作成日
      topic.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      topic.setUpdateDate(Calendar.getInstance().getTime());

      // ファイルをデータベースに登録する．
      if (!ImproveopeUtils.insertFileDataDelegate(
        rundata,
        context,
        topic,
        fileuploadList,
        folderName,
        msgList)) {
        return false;
      }

      Database.commit();

      // 投稿ポイント付与
      ImproveopeUtils.grantTopicPoint(topic);

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        topic.getTopicId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        topic.getTopicName());


      /* 自分以外の全員に新着ポートレット登録 */
      // アクティビティ
      ALEipUser user = ALEipUtils.getALEipUser(uid);
      ImproveopeUtils.createTopicActivity(
        topic,
        user.getName().getValue(),
        true);


      // 添付ファイル保存先のフォルダを削除
      ALStorageService.deleteTmpFolder(uid, folderName);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * トピックカテゴリをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
//  private boolean insertCategoryData(RunData rundata, Context context,
//      List<String> msgList) {
//    try {
//
//      setAclPortletFeature(ALAccessControlConstants.POERTLET_FEATURE_MSGBOARD_CATEGORY);
//      doCheckAclPermission(
//        rundata,
//        context,
//        ALAccessControlConstants.VALUE_ACL_INSERT);
//      setAclPortletFeature(ALAccessControlConstants.POERTLET_FEATURE_MSGBOARD_TOPIC);
//
//      int userid = ALEipUtils.getUserId(rundata);
//
//      TurbineUser tuser =
//        Database.get(TurbineUser.class, Integer.valueOf(userid));
//
//      // 新規オブジェクトモデル
//      category = Database.create(EipTMsgboardCategory.class);
//      // カテゴリ名
//      category.setCategoryName(category_name.getValue());
//      // 公開区分
//      category.setPublicFlag(ImproveopeUtils.PUBLIC_FLG_VALUE_PUBLIC);
//      // カテゴリメモ
//      category.setNote("");
//      // ユーザーID
//      category.setTurbineUser(tuser);
//      // 作成日
//      category.setCreateDate(Calendar.getInstance().getTime());
//      // 更新日
//      category.setUpdateDate(Calendar.getInstance().getTime());
//
//      // マップの登録
//      EipTMsgboardCategoryMap map =
//        Database.create(EipTMsgboardCategoryMap.class);
//      map.setEipTMsgboardCategory(category);
//      map.setUserId(Integer.valueOf(userid));
//      map.setStatus(ImproveopeUtils.STAT_VALUE_OWNER);
//
//      Database.commit();
//
//      // イベントログに保存
//      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
//        category.getCategoryId(),
//        ALEventlogConstants.PORTLET_TYPE_MSGBOARD_CATEGORY,
//        category.getCategoryName());
//
//    } catch (ALPermissionException e) {
//      msgList.add(ALAccessControlConstants.DEF_PERMISSION_ERROR_STR);
//      return false;
//    } catch (Exception ex) {
//      Database.rollback();
//      logger.error("Exception", ex);
//      msgList.add("エラーが発生しました。");
//      return false;
//    }
//    return true;
//  }

  /**
   * データベースに格納されているトピックを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 新規オブジェクトモデル
      OriTImproveopeTopic topic =
        ImproveopeUtils.getOriTImproveopeParentTopic(rundata, context, false);
      // トピック名
      topic.setTopicName(topic_name.getValue());

      // メモ
      topic.setNote(note.getValue());
      // 更新者
      topic.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      topic.setUpdateDate(Calendar.getInstance().getTime());

      // ファイルをデータベースに登録する．
      if (!ImproveopeUtils.insertFileDataDelegate(
        rundata,
        context,
        topic,
        fileuploadList,
        folderName,
        msgList)) {
        return false;
      }

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        topic.getTopicId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        topic.getTopicName());

      /* 自分以外の全員に新着ポートレット登録 */
      // アクティビティ
      ALEipUser user = ALEipUtils.getALEipUser(uid);
      ImproveopeUtils.createTopicActivity(
      topic,
      user.getName().getValue(),
      false);

      // 添付ファイル保存先のフォルダを削除
      ALStorageService.deleteTmpFolder(uid, folderName);
    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    try {
      fileuploadList = ImproveopeUtils.getFileuploadList(rundata);
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    return res;
  }

  /**
   * カテゴリIDを取得します。 <BR>
   *
   * @return
   */
  public ALNumberField getCategoryId() {
    return category_id;
  }

  /**
   * メモを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  /**
   * トピック名を取得します。 <BR>
   *
   * @return
   */
  public ALStringField getTopicName() {
    return topic_name;
  }

  /**
   * カテゴリ一覧を取得します。 <BR>
   *
   * @return
   */
//  public List<ImproveopeCategoryResultData> getCategoryList() {
//    return categoryList;
//  }

  /**
   * @return
   */
  public boolean isNewCategory() {
    return is_new_category;
  }

  /**
   * @return
   */
  public void setIsNewCategory(boolean bool) {
    is_new_category = bool;
  }

  /**
   * カテゴリ名を取得します。
   *
   * @return
   */
  public ALStringField getCategoryName() {
    return category_name;
  }

  public List<FileuploadLiteBean> getAttachmentFileNameList() {
    return fileuploadList;
  }

  public String getFolderName() {
    return folderName;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  public void setAclPortletFeature(String aclPortletFeature) {
    this.aclPortletFeature = aclPortletFeature;
  }

  /**
   * 他ユーザのトピックを編集する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclUpdateTopicOthers() {
    return hasAclUpdateTopicOthers;
  }

  /**
   * 他ユーザのトピックを削除する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclDeleteTopicOthers() {
    return hasAclDeleteTopicOthers;
  }
}
