/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriMPoint;
import com.aimluck.eip.cayenne.om.portlet.OriTPointHistory;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 掲示板トピックのフォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopeGrantFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopeGrantFormData.class.getName());


  /** ポイント名 */
  private ALNumberField point_name;

  /** ポイント */
  private ALStringField point;

  /** 対象者一覧 */
  private List<ALEipUser> mapList;

  /** 対象者ID */
  private int memberid;

  /** ポイントマスタ */
  private List<OriMPoint> pointMstList;

  private int uid;

  private String orgId;

  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclCategoryList;

  /** 他ユーザーの作成したトピックの編集権限 */
  private boolean hasAclUpdateTopicOthers;

  /** 他ユーザーの作成したトピックの削除権限 */
  private boolean hasAclDeleteTopicOthers;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    uid = ALEipUtils.getUserId(rundata);

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclCategoryList =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF,
        ALAccessControlConstants.VALUE_ACL_LIST);

    hasAclDeleteTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_DELETE);

    hasAclUpdateTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_UPDATE);

  }




  @Override
  public void initField() {
  	point_name = new ALNumberField();
  	point_name.setFieldName("ポイント名");

  	point = new ALStringField();
  	point.setCharacterType(ALStringField.TYPE_ASCII);
  	point.setFieldName("ポイント数");

    // 対象者リスト
    mapList = new ArrayList<ALEipUser>();
    // ポイントマスタ
    pointMstList = new ArrayList<OriMPoint>();

  }


  @Override
  protected void setValidator() throws ALPageNotFoundException,
  		ALDBErrorException {

	  point.setNotNull(true);
	  point.setTrim(true);
  }



  @Override
  protected boolean validate(List<String> msgList)
  		throws ALPageNotFoundException, ALDBErrorException {

	 point_name.validate(msgList);

	 if (point_name.getValueAsString().equals("")) {
		 msgList.add("『 <span class='em'>ポイント名</span> 』を選択してください。");
	 }

	 point.validate(msgList);

	 // 対象者
     if (mapList == null || mapList.size() <= 0) {
       msgList.add("『 <span class='em'>付与対象者</span> 』を指定してください。");
     } else if (!(mapList.get(0) instanceof ALEipUser)) {
       msgList.add("付与対象者のユーザーが全て無効、もしくは削除されています。有効なユーザーを選択してください。");
     }

    return (msgList.size() == 0);
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
    	// ポイントマスタの読み込み
    	SelectQuery<OriMPoint> pointMstquery = Database.query(OriMPoint.class);
    	pointMstList = pointMstquery.fetchList();

        String userNames[] = rundata.getParameters().getStrings("members");
        if (userNames != null && userNames.length > 0) {
          SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
          Expression exp =
            ExpressionFactory.inExp(TurbineUser.LOGIN_NAME_PROPERTY, userNames);
          query.setQualifier(exp);
          mapList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
        }

        memberid = rundata.getParameters().getInt("memberid");

    }

    return res;
  }

  @Override
  protected boolean loadFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {
  	// TODO 自動生成されたメソッド・スタブ
  	return false;
  }



  @Override
  protected boolean insertFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {

	  try {

		  // 権限があるばいいのみ実行
		  if (hasAclUpdateTopicOthers) {

		  for (ALEipUser user : mapList) {
			  int userid = (int) user.getUserId().getValue();
			  // 新規オブジェクト
			  OriTPointHistory pointHistory = Database.create(OriTPointHistory.class);
			  // ユーザーID
			  pointHistory.setUserId(userid);
			  // ポイントID
			  pointHistory.setPointId(Integer.valueOf(point_name.getValueAsString()));
			  // ポイント
			  pointHistory.setPoint(Integer.valueOf(point.getValue()));
			  // 作成者
			  pointHistory.setCreateUserId(uid);
			  // 作成時間
			  pointHistory.setCreateDate(Calendar.getInstance().getTime());

		  }

	        Database.commit();

	        // イベントログに保存
	        ALEventlogFactoryService.getInstance().getEventlogHandler().log(
	          0,
	          ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
	          "ポイント付与");
		  }

	      } catch (Exception ex) {
	        logger.error("Exception", ex);
	        return false;
	      }
	      return true;

  }



  @Override
  protected boolean updateFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {
  	// TODO 自動生成されたメソッド・スタブ
  	return false;
  }



  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {
  	// TODO 自動生成されたメソッド・スタブ
  	return false;
  }


  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  public void setAclPortletFeature(String aclPortletFeature) {
    this.aclPortletFeature = aclPortletFeature;
  }

  /**
   * 他ユーザのトピックを編集する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclUpdateTopicOthers() {
    return hasAclUpdateTopicOthers;
  }

  /**
   * 他ユーザのトピックを削除する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclDeleteTopicOthers() {
    return hasAclDeleteTopicOthers;
  }




/**
 * @return point_name
 */
public ALNumberField getPointName() {
	return point_name;
}


/**
 * @return point
 */
public ALStringField getPoint() {
	return point;
}


/**
 * @return mapList
 */
public List<ALEipUser> getMapList() {
	return mapList;
}




/**
 * @return pointMstList
 */
public List<OriMPoint> getPointMstList() {
	return pointMstList;
}

/**
 * @returun memberid
 */
public int getMemberId() {
  return memberid;
}
/**
 * @returun memberid
 */
public String getMemberLoginName() {
  try {
    ALEipUser user = ALEipUtils.getALEipUser(memberid);
    return user.getName().toString();
  } catch (ALDBErrorException e) {
    e.printStackTrace();
  }
  return "";
}
public String getMemberName() {
    try {
      ALEipUser user = ALEipUtils.getALEipUser(memberid);
      return user.getAliasName().toString();
    } catch (ALDBErrorException e) {
      e.printStackTrace();
    }
    return "";
}

}
