/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApply;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApplyFile;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApplyMemberMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.fileupload.beans.FileuploadLiteBean;
import com.aimluck.eip.improveope.util.ImproveopeApplyUtils;
import com.aimluck.eip.improveope.util.ImproveopeUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.services.storage.ALStorageService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 業務改善掲示板のフォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopeApplyFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopeApplyFormData.class.getName());

  /** トピック名 */
  private ALStringField apply_name;

  /** メモ */
  private ALStringField note;

  /** ステータス */
  private ALStringField status;

  /** 添付ファイルリスト */
  private List<FileuploadLiteBean> fileuploadList = null;

  /** 添付フォルダ名 */
  private String folderName = null;

  /** 提案者ユーザIDリスト */
  private ALStringField members;

  /** 提案者一覧 */
  private List<ALEipUser> mapList;


  private int uid;

  private String orgId;

  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclCategoryList;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();

    folderName = rundata.getParameters().getString("folderName");

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclCategoryList =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF,
        ALAccessControlConstants.VALUE_ACL_LIST);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {
    // トピック名
    apply_name = new ALStringField();
    apply_name.setFieldName("タイトル");
    apply_name.setTrim(true);
    // メモ
    note = new ALStringField();
    note.setFieldName("内容");
    note.setTrim(false);

    // 提案者のリスト
    members = new ALStringField();
    members.setFieldName("提案者");
    members.setTrim(true);
    // 提案者リスト
    mapList = new ArrayList<ALEipUser>();

    fileuploadList = new ArrayList<FileuploadLiteBean>();

    // ステータス
    status = new ALStringField();

    aclPortletFeature =
      ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF;
  }

  /**
   * 申請の各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // タイトル名必須項目
    apply_name.setNotNull(true);
    // タイトル名の文字数制限
    apply_name.limitMaxLength(100);
    // メモ必須項目
    note.setNotNull(true);
    // メモの文字数制限
    note.limitMaxLength(10000);


  }

  /**
   * トピックのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // トピック名
    apply_name.validate(msgList);
    // メモ
    note.validate(msgList);

    // 提案者
    if (mapList == null || mapList.size() <= 0) {
      msgList.add("『 <span class='em'>提案者</span> 』を指定してください。");
    } else if (!(mapList.get(0) instanceof ALEipUser)) {
      msgList.add("提案者のユーザーが全て無効、もしくは削除されています。有効なユーザーを一人以上選択してください。");
    }

    return (msgList.size() == 0);

  }

  /**
   * トピックをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      OriTImproveopeApply topic =
        ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);
      if (topic == null) {
        return false;
      }

      // タイトル
      apply_name.setValue(topic.getApplyName());

      // 内容
      note.setValue(topic.getNote());

      // ステータス
      status.setValue(topic.getStatus());

      // ファイル
      SelectQuery<OriTImproveopeApplyFile> query =
        Database.query(OriTImproveopeApplyFile.class);
      query.andQualifier(ExpressionFactory.matchDbExp(
    		  OriTImproveopeApplyFile.ORI_TIMPROVEOPE_APPLY_PROPERTY,
        topic.getApplyId()));
      List<OriTImproveopeApplyFile> improveopeFileList = query.fetchList();
      for (OriTImproveopeApplyFile file : improveopeFileList) {
        FileuploadLiteBean fbean = new FileuploadLiteBean();
        fbean.initField();
        fbean.setFileId(file.getFileId());
        fbean.setFileName(file.getFileName());
        fileuploadList.add(fbean);
      }

      // 提案者
      List<OriTImproveopeApplyMemberMap> maps = ImproveopeApplyUtils.getOriTImproveopeApplyMemberMap(topic);
      OriTImproveopeApplyMemberMap map = null;
      int size = maps.size();
      for (int i = 0; i < size; i++) {
        map = maps.get(i);
        int user_id = map.getUserId().intValue();
        mapList.add(ALEipUtils.getALEipUser(user_id));
      }


    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * トピックをデータベースから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    try {

      // FIX_ME イベントログのために一度IDと名前を取得
      int parentid =
        Integer.parseInt(ALEipUtils.getTemp(
          rundata,
          context,
          ALEipConstants.ENTITY_ID));

      OriTImproveopeApply parent =
        Database.get(OriTImproveopeApply.class, (long) parentid);

      // オブジェクトモデルを取得
      List<OriTImproveopeApply> list;

      list = ImproveopeApplyUtils.getOriTImproveopeApplyListToDeleteApply(
            rundata,
            context);

      if (list == null) {
        // 指定した トピック ID のレコードが見つからない場合
        logger.debug("[ImproveopeApplyFormData] Not found List...");
        throw new ALPageNotFoundException();
      }

      List<Integer> applyIdList = new ArrayList<Integer>();
      OriTImproveopeApply apply;
      int size = list.size();
      for (int i = 0; i < size; i++) {
    	apply = list.get(i);
        applyIdList.add(apply.getApplyId());
      }

      // トピックを削除
      SelectQuery<OriTImproveopeApply> query =
        Database.query(OriTImproveopeApply.class);
      Expression exp =
        ExpressionFactory.inDbExp(
       		OriTImproveopeApply.APPLY_ID_PK_COLUMN,
          applyIdList);
      query.setQualifier(exp);

      List<OriTImproveopeApply> applys = query.fetchList();

      List<String> fpaths = new ArrayList<String>();
      if (applys.size() > 0) {
        int tsize = applys.size();
        for (int i = 0; i < tsize; i++) {
          List<?> files = applys.get(i).getOriTImproveopeApplyFile();
          if (files != null && files.size() > 0) {
            int fsize = files.size();
            for (int j = 0; j < fsize; j++) {
              fpaths.add(((OriTImproveopeApplyFile) files.get(j)).getFilePath());
            }
          }
        }
      }

      Database.deleteAll(applys);
      Database.commit();

      if (fpaths.size() > 0) {
        // ローカルファイルに保存されているファイルを削除する．
        int fsize = fpaths.size();
        for (int i = 0; i < fsize; i++) {
          ALStorageService.deleteFile(ImproveopeUtils.getSaveDirPath(orgId, uid)
            + fpaths.get(i));
        }
      }

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        parent.getApplyId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        parent.getApplyName());

    } catch (Exception e) {
      Database.rollback();
      logger.error("[ImproveopeApplyFormData]", e);
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * トピックをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 新規オブジェクトモデル
      OriTImproveopeApply apply = Database.create(OriTImproveopeApply.class);
      // トピック名
      apply.setApplyName(apply_name.getValue());
      // ユーザーID
      apply.setOwnerId(Integer.valueOf(uid));
      // メモ
      apply.setNote(note.getValue());
      // ステータス
      apply.setStatus(ImproveopeApplyUtils.APPLY_STATUS_APPLY);
      // 作成者
      apply.setCreateUserId(Integer.valueOf(uid));
      // 更新者
      apply.setUpdateUserId(Integer.valueOf(uid));
      // 作成日
      apply.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      apply.setUpdateDate(Calendar.getInstance().getTime());

      // ファイルをデータベースに登録する．
      if (!ImproveopeApplyUtils.insertFileDataDelegate(
        rundata,
        context,
        apply,
        fileuploadList,
        folderName,
        msgList)) {
        return false;
      }

      // 提案者
      for (ALEipUser user : mapList) {
    	  OriTImproveopeApplyMemberMap map = Database.create(OriTImproveopeApplyMemberMap.class);
        int userid = (int) user.getUserId().getValue();

        map.setOriTImproveopeApply(apply);
        map.setUserId(Integer.valueOf(userid));
      }

      Database.commit();

      // 申請ポイント付与
      ImproveopeUtils.grantApplyPoint(apply);

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
    		  apply.getApplyId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        apply.getApplyName());



      /* 本部権限保持者と提案者へアクティビティー */
      List<String> ActivityList = ImproveopeApplyUtils.getAdminMemberList();
      TurbineUser tuser = null;
      for (ALEipUser user : mapList) {
        try {
          tuser = ALEipUtils.getTurbineUser(Integer.valueOf(user.getUserId().toString()));
          if (ActivityList.indexOf(tuser.getLoginName()) == -1) {
            ActivityList.add(tuser.getLoginName());
          }
        } catch (ALDBErrorException e) {}
      }
      ALEipUser user = ALEipUtils.getALEipUser(uid);
      ImproveopeApplyUtils.createApplyActivity(
        apply,
        user.getName().getValue(),
        ActivityList,
        true);


      // 添付ファイル保存先のフォルダを削除
      ALStorageService.deleteTmpFolder(uid, folderName);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }


  /**
   * データベースに格納されているトピックを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 新規オブジェクトモデル
      OriTImproveopeApply apply =
        ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);
      // タイトル
      apply.setApplyName(apply_name.getValue());
      // メモ
      apply.setNote(note.getValue());
      // 更新者
      apply.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      apply.setUpdateDate(Calendar.getInstance().getTime());

      // ファイルをデータベースに登録する．
      if (!ImproveopeApplyUtils.insertFileDataDelegate(
        rundata,
        context,
        apply,
        fileuploadList,
        folderName,
        msgList)) {
        return false;
      }

      // 古いマップデータを削除
      List<OriTImproveopeApplyMemberMap> tmp_map = ImproveopeApplyUtils.getOriTImproveopeApplyMemberMap(apply);
      Database.deleteAll(tmp_map);
      // 提案者
      for (ALEipUser user : mapList) {
    	  OriTImproveopeApplyMemberMap map = Database.create(OriTImproveopeApplyMemberMap.class);
        int userid = (int) user.getUserId().getValue();

        map.setOriTImproveopeApply(apply);
        map.setUserId(Integer.valueOf(userid));
      }

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
	    apply.getApplyId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        apply.getApplyName());

      /* 本部権限保持者と提案者へアクティビティー */
      List<String> ActivityList = ImproveopeApplyUtils.getAdminMemberList();
      TurbineUser tuser = null;
      for (ALEipUser user : mapList) {
        try {
          tuser = ALEipUtils.getTurbineUser(Integer.valueOf(user.getUserId().toString()));
          if (ActivityList.indexOf(tuser.getLoginName()) == -1) {
            ActivityList.add(tuser.getLoginName());
          }
        } catch (ALDBErrorException e) {}
      }
      ALEipUser user = ALEipUtils.getALEipUser(uid);
      ImproveopeApplyUtils.createApplyActivity(
        apply,
        user.getName().getValue(),
        ActivityList,
        false);

      // 添付ファイル保存先のフォルダを削除
      ALStorageService.deleteTmpFolder(uid, folderName);
    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    try {
      fileuploadList = ImproveopeApplyUtils.getFileuploadList(rundata);

      String userNames[] = rundata.getParameters().getStrings("members");
      if (userNames != null && userNames.length > 0) {
        SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
        Expression exp =
          ExpressionFactory.inExp(TurbineUser.LOGIN_NAME_PROPERTY, userNames);
        query.setQualifier(exp);
        mapList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    return res;
  }

  /**
   * メモを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  /**
   * タイトルを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getApplyName() {
    return apply_name;
  }

  /**
   * ステータスを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getStatus() {
    return status;
  }

  /**
   * 申請中かどうか
   * @return
   */
  public boolean hasApply() {
	  if (status.toString().equals(ImproveopeApplyUtils.APPLY_STATUS_APPLY)) {
		  return true;
	  }
	  return false;
  }

  public List<FileuploadLiteBean> getAttachmentFileNameList() {
    return fileuploadList;
  }

  public String getFolderName() {
    return folderName;
  }

  /**
   * グループメンバーを取得します。 <BR>
   *
   * @return
   */
  public List<ALEipUser> getMapList() {
    return mapList;
  }

  /**
   * グループメンバーを格納します。 <BR>
   *
   * @param str
   * @return
   */
  public void setMapList(ArrayList<ALEipUser> list) {
    mapList = list;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  public void setAclPortletFeature(String aclPortletFeature) {
    this.aclPortletFeature = aclPortletFeature;
  }


}
