/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.exttimecard;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.portal.portlets.VelocityPortlet;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.eip.account.util.AccountUtils;
import com.aimluck.eip.cayenne.om.account.EipMPost;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecard;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecardSystem;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecardSystemMap;
import com.aimluck.eip.cayenne.om.portlet.ExtTimecardAggregate;
import com.aimluck.eip.cayenne.om.portlet.ExtTimecardAggregateUser;
import com.aimluck.eip.cayenne.om.portlet.OriTPayrollSystem;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALBaseUser;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipGroup;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipPost;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;
import com.aimluck.eip.exttimecard.util.ExtTimecardUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.util.ALEipUtils;

/**
 * タイムカード集計の一覧を処理するクラスです。 <br />
 *
 *
 */

public class ExtTimecardSummaryListSelectData extends
    ALAbstractSelectData<EipTExtTimecard, EipTExtTimecard> implements ALData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ExtTimecardSummaryListSelectData.class.getName());

  /** <code>target_group_name</code> 表示対象の部署名 */
  private String target_group_name;

  /** <code>target_user_id</code> 表示対象のユーザ ID */
  private String target_user_id;

  /** <code>myGroupList</code> グループリスト（My グループと部署） */
  private List<ALEipGroup> myGroupList = null;

  /** <code>userList</code> 表示切り替え用のユーザリスト */
  private List<ALEipUser> userList = null;

  /** 一覧データ */
  private List<Object> list;

  /** <code>userid</code> ユーザーID */
  private String userid;

  /** <code>TARGET_GROUP_NAME</code> グループによる表示切り替え用変数の識別子 */
  private final String TARGET_GROUP_NAME = "target_group_name";

  /** <code>TARGET_USER_ID</code> ユーザによる表示切り替え用変数の識別子 */
  private final String TARGET_USER_ID = "target_user_id";

  private String nowtime;

  /** ユーザーマップ */
  private Map<Integer, List<ExtTimecardResultData>> usermap;

  /** 日付マップ */
  private Map<Integer, ExtTimecardSummaryResultData> datemap;

  /** アクセス権限の機能名 */
  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  private boolean hasAclSummaryOther;

  /** 全社員の承認権限の有無 */
  private boolean hasAclSummaryAdmin;

  /** 他ユーザーのxlsエクスポート権限 */
  private boolean hasAclXlsExport;

  /** <code>viewMonth</code> 現在の月 */
  private ALDateTimeField viewMonth;

  /** <code>prevMonth</code> 前の月 */
  private ALDateTimeField prevMonth;

  /** <code>nextMonth</code> 次の月 */
  private ALDateTimeField nextMonth;

  /** <code>currentMonth</code> 今月 */
  private ALDateTimeField currentMonth;

  /** <code>today</code> 今日 */
  private ALDateTimeField today;

  /** <code>viewStart</code> 表示開始日時 */
  private ALDateTimeField viewStart;

  /** <code>viewEnd</code> 表示終了日時 */
  private ALDateTimeField viewEnd;

  /** <code>viewEndCrt</code> 表示終了日時 (Criteria) */
  private ALDateTimeField viewEndCrt;

  private final String MODE = "summary";

  /** <code>viewTodo</code> ToDo 表示設定 */
  protected int viewTodo;

  /** 開始日 */
  private int startDay;

  /** 現在有効なユーザー数  */
  private int current_user_count;

  /**
   *
   */
  @Override
  public void initField() {
  }

  /**
   *
   * @param action
   * @param rundata
   * @param context
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    this.initField();

    // POST/GET から yyyy-MM の形式で受け渡される。
    // 現在の月
    viewMonth = new ALDateTimeField("yyyy-MM");
    viewMonth.setNotNull(true);
    // 前の月
    prevMonth = new ALDateTimeField("yyyy-MM");
    // 次の月
    nextMonth = new ALDateTimeField("yyyy-MM");
    // 今月
    currentMonth = new ALDateTimeField("yyyy-MM");
    // 表示開始日時
    viewStart = new ALDateTimeField("yyyy-MM-dd");
    // 表示終了日時
    viewEnd = new ALDateTimeField("yyyy-MM-dd");
    // 表示終了日時 (Criteria)
    viewEndCrt = new ALDateTimeField("yyyy-MM-dd");

    startDay = 1;

    // 自ポートレットからのリクエストであれば、パラメータを展開しセッションに保存する。
    if (ALEipUtils.isMatch(rundata, context)) {
      // スケジュールの表示開始日時
      // e.g. 2004-3-14
      if (rundata.getParameters().containsKey("view_month")) {
        ALEipUtils.setTemp(rundata, context, "view_month", rundata
          .getParameters()
          .getString("view_month"));
      }
    }

    // ログインユーザの ID を設定する．
    userid = Integer.toString(ALEipUtils.getUserId(rundata));

    // My グループの一覧を取得する．
    List<ALEipGroup> myGroups = ALEipUtils.getMyGroups(rundata);
    myGroupList = new ArrayList<ALEipGroup>();
    int length = myGroups.size();
    for (int i = 0; i < length; i++) {
      myGroupList.add(myGroups.get(i));
    }

    try {
      // スケジュールを表示するユーザ ID をセッションに設定する．
      String userFilter = ALEipUtils.getTemp(rundata, context, TARGET_USER_ID);
      if (userFilter == null || userFilter.equals("")) {
        VelocityPortlet portlet = ALEipUtils.getPortlet(rundata, context);
        userFilter = portlet.getPortletConfig().getInitParameter("p3a-user");
      }

      if (userFilter != null && (!userFilter.equals(""))) {
        int paramId = -1;
        try {
          paramId = Integer.parseInt(userFilter);
          if (paramId > 3) {
            ALEipUser user = ALEipUtils.getALEipUser(paramId);
            if (user != null) {
              // 指定したユーザが存在する場合，セッションに保存する．
              ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userFilter);
            } else {
              ALEipUtils.removeTemp(rundata, context, TARGET_USER_ID);
            }
          }
        } catch (NumberFormatException e) {
        }
      } else {
        ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userid);
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    // アクセス権
    if (target_user_id == null
      || "".equals(target_user_id)
      || userid.equals(target_user_id)) {
      aclPortletFeature =
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_SELF;
    } else {
      aclPortletFeature =
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_OTHER;
    }
    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();
    hasAclSummaryOther =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_OTHER,
        ALAccessControlConstants.VALUE_ACL_LIST);
    hasAclSummaryAdmin =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_ADMIN,
        ALAccessControlConstants.VALUE_ACL_LIST);
    hasAclXlsExport =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_ADMIN,
        ALAccessControlConstants.VALUE_ACL_EXPORT);

    if (!hasAclSummaryOther && !hasAclSummaryAdmin) {
      // 他ユーザーの閲覧権限がないときには、グループを未選択にする。
      target_group_name = "only";
      aclPortletFeature =
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_SELF;
      hasAclXlsExport =
        aclhandler.hasAuthority(
          ALEipUtils.getUserId(rundata),
          ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_SELF,
          ALAccessControlConstants.VALUE_ACL_EXPORT);
    }

    datemap = new LinkedHashMap<Integer, ExtTimecardSummaryResultData>();

    usermap = new LinkedHashMap<Integer, List<ExtTimecardResultData>>();

    /** 現在のユーザを取得 */
    if (target_user_id != null && !target_user_id.isEmpty()) {
      /** 勤務形態通常のstartDayを取得 */
      SelectQuery<EipTExtTimecardSystemMap> default_query =
        Database.query(EipTExtTimecardSystemMap.class);
      Expression exp =
        ExpressionFactory.matchExp(
          EipTExtTimecardSystemMap.USER_ID_PROPERTY,
          target_user_id);
      default_query.setQualifier(exp);
      ResultList<EipTExtTimecardSystemMap> map_list =
        default_query.getResultList();
      if (!map_list.isEmpty()) {
        startDay = map_list.get(0).getEipTExtTimecardSystem().getStartDay();
      }
    } else if (userid != null && !userid.isEmpty()) {
      /** 勤務形態通常のstartDayを取得 */
      SelectQuery<EipTExtTimecardSystemMap> default_query =
        Database.query(EipTExtTimecardSystemMap.class);
      Expression exp =
        ExpressionFactory.matchExp(
          EipTExtTimecardSystemMap.USER_ID_PROPERTY,
          userid);
      default_query.setQualifier(exp);
      ResultList<EipTExtTimecardSystemMap> map_list =
        default_query.getResultList();
      if (!map_list.isEmpty()) {
        startDay = map_list.get(0).getEipTExtTimecardSystem().getStartDay();
      }
    }

    // 今日の日付
    today = new ALDateTimeField("yyyy-MM-dd");
    Calendar to = Calendar.getInstance();
    to.set(Calendar.HOUR_OF_DAY, 0);
    to.set(Calendar.MINUTE, 0);
    today.setValue(to.getTime());

    // 現在の月
    String tmpViewMonth = ALEipUtils.getTemp(rundata, context, "view_month");
    if (tmpViewMonth == null || tmpViewMonth.equals("")) {
      Calendar cal = Calendar.getInstance();
      cal.set(Calendar.DATE, 1);
      cal.set(Calendar.HOUR_OF_DAY, 0);
      cal.set(Calendar.MINUTE, 0);
      if (Integer.parseInt(today.getDay().toString()) < startDay) {
        cal.add(Calendar.MONTH, -1);
      }
      viewMonth.setValue(cal.getTime());
    } else {
      viewMonth.setValue(tmpViewMonth);
      if (!viewMonth.validate(new ArrayList<String>())) {
        ALEipUtils.removeTemp(rundata, context, "view_month");
        throw new ALPageNotFoundException();
      }
    }

    if (Integer.parseInt(today.getMonth()) == Integer.parseInt(viewMonth
      .getMonth()
      .toString())) {
      currentMonth.setValue(to.getTime());
    } else {
      Calendar tmp_cal = Calendar.getInstance();
      tmp_cal.set(Calendar.DATE, 1);
      tmp_cal.set(Calendar.HOUR_OF_DAY, 0);
      tmp_cal.set(Calendar.MINUTE, 0);
      tmp_cal.add(Calendar.MONTH, -1);
      currentMonth.setValue(tmp_cal.getTime());
    }

    // 表示開始日時
    Calendar cal = Calendar.getInstance();
    cal.set(Calendar.MONTH, Integer.parseInt(viewMonth.getMonth()) - 1);
    cal.set(Calendar.DATE, startDay);
    Date startDate = cal.getTime();
    viewStart.setValue(startDate);

    // 表示終了日時
    cal.add(Calendar.MONTH, 1);
    cal.add(Calendar.DATE, -1);
    Date endDate = cal.getTime();
    viewEnd.setValue(endDate);
    viewEndCrt.setValue(endDate);

    // 次の月、前の月
    Calendar cal2 = Calendar.getInstance();
    cal2.setTime(viewMonth.getValue());
    cal2.add(Calendar.MONTH, 1);
    nextMonth.setValue(cal2.getTime());
    cal2.add(Calendar.MONTH, -2);
    prevMonth.setValue(cal2.getTime());

    ALEipUtils.setTemp(rundata, context, "tmpStart", viewStart.toString()
      + "-00-00");
    ALEipUtils.setTemp(rundata, context, "tmpEnd", viewStart.toString()
      + "-00-00");

    // 現在のユーザー数
    current_user_count = ALEipUtils.getCurrentUserNum(rundata);

    setupLists(rundata, context);
  }

  /**
   * 一覧表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FASLE 失敗
   */
  @Override
  public boolean doViewList(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);
      doCheckAclPermission(
        rundata,
        context,
        ALAccessControlConstants.VALUE_ACL_LIST);
      action.setMode(ALEipConstants.MODE_LIST);
      for (int i = 0; i < userList.size(); i++) {
        ALEipUser eipUser = userList.get(i);
        List<EipTExtTimecard> aList =
          selectList(rundata, context, eipUser.getUserId().getValueAsString());
        if (aList != null) {
          list = new ArrayList<Object>();
          Object obj = null;
          int size = aList.size();
          for (int j = 0; j < size; j++) {
            obj = getResultData(aList.get(j));
            if (obj != null) {
              list.add(obj);
            }
          }
        }
      }
      action.setResultData(this);
      action.putData(rundata, context);
      ALEipUtils.removeTemp(rundata, context, ALEipConstants.ENTITY_ID);
      return (list != null);
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }

  }

  /**
   *
   * @param rundata
   * @param context
   * @param target_user_id
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  protected List<EipTExtTimecard> selectList(RunData rundata, Context context,
      String target_user_id) throws ALPageNotFoundException, ALDBErrorException {
    try {
      // 指定グループや指定ユーザをセッションに設定する．
      setupLists(rundata, context);

      if (!"".equals(target_user_id)) {

        SelectQuery<EipTExtTimecard> query =
          getSelectQuery(rundata, context, target_user_id);
        buildSelectQueryForListView(query);
        query.orderAscending(EipTExtTimecard.PUNCH_DATE_PROPERTY);

        return query.getResultList();
      } else {
        return null;
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
  *
  * @param rundata
  * @param context
  * @param target_user_id
  * @return
  * @throws ALPageNotFoundException
  * @throws ALDBErrorException
  */
 protected List<EipTExtTimecard> selectCloseList(RunData rundata, Context context,
     String target_user_id) throws ALPageNotFoundException, ALDBErrorException {
   try {

     if (!"".equals(target_user_id)) {

       SelectQuery<EipTExtTimecard> query =
         getSelectQuery(rundata, context, target_user_id);
       buildSelectQueryForListView(query);
       query.orderAscending(EipTExtTimecard.PUNCH_DATE_PROPERTY);

       return query.getResultList();
     } else {
       return null;
     }
   } catch (Exception ex) {
     logger.error("Exception", ex);
     return null;
   }
 }
  /**
   *
   * @param rundata
   * @param context
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected ResultList<EipTExtTimecard> selectList(RunData rundata,
      Context context) throws ALPageNotFoundException, ALDBErrorException {
    try {
      // 指定グループや指定ユーザをセッションに設定する．
      setupLists(rundata, context);

      if (!"".equals(target_user_id)) {

        SelectQuery<EipTExtTimecard> query =
          getSelectQuery(rundata, context, target_user_id);
        buildSelectQueryForListView(query);
        query.orderAscending(EipTExtTimecard.PUNCH_DATE_PROPERTY);

        return query.getResultList();
      } else {
        return null;
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   *
   * @param rundata
   * @param context
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected EipTExtTimecard selectDetail(RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    return null;
  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   *
   * @param obj
   * @return
   */
  @Override
  protected Object getResultData(EipTExtTimecard record)
      throws ALPageNotFoundException, ALDBErrorException {
    try {
      int user_id = record.getUserId();

      ExtTimecardResultData rd = new ExtTimecardResultData();
      rd.initField();
      rd.setPunchDate(record.getPunchDate());
      rd.setRefixFlag(record.getCreateDate(), record.getUpdateDate());
      rd.setClockInTime(record.getClockInTime());
      rd.setClockOutTime(record.getClockOutTime());

      // morimoto-juichi add 2012/10/09
      rd.setStatus(record.getStatus());
      rd.setUsedTime(record.getUsedTime());
      rd.setWaitingTime(record.getWaitingTime());
      rd.setWatchFlag(record.getWatchFlag());
      rd.setSupportFlag(record.getSupportFlag());
      rd.setMoveFlag(record.getMoveFlag());
      rd.setCarFlag(record.getCarFlag());
      rd.setHelp1Flag(record.getHelp1Flag());
      rd.setHelp2Flag(record.getHelp2Flag());
      rd.setBreakTime(record.getBreakTime());
      rd.setAddholidayFlag(record.getAddholidayFlag());
      rd.setSubholidayFlag(record.getSubholidayFlag());

      for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
        rd.setOutgoingTime(record.getOutgoingTime(i), i);
        rd.setComebackTime(record.getComebackTime(i), i);
      }
      rd.setType(record.getType());

      List<ExtTimecardResultData> list;
      if (usermap.containsKey(user_id)) {
        list = usermap.get(user_id);
        list.add(rd);
        usermap.put(user_id, list);
      } else {
        list = new ArrayList<ExtTimecardResultData>();
        list.add(rd);
        usermap.put(user_id, list);
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
    return null;
  }

  /**
   *
   * @param obj
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected Object getResultDataDetail(EipTExtTimecard obj)
      throws ALPageNotFoundException, ALDBErrorException {
    return null;
  }

  /*
   * (非 Javadoc)
   */
  @Override
  protected Attributes getColumnMap() {
    return null;
  }

  /**
   * ログイン中のユーザー情報を取得
   *
   */

  private List<ALEipUser> getUserList(int userid) {
    List<ALEipUser> list = new ArrayList<ALEipUser>();
    ALEipUser user = new ALEipUser();
    try {
      user = ALEipUtils.getALEipUser(userid);
    } catch (NumberFormatException e1) {
    } catch (ALDBErrorException e1) {
    }
    list.add(user);
    return list;
  }

  /**
   * 指定グループや指定ユーザをセッションに設定する．
   *
   * @param rundata
   * @param context
   * @throws ALDBErrorException
   */
  private void setupLists(RunData rundata, Context context) {
    target_group_name = getTargetGroupName(rundata, context);
    if ((target_group_name != null)
      && (!target_group_name.equals(""))
      && (!target_group_name.equals("all"))
      && (!target_group_name.equals("only"))) {
      userList = ALEipUtils.getMainPostUsers(target_group_name);
    } else if (target_group_name.equals("only")) {

      userList = getUserList(Integer.parseInt(userid));
    } else {
      userList = ALEipUtils.getUsers("LoginUser");
    }

    if (userList == null || userList.size() == 0) {
      target_user_id = "";
      ALEipUtils.removeTemp(rundata, context, TARGET_USER_ID);
      return;
    }

    target_user_id = getTargetUserId(rundata, context);
  }

  /**
   * 指定されたユーザーIDの締め処理を行う
   *
   * @param action
   * @param rundata
   * @param context
   * @param user_id
   * @return
   */
  public boolean close(ALAction action, RunData rundata, Context context,
      int user_id) {
    try {
      init(action, rundata, context);
      userList = getUserList(user_id);
      doCheckAclPermission(
        rundata,
        context,
        ALAccessControlConstants.VALUE_ACL_LIST);

      for (int i = 0; i < userList.size(); i++) {
	      ALEipUser eipUser = userList.get(i);
	      List<EipTExtTimecard> aList = selectCloseList(rundata, context, eipUser.getUserId().getValueAsString());
	      if (aList != null) {
	        list = new ArrayList<Object>();
	        Object obj = null;
	        int size = aList.size();
	        for (int j = 0; j < size; j++) {
	          obj = getResultData(aList.get(j));
	          if (obj != null) {
	            list.add(obj);
	          }
	        }
	      }
      }

      List<ExtTimecardSummaryResultData> resultlist = getGroupExtTimecards();
      ExtTimecardSummaryResultData result = resultlist.get(0);

      // 新規オブジェクトモデル
      ExtTimecardAggregateUser aggregate =
        Database.create(ExtTimecardAggregateUser.class);

      // --------------------------------
      // 全員に設定する項目
      // --------------------------------
      // 締め日
      SimpleDateFormat fm = new SimpleDateFormat("yyyy-MM-dd");
      Date cutoff_date = Calendar.getInstance().getTime();
      aggregate.setCutoffDate(fm.format(cutoff_date));
      // 対象年月
      aggregate.setAggregateYears(viewMonth.toString("yyyy-MM"));
      // ユーザーID
      aggregate.setUserId(Integer.valueOf(result.getUserId().toString()));
      // ユーザー名
      aggregate.setUserName(result.getUserName());
      // ユーザー名カナ
      aggregate.setUserNameKn(result.getUserNameKana());
      // ユーザー並び順
      aggregate.setUserOrder(Integer.valueOf(result.getUserOrder()));
      // 社員番号
      aggregate.setEmployeeNumber(result.getEmployeeNumber());
      // 部署
      aggregate.setPost(result.getPostName());
      // 部署コード
      aggregate.setPostCd(result.getPostCode());
      // 勤務形態
      aggregate.setWorkingArrangements(result.getSystemName().toString());
      // 役職
      aggregate.setPosition(result.getPositionName().toString());
      // 給与体系
      aggregate.setPayrollSystem(result.getPayrollName());
      // 給与体系コード
      aggregate.setPayrollSystemCd(result.getPayrollCode().toString());
      // 給与体系タイプ
      aggregate.setPayrollSystemType(result.getPayrollType().toString());
      // 作成日
      aggregate.setCreateDate(cutoff_date);
      // ログインユーザー
      aggregate.setCreateUserId(ALEipUtils.getUserId(rundata));
      // --------------------------------
      // 社員のみ設定する項目
      // --------------------------------
      if ("B".equals(result.getPayrollType())) {
        // 所定労働日数
        aggregate.setPrescribedWorkingDays(result
          .getPrescribedWorkingDay()
          .toString());
        // 所定労働時間
        aggregate.setScheduledWorkingHours(result
          .getScheduledWorkingHour()
          .toString());
        // 総就労時間
        aggregate.setTotalWorkingHours(result.getWorkHour().toString());
        // 法定内残業
        aggregate.setWithinLegalOvertime(result
          .getWithinLegalOvertime()
          .toString());
        // 法定外残業
        aggregate.setOutsideLegalOvertime(result
          .getOutsideLegalOvertime()
          .toString());
        // 60ｈ越残業
        aggregate.setSixtyHoursOvertime(result.getOvertime60().toString());
      }
      // --------------------------------
      // 雑給のみ設定する項目
      // --------------------------------
      if ("C".equals(result.getPayrollType())) {
        // 出勤時間
        aggregate.setAttendanceTime(result.getWorkHour().toString());
        // 日給
        aggregate.setDailyWage(result.getNikyuDay().toString());
      }
      // --------------------------------
      // 社員・雑給のみ設定する項目
      // --------------------------------
      if ("B".equals(result.getPayrollType())
        || "C".equals(result.getPayrollType())) {
        // 勤務日数
        aggregate.setAttendanceDays(result.getWorkDay().toString());
        // 深夜加算
        aggregate.setMidnightAddition(result.getMidnightHour().toString());
        // 休日加算
        aggregate.setHolidayAddition(result.getAddholidayHour().toString());
        // 車出手当
        aggregate.setOutCarAllowance(result.getCarNumber().toString());
        // 応援手当
        aggregate.setSupportAllowance(result.getSupportNumber().toString());
        // 遠隔手当１
        aggregate.setRemoteAllowance1(result.getHelp1Number().toString());
        // 遠隔手当２
        aggregate.setRemoteAllowance2(result.getHelp2Number().toString());
        // 休消手当
        aggregate.setHolidayConsumptionAllowance(result
          .getSubholidayNumber()
          .toString());
        // 当直手当
        aggregate.setDutyAllowance(result.getWatchNumber().toString());
        // 有休
        aggregate.setPaidLeave(result.getUsedTime().toString());
        // 慶弔休
        aggregate.setCondolence(result.getKeicho().toString());
      }

      // タイムカードを登録
      Database.commit();

      action.setResultData(this);
      action.putData(rundata, context);
      ALEipUtils.removeTemp(rundata, context, ALEipConstants.ENTITY_ID);
      return (resultlist != null);
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }

  }

  /**
   * 指定されたユーザーIDの締め解除処理を行う
   *
   * @param action
   * @param rundata
   * @param context
   * @param user_id
   * @return
   */
  public boolean closecancel(ALAction action, RunData rundata, Context context,
      int user_id) {
    try {
      init(action, rundata, context);
      userList = getUserList(user_id);
      doCheckAclPermission(
        rundata,
        context,
        ALAccessControlConstants.VALUE_ACL_LIST);

      String yyyymm = viewMonth.toString("yyyy-MM");
      SelectQuery<ExtTimecardAggregateUser> query =
        Database.query(ExtTimecardAggregateUser.class);
      Expression exp0 =
        ExpressionFactory.matchExp(
          ExtTimecardAggregateUser.USER_ID_PROPERTY,
          user_id);
      Expression exp1 =
        ExpressionFactory.matchExp(
          ExtTimecardAggregateUser.AGGREGATE_YEARS_PROPERTY,
          yyyymm);
      query.setQualifier(exp0.andExp(exp1));
      List<ExtTimecardAggregateUser> ag = query.fetchList();

      if (ag != null && ag.size() > 0) {
        for (int i = 0; i < ag.size(); i++) {
          Database.delete(ag.get(i));
        }
        // タイムカードを登録
        Database.commit();
      }

      action.setResultData(this);
      action.putData(rundata, context);
      ALEipUtils.removeTemp(rundata, context, ALEipConstants.ENTITY_ID);
      return (ag != null);
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }

  }

  /**
   * 表示月の締め解除処理を行う
   *
   * @param action
   * @param rundata
   * @param context
   * @param user_id
   * @return
   */
  public void monthclosecancel(RunData rundata, Context context) {
    try {

      String yyyymm = viewMonth.toString("yyyy-MM");
      SelectQuery<ExtTimecardAggregate> query =
        Database.query(ExtTimecardAggregate.class);
      Expression exp0 =
        ExpressionFactory.matchExp(
          ExtTimecardAggregateUser.AGGREGATE_YEARS_PROPERTY,
          yyyymm);
      query.setQualifier(exp0);
      List<ExtTimecardAggregate> ag = query.fetchList();

      if (ag != null && ag.size() > 0) {
        for (int i = 0; i < ag.size(); i++) {
          Database.delete(ag.get(i));
        }
        // タイムカードを登録
        Database.commit();
      }

	 } catch (Exception ex) {
		 logger.error("Exception[getExtTimecardAggregate]", ex);
	 }
  }

  /**
   * 表示月の締め処理を行う
   *
   * @param action
   * @param rundata
   * @param context
   * @param user_id
   * @return
   */
  public void monthclose(RunData rundata, Context context) {
    try {

      String yyyymm = viewMonth.toString("yyyy-MM");

      SimpleDateFormat fm = new SimpleDateFormat("yyyy-MM-dd");
      Date cutoff_date = Calendar.getInstance().getTime();


      // 新規オブジェクトモデル
      ExtTimecardAggregate aggregate =
        Database.create(ExtTimecardAggregate.class);

      // 対象年月
      aggregate.setAggregateYears(yyyymm);
      // 締め日
      aggregate.setCutoffDate(fm.format(cutoff_date));
      // 締めフラグ
      aggregate.setSimeFlg("1");
      // 締めユーザーID
      aggregate.setCreateUserId(ALEipUtils.getUserId(rundata));
      // 作成日
      aggregate.setCreateDate(cutoff_date);

      // 登録
      Database.commit();

	 } catch (Exception ex) {
		 logger.error("Exception[getExtTimecardAggregate]", ex);
	 }
  }

  /**
   * 表示切り替えで指定したグループ ID を取得する．
   *
   * @param rundata
   * @param context
   * @return
   */
  private String getTargetGroupName(RunData rundata, Context context) {
    String target_group_name = null;
    String idParam = null;
    if (ALEipUtils.isMatch(rundata, context)) {
      // 自ポートレットへのリクエストの場合に，グループ名を取得する．
      idParam = rundata.getParameters().getString(TARGET_GROUP_NAME);
    }
    target_group_name = ALEipUtils.getTemp(rundata, context, TARGET_GROUP_NAME);

    if (idParam == null && target_group_name == null) {
      ALEipUtils.setTemp(rundata, context, TARGET_GROUP_NAME, "only");
      target_group_name = "only";
    } else if (idParam != null) {
      ALEipUtils.setTemp(rundata, context, TARGET_GROUP_NAME, idParam);
      target_group_name = idParam;
    }

    if ((target_group_name == null)
      || (target_group_name.equals(""))
      || (target_group_name.equals("all") && !getHasAclSummaryAdmin())) {
      Map<Integer, ALEipPost> map = getPostMap();
      if (map != null && map.size() > 0) {
        for (Integer key : map.keySet()) {
          target_group_name = map.get(key).getGroupName().toString();
        }
      }
    }

    // 他人の集計を閲覧する権限がなければ自分のみ
    if (!hasAclSummaryOther && !hasAclSummaryAdmin) {
      target_group_name = "only";
    }
    return target_group_name;
  }

  /**
   * 表示切り替えで指定したユーザ ID を取得する．
   *
   * @param rundata
   * @param context
   * @return
   */
  private String getTargetUserId(RunData rundata, Context context) {
    String target_user_id = null;
    String idParam = null;
    if (ALEipUtils.isMatch(rundata, context)) {
      // 自ポートレットへのリクエストの場合に，ユーザ ID を取得する．
      idParam = rundata.getParameters().getString(TARGET_USER_ID);
    }
    target_user_id = ALEipUtils.getTemp(rundata, context, TARGET_USER_ID);

    if (idParam == null && (target_user_id == null)) {
      // ログインユーザのスケジュールを表示するため，ログイン ID を設定する．
      ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userid);
      target_user_id = userid;
    } else if (idParam != null) {
      if (idParam.equals("none")) {
        // グループで表示を切り替えた場合，
        // ログインユーザもしくはユーザリストの一番初めのユーザを
        // 表示するため，ユーザ ID を設定する．
        ALEipUser eipUser = null;
        boolean found = false;
        int length = userList.size();
        for (int i = 0; i < length; i++) {
          eipUser = userList.get(i);
          String eipUserId = eipUser.getUserId().getValueAsString();
          if (userid.equals(eipUserId)) {
            ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userid);
            target_user_id = userid;
            found = true;
            break;
          }
        }
        if (!found) {
          eipUser = userList.get(0);
          String userId = eipUser.getUserId().getValueAsString();
          ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userId);
          target_user_id = userId;
        }
      } else {
        // ユーザで表示を切り替えた場合，指定したユーザの ID を設定する．
        ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, idParam);
        target_user_id = idParam;
      }
    }
    return target_user_id;
  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery<EipTExtTimecard> getSelectQuery(RunData rundata,
      Context context, String target_user_id) {
    SelectQuery<EipTExtTimecard> query = Database.query(EipTExtTimecard.class);

    /** 勤務形態通常のstartDayを取得 */
    startDay = 1;
    SelectQuery<EipTExtTimecardSystemMap> default_query =
      Database.query(EipTExtTimecardSystemMap.class);
    Expression exp =
      ExpressionFactory.matchExp(
        EipTExtTimecardSystemMap.USER_ID_PROPERTY,
        target_user_id);
    default_query.setQualifier(exp);
    ResultList<EipTExtTimecardSystemMap> map_list =
      default_query.getResultList();
    if (!map_list.isEmpty()) {
      startDay = map_list.get(0).getEipTExtTimecardSystem().getStartDay();
    }

    Expression exp1 =
      ExpressionFactory.matchExp(EipTExtTimecard.USER_ID_PROPERTY, new Integer(
        target_user_id));
    query.setQualifier(exp1);

    Calendar cal = Calendar.getInstance();
    cal.set(Calendar.YEAR, Integer.parseInt(viewMonth.getYear()));
    cal.set(Calendar.MONTH, Integer.parseInt(viewMonth.getMonth()) - 1);
    cal.set(Calendar.DAY_OF_MONTH, startDay);
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);

    Date date = cal.getTime();
    Expression exp11 =
      ExpressionFactory.greaterOrEqualExp(
        EipTExtTimecard.PUNCH_DATE_PROPERTY,
        date);

    cal.add(Calendar.MONTH, +1);
    cal.add(Calendar.MILLISECOND, -1);
    Expression exp12 =
      ExpressionFactory.lessOrEqualExp(EipTExtTimecard.PUNCH_DATE_PROPERTY, cal
        .getTime());
    query.andQualifier(exp11.andExp(exp12));

    return buildSelectQueryForFilter(query, rundata, context);
  }

  /**
   * ユーザー毎のタイムカード一覧を取得する。
   *
   * @return
   */
  public List<ExtTimecardSummaryResultData> getUserExtTimecards() {
    List<ExtTimecardSummaryResultData> list =
      new ArrayList<ExtTimecardSummaryResultData>();
    Set<Integer> userset = usermap.keySet();

    /** 以下、ユーザーごとの処理 */
    for (Integer user_id : userset) {
      List<ExtTimecardResultData> userlist = usermap.get(user_id);
      ExtTimecardSummaryResultData summary_rd =
        new ExtTimecardSummaryResultData();
      /** 就業、残業、休出日数 */
      int work_day = 0, overtime_day = 0, off_day = 0;
      /** 就業、残業、休出時間 */
      float work_hour = 0, overtime_hour = 0, off_hour = 0;
      /** 遅刻、早退、欠勤 */
      int late_coming_day = 0, early_leaving_day = 0, absent_day = 0;
      /** 有休、代休 */
      int paid_holiday = 0, compensatory_holiday = 0;
      /** 有休利用時間 */
      int used_time = 0;
      /** 待機時間 */
      int waiting_time = 0;
      /** 手当て */
      int watch_number = 0, support_number = 0, move_number = 0, car_number = 0;
      /** その他、未入力 */
      int other_day = 0, no_input = 0;
      summary_rd.initField();

      /** タイムカード設定を取得 */
      EipTExtTimecardSystem timecard_system =
        ExtTimecardUtils.getEipTExtTimecardSystemByUserId(user_id);

      /**
       * userlistにはユーザー日ごとのタイムカードのResultDataがリストで入っているため、
       * ListResultDataに代入して各日数・時間を計算させる。
       */

      for (Object obj : userlist) {
        ExtTimecardResultData rd = (ExtTimecardResultData) obj;
        ExtTimecardListResultData lrd = new ExtTimecardListResultData();
        lrd.setRd(rd);
        lrd.setTimecardSystem(timecard_system);
        String type = rd.getType().getValue();

        // morimoto-junichi add 2012/10/09
        if (rd.getStatus().equals("1")) {

          if (type.equals(EipTExtTimecard.TYPE_WORK)
            || type.equals(EipTExtTimecard.TYPE_TIME_HOLIDAY)) {
            /** 出勤 */
            if (lrd.getWorkHour() != ExtTimecardListResultData.NO_DATA) {
              work_day++;
              work_hour += lrd.getWorkHourWithoutRestHour();
            }
            if (lrd.getOvertimeHour() != ExtTimecardListResultData.NO_DATA) {
              overtime_day++;
              overtime_hour += lrd.getOvertimeHourWithoutRestHour();
            }
            if (lrd.getOffHour() != ExtTimecardListResultData.NO_DATA) {
              off_day++;
              off_hour += lrd.getOffHour();
            }
            if (lrd.isLateComing()) {
              late_coming_day++;
            }
            if (lrd.isEarlyLeaving()) {
              early_leaving_day++;
            }
            /** 時間有休 */
            if (type.equals(EipTExtTimecard.TYPE_TIME_HOLIDAY)) {
              used_time += Integer.parseInt(rd.getUsedTime().toString());
            }
          } else if (type.equals(EipTExtTimecard.TYPE_ABSENT)) {
            /** 欠勤 */
            absent_day++;
          } else if (type.equals(EipTExtTimecard.TYPE_HOLIDAY)) {
            /** 有休 */
            paid_holiday++;
            used_time += 8;
          } else if (type.equals(EipTExtTimecard.TYPE_COMPENSATORY)) {
            /** 代休 */
            compensatory_holiday++;
          } else if (type.equals(EipTExtTimecard.TYPE_ETC)) {
            /** その他 */
            other_day++;
          }

        } // morimoto

      }

      /** ユーザーごとの合計をSummaryResultDataに代入する */
      work_hour = ExtTimecardUtils.roundHour(work_hour);
      overtime_hour = ExtTimecardUtils.roundHour(overtime_hour);
      off_hour = ExtTimecardUtils.roundHour(off_hour);

      summary_rd.setWorkDayHour(work_day, work_hour);
      summary_rd.setOvertimeDayHour(overtime_day, overtime_hour);
      summary_rd.setOffDayHour(off_day, off_hour);
      summary_rd.setLateComingDay(late_coming_day);
      summary_rd.setEarlyLeavingDay(early_leaving_day);
      summary_rd.setAbsentDay(absent_day);
      summary_rd.setPaidHoliday(paid_holiday);
      summary_rd.setCompensatoryHoliday(compensatory_holiday);
      summary_rd.setOtherDay(other_day);
      summary_rd.setNoInput(no_input);

      list.add(summary_rd);
    }
    return list;
  }

  /**
   * グループ毎のタイムカード一覧を取得する。
   *
   * @return
   */

  public List<ExtTimecardSummaryResultData> getGroupExtTimecards() {
    List<ExtTimecardSummaryResultData> list =
      new ArrayList<ExtTimecardSummaryResultData>();
    /** 以下、ユーザーごとの処理 */
    for (int i = 0; i < userList.size(); i++) {
      ALEipUser eipUser = userList.get(i);
      int user_id = Integer.parseInt(eipUser.getUserId().getValueAsString());
      ALBaseUser base_user = ALEipUtils.getBaseUser(user_id);

      if (ExtTimecardUtils.getExtTimecardAggregateUser(user_id, viewMonth
        .toString("yyyy-MM"))) {
        // 〆処理済みの場合は、締めテーブルから取得する
        ExtTimecardSummaryResultData summary_rd =
          getAggregateResultData(user_id);
        list.add(summary_rd);
      } else {
        // 給与体系
        OriTPayrollSystem payroll =
          AccountUtils.getOriTPayrollSystem(base_user.getPayrollSystemId());
        String payroll_type = payroll.getPayrollSystemType();
        String payroll_name = payroll.getPayrollSystemName();
        String payroll_code = payroll.getPayrollSystemCd();
        String user_order = String.valueOf(ALEipUtils.getUserOrder(user_id));

        List<ExtTimecardResultData> userlist = usermap.get(user_id);
        ExtTimecardSummaryResultData summary_rd =
          new ExtTimecardSummaryResultData();
        /** 就業、残業、休出日数 */
        int work_day = 0, overtime_day = 0, off_day = 0;
        /** 就業、残業、休出時間、休日加算、深夜加算 */
        float work_hour = 0, overtime_hour = 0, off_hour = 0, addholiday_hour =
          0, midnight_hour = 0;
        /** 遅刻、早退、欠勤 */
        int late_coming_day = 0, early_leaving_day = 0, absent_day = 0;
        /** 有休、代休、慶弔休 */
        int paid_holiday = 0, compensatory_holiday = 0, keicho = 0;
        /** 有休利用時間 */
        int used_time = 0;
        /** 待機時間 */
        int waiting_time = 0;
        /** 手当て */
        int watch_number = 0, support_number = 0, move_number = 0, car_number =
          0, help1_number = 0, help2_number = 0, subholiday_number = 0;
        /** その他、未入力 */
        int other_day = 0, no_input = 0;
        /** 日給 */
        int nikyu_day = 0;
        /** 申請、却下 */
        int entry_number = 0, rejection_number = 0;

        summary_rd.initField();
        /** 承認 */
        int approval = 0;

        /** タイムカード設定を取得 */
        EipTExtTimecardSystem timecard_system =
          ExtTimecardUtils.getEipTExtTimecardSystemByUserId(user_id);

        /**
         * userlistにはユーザー日ごとのタイムカードのResultDataがリストで入っているため、
         * ListResultDataに代入して各日数・時間を計算させる。
         */
        if (userlist != null) {
          for (ExtTimecardResultData rd : userlist) {
            ExtTimecardListResultData lrd = new ExtTimecardListResultData();
            lrd.initField();
            lrd.setDate(rd.getPunchDate().getValue());
            lrd.setRd(rd);
            lrd.setTimecardSystem(timecard_system);
            String type = rd.getType().getValue();

            // morimoto-junichi add 2012/10/09
            if (rd.getStatus().equals("1")) {

              if (type.equals(EipTExtTimecard.TYPE_WORK)
                || type.equals(EipTExtTimecard.TYPE_TIME_HOLIDAY)) {
                /** 出勤 */
                if (lrd.getWorkHour() != ExtTimecardListResultData.NO_DATA) {
                  work_day++;
                  work_hour += lrd.getWorkHourWithoutRestHour();

                  /** 休日加算 */
                  if ("1".equals(rd.getAddholidayFlag().toString())) {
                    addholiday_hour += lrd.getWorkHourWithoutRestHour();
                  }

                  /** 深夜加算 */
                  if (!"1".equals(rd.getWatchFlag().toString()) && !"2".equals(rd.getWatchFlag().toString())) {
                    midnight_hour += lrd.getMidnightHour();
                  }
                }
                /** 時間有休 */
                if (!"".equals(rd.getUsedTime().toString())
                  && !"0".equals(rd.getUsedTime().toString())) {
                  used_time += Integer.parseInt(rd.getUsedTime().toString());
                  work_hour += Integer.parseInt(rd.getUsedTime().toString());
                }
                /** 当直代行 */
                waiting_time +=
                  Integer.parseInt(rd.getWaitingTime().toString());
                /** 当直 */
                if (!rd.getWatchFlag().toString().equals("0")) {
                watch_number += 1;
                }
                /** 一人応援 */
                support_number +=
                  Integer.parseInt(rd.getSupportFlag().toString());
                /** 移動手当て */
                move_number += Integer.parseInt(rd.getMoveFlag().toString());
                /** 車出手当て */
                car_number += Integer.parseInt(rd.getCarFlag().toString());
                /** 遠隔手当１ */
                help1_number += Integer.parseInt(rd.getHelp1Flag().toString());
                /** 遠隔手当２ */
                help2_number += Integer.parseInt(rd.getHelp2Flag().toString());
                /** 休消手当 */
                subholiday_number +=
                  Integer.parseInt(rd.getSubholidayFlag().toString());
              } else if (type.equals(EipTExtTimecard.TYPE_ABSENT)) {
                /** 欠勤 */
                absent_day++;
              } else if (type.equals(EipTExtTimecard.TYPE_HOLIDAY)) {
                /** 有休 */
                paid_holiday++;
                used_time += 8;
                work_day++; // 出勤日数 +1
                work_hour += 8; // 就労時間 +8
              } else if (type.equals(EipTExtTimecard.TYPE_KEICHO)) {
                /** 慶弔休 */
                keicho++;
                work_day++; // 出勤日数 +1
                work_hour += 8; // 就労時間 +8
              } else if (type.equals(EipTExtTimecard.TYPE_TIME_NIKYU)) {
                /** 日給 */
                nikyu_day++;
              }
              approval++;
            } else if (rd.getStatus().equals("0")) {
              /** 却下 */
              rejection_number++;
            } else if (rd.getStatus().equals("9")) {
              // タイムカードへの打刻が完了したレコードのみ抽出する
              if ((rd.getType().getValue().equals("A") // 欠勤である
                || rd.getType().getValue().equals("H") // 有休である
                || rd.getType().getValue().equals("C") // 代休である
                || rd.getType().getValue().equals("E") // その他である
              || rd.getType().getValue().equals("K")) // 慶弔休である
                || (rd.getType().getValue().equals("P") && !rd // 出勤でかつ退勤時刻を打刻済み
                  .getClockOutTime()
                  .getTime()
                  .toString()
                  .equals(""))
                || (rd.getType().getValue().equals("N") && !rd // 日給でかつ退勤時刻を打刻済み
                  .getClockOutTime()
                  .getTime()
                  .toString()
                  .equals(""))
                || (rd.getType().getValue().equals("T") && !rd // 時間有休でかつ退勤時刻を打刻済み
                  .getClockOutTime()
                  .getTime()
                  .toString()
                  .equals(""))) {
                /** 申請中 */
                entry_number++;
              }
            }

          } // morimoto
        } else {
          /** 未入力数の計算のため、長さ0のArrayListをつくる */
          userlist = new ArrayList<ExtTimecardResultData>();
        }

        /** 未入力数を月日数からデータ数を引いて計算する */
        Calendar cal = Calendar.getInstance();
        cal.setTime(getViewMonth().getValue());
        // no_input = cal.getActualMaximum(Calendar.DAY_OF_MONTH) -
        // userlist.size();
        no_input = cal.getActualMaximum(Calendar.DAY_OF_MONTH) - approval;

        /** ユーザーごとの合計をSummaryResultDataに代入する */
        work_hour = ExtTimecardUtils.roundHour(work_hour);
        overtime_hour = ExtTimecardUtils.roundHour(overtime_hour);
        off_hour = ExtTimecardUtils.roundHour(off_hour);
        addholiday_hour = ExtTimecardUtils.roundHour(addholiday_hour);
        midnight_hour = ExtTimecardUtils.roundHour(midnight_hour);

        summary_rd.setviewMonth(viewMonth);
        summary_rd.setUserId(user_id);
        summary_rd.setUserName(eipUser.getAliasName().getValue());
        summary_rd.setUserNameKana(base_user.getLastNameKana()
          + ' '
          + base_user.getFirstNameKana());
        summary_rd.setUserOrder(user_order);
        summary_rd.setEmployeeNumber(base_user.getEmployeeNumber());
        summary_rd.setSystemName(timecard_system.getSystemName());
        summary_rd.setWorkDayHour(work_day, work_hour);
        summary_rd.setOvertimeDayHour(overtime_day, overtime_hour);
        summary_rd.setOffDayHour(off_day, off_hour);
        summary_rd.setLateComingDay(late_coming_day);
        summary_rd.setEarlyLeavingDay(early_leaving_day);
        summary_rd.setAbsentDay(absent_day);
        summary_rd.setPaidHoliday(paid_holiday);
        summary_rd.setKeicho(keicho);
        summary_rd.setCompensatoryHoliday(compensatory_holiday);
        summary_rd.setOtherDay(other_day);
        summary_rd.setNoInput(no_input);
        summary_rd.setUsedTime(used_time);
        summary_rd.setWaitingTime(waiting_time);
        summary_rd.setWatchNumber(watch_number);
        summary_rd.setSupportNumber(support_number);
        summary_rd.setMoveNumber(move_number);
        summary_rd.setCarNumber(car_number);
        summary_rd.setHelp1Number(help1_number);
        summary_rd.setHelp2Number(help2_number);
        summary_rd.setAddholidayHour(addholiday_hour);
        summary_rd.setSubholidayNumber(subholiday_number);
        summary_rd.setNikyuDay(nikyu_day);
        summary_rd.setMidnightHour(midnight_hour);
        summary_rd.setPayrollType(payroll_type);
        summary_rd.setPayrollName(payroll_name);
        summary_rd.setPayrollCode(payroll_code);
        // 部署
        EipMPost post = ALEipUtils.getPostInfo(user_id);
        if (post != null) {
          summary_rd.setPostCode(post.getPostCd());
          summary_rd.setPostName(post.getPostName());
        }
        // 役職
        String position_name =
          ALEipUtils.getPositionName(base_user.getPositionId());
        if (position_name != null) {
          summary_rd.setPositionName(position_name);
        }
        // 所定労働日数
        String yyyymm = viewMonth.toString("yyyy-MM");
        summary_rd.setPrescribedWorkingDay(AccountUtils
          .getPrescribedWorkingDays(Integer.valueOf(eipUser
            .getUserId()
            .toString()), yyyymm));
        // 所定労働時間
        float scheduled_working_hour =
          AccountUtils.getScheduledWorkingHours(Integer.valueOf(eipUser
            .getUserId()
            .toString()), yyyymm);
        summary_rd.setScheduledWorkingHour(scheduled_working_hour);
        // 申請中
        summary_rd.setEntryNumber(entry_number);
        // 却下
        summary_rd.setRejectionNumber(rejection_number);
        // ================================================
        // 残業を計算する
        //
        // (残業時間) = (総就労時間) - (当直代行時間) - (所定労働時間) - (みなし残業)
        // (申請する残業時間) = (残業時間) + (当直代行時間) ※当直代行時間については必ず残業時間へ含めるため
        //
        float minasi =
          Float.valueOf(ALEipUtils.getDeemedOvertimeHour(base_user
            .getPositionId()));
        float waiting_hour = Float.valueOf(waiting_time) / 60;
        float zangyo =
          work_hour - waiting_hour - scheduled_working_hour - minasi;
        if (zangyo < 0) {
          zangyo = 0;
        }
        zangyo = zangyo + waiting_hour;
        // ================================================
        // 算出した残業時間を【法定内残業】と【法定外残業】へ割り振る
        // 法定労働時間
        float legal_working_hour = getLegalWorkingHour(yyyymm);
        // 法定内残業の最大値(法定労働時間 - 所定労働時間） もしくは （法定労働時間 - 総労働時間)のいずれか大きい方
        float within_legal_overtime_max = 0;
        float within_legal_overtime_max1 =
          (legal_working_hour - scheduled_working_hour < 0)
            ? 0
            : (legal_working_hour - scheduled_working_hour);
        float within_legal_overtime_max2 =
          (legal_working_hour - work_hour < 0)
            ? 0
            : (legal_working_hour - work_hour);
        if (within_legal_overtime_max1 > within_legal_overtime_max2) {
          within_legal_overtime_max = within_legal_overtime_max1;
        } else {
          within_legal_overtime_max = within_legal_overtime_max2;
        }
        // 法定内残業
        float within_legal_overtime = 0;
        // 法定外残業
        float outside_legal_overtime = 0;
        if (within_legal_overtime_max < zangyo) {
          within_legal_overtime = within_legal_overtime_max;
          outside_legal_overtime = zangyo - within_legal_overtime;
        } else {
          within_legal_overtime = zangyo;
          outside_legal_overtime = 0;
        }
        // 60h越残業
        float overtime_60 = 0;
        if (zangyo > 60) {
          overtime_60 = zangyo - 60;
        }
        within_legal_overtime =
          ExtTimecardUtils.roundHour(within_legal_overtime);
        outside_legal_overtime =
          ExtTimecardUtils.roundHour(outside_legal_overtime);
        overtime_60 = ExtTimecardUtils.roundHour(overtime_60);
        summary_rd.setWithinLegalOvertime(within_legal_overtime);
        summary_rd.setOutsideLegalOvertime(outside_legal_overtime);
        summary_rd.setOvertime60(overtime_60);
        // ================================================
        list.add(summary_rd);
      }
    }
    return list;
  }

  /**
   * 月の日数からその月の法定労働時間を返す
   *
   * @param yyyyMM
   * @return
   */
  public float getLegalWorkingHour(String yyyyMM) {
    String[] ym = yyyyMM.split("-");
    int year = Integer.valueOf(ym[0]);
    int month = Integer.valueOf(ym[1]);

    Calendar cal = new GregorianCalendar(year, (month - 1), 1);
    int maxDate = cal.getActualMaximum(Calendar.DAY_OF_MONTH);

    float working_hour = 0;
    switch (maxDate) {
      case 28:
        working_hour = 160;
        break;
      case 29:
        working_hour = 165;
        break;
      case 30:
        working_hour = 171;
        break;
      case 31:
        working_hour = 177;
        break;
      default:
        working_hour = 177;
        break;
    }

    return working_hour;
  }

  /**
   * 指定されたユーザーIDの集計情報を締め情報からセットする
   *
   * @param user_id
   * @return
   */
  public ExtTimecardSummaryResultData getAggregateResultData(int userid) {
    ExtTimecardSummaryResultData summary_rd =
      new ExtTimecardSummaryResultData();
    summary_rd.initField();

    try {
      // 締め情報を取得
      String yyyymm = viewMonth.toString("yyyy-MM");
      SelectQuery<ExtTimecardAggregateUser> query =
        Database.query(ExtTimecardAggregateUser.class);
      Expression exp0 =
        ExpressionFactory.matchExp(
          ExtTimecardAggregateUser.USER_ID_PROPERTY,
          userid);
      Expression exp1 =
        ExpressionFactory.matchExp(
          ExtTimecardAggregateUser.AGGREGATE_YEARS_PROPERTY,
          yyyymm);
      query.setQualifier(exp0.andExp(exp1));
      List<ExtTimecardAggregateUser> ag_list = query.fetchList();

      if (ag_list != null && ag_list.size() > 0) {
        ExtTimecardAggregateUser ag = ag_list.get(0);
        summary_rd.setviewMonth(viewMonth);
        summary_rd.setUserId(ag.getUserId());
        summary_rd.setUserName(ag.getUserName());
        summary_rd.setUserNameKana(ag.getUserNameKn());
        summary_rd.setUserOrder(ag.getUserOrder().toString());
        summary_rd.setEmployeeNumber(ag.getEmployeeNumber());
        summary_rd.setSystemName(ag.getWorkingArrangements());
        if ("B".equals(ag.getPayrollSystemType())) {
          summary_rd.setWorkDayHour(getDefaultInt(ag
            .getAttendanceDays()), getDefaultFloat(ag.getTotalWorkingHours()));
        } else {
          summary_rd.setWorkDayHour(getDefaultInt(ag
            .getAttendanceDays()), getDefaultFloat(ag.getAttendanceTime()));
        }
        summary_rd.setOvertimeDayHour(0, 0);
        summary_rd.setOffDayHour(0, 0);
        summary_rd.setLateComingDay(0);
        summary_rd.setEarlyLeavingDay(0);
        summary_rd.setAbsentDay(0);
        summary_rd.setPaidHoliday(0);
        summary_rd.setKeicho(getDefaultInt(ag.getCondolence()));
        summary_rd.setCompensatoryHoliday(0);
        summary_rd.setOtherDay(0);
        summary_rd.setNoInput(0);
        summary_rd.setUsedTime(getDefaultInt(ag.getPaidLeave()));
        summary_rd.setWaitingTime(0);
        summary_rd.setWatchNumber(getDefaultInt(ag.getDutyAllowance()));
        summary_rd.setSupportNumber(getDefaultInt(ag.getSupportAllowance()));
        summary_rd.setMoveNumber(0);
        summary_rd.setCarNumber(getDefaultInt(ag.getOutCarAllowance()));
        summary_rd.setHelp1Number(getDefaultInt(ag.getRemoteAllowance1()));
        summary_rd.setHelp2Number(getDefaultInt(ag.getRemoteAllowance2()));
        summary_rd.setAddholidayHour(getDefaultFloat(ag.getHolidayAddition()));
        summary_rd.setSubholidayNumber(getDefaultInt(ag
          .getHolidayConsumptionAllowance()));
        summary_rd.setNikyuDay(getDefaultInt(ag.getDailyWage()));
        summary_rd.setMidnightHour(getDefaultFloat(ag.getMidnightAddition()));
        summary_rd.setPayrollType(ag.getPayrollSystemType());
        summary_rd.setPayrollName(ag.getPayrollSystem());
        summary_rd.setPayrollCode(ag.getPayrollSystemCd());
        summary_rd.setPostCode(ag.getPostCd());
        summary_rd.setPostName(ag.getPost());
        summary_rd.setPositionName(ag.getPosition());
        summary_rd.setPrescribedWorkingDay(getDefaultInt(ag
          .getPrescribedWorkingDays()));
        summary_rd.setScheduledWorkingHour(getDefaultFloat(ag
          .getScheduledWorkingHours()));
        summary_rd.setEntryNumber(0);
        summary_rd.setRejectionNumber(0);
        summary_rd.setWithinLegalOvertime(getDefaultFloat(ag
          .getWithinLegalOvertime()));
        summary_rd.setOutsideLegalOvertime(getDefaultFloat(ag
          .getOutsideLegalOvertime()));
        summary_rd.setOvertime60(getDefaultFloat(ag.getSixtyHoursOvertime()));
      }

    } catch (Exception ex) {
      logger.error("Exception[getExtTimecardAggregateUser]", ex);
    }

    return summary_rd;

  }

  public int getDefaultInt(String num) {
	  if ("".equals(num) || num == null) {
		  return 0;
	  } else {
		  return Integer.valueOf(num);
	  }
  }
  public float getDefaultFloat(String num) {
	  if ("".equals(num) || num == null) {
		  return 0;
	  } else {
		  return Float.valueOf(num);
	  }
  }

  /**
   * @return
   */
  public List<Integer> getDateListKeys() {
    try {
      List<Integer> list = new ArrayList<Integer>();
      Set<Integer> set = datemap.keySet();
      Iterator<Integer> iter = set.iterator();
      while (iter.hasNext()) {
        list.add(iter.next());
      }

      if (list.size() > 1) {
        for (int i = 0; i < list.size() - 1; i++) {
          ExtTimecardSummaryResultData listrd1 = datemap.get(list.get(i));
          ExtTimecardSummaryResultData listrd2 = datemap.get(list.get(i + 1));
          int listrd1_size = listrd1.getList().size();
          if (listrd1_size > 0) {
            ExtTimecardResultData listrd1_lastrd =
              listrd1.getList().get(listrd1_size - 1);

            ExtTimecardResultData listrd2_firstrd = listrd2.getList().get(0);
            if (ExtTimecardUtils.WORK_FLG_OFF.equals(listrd2_firstrd
              .getWorkFlag()
              .getValue())
              && ExtTimecardUtils.WORK_FLG_ON.equals(listrd1_lastrd
                .getWorkFlag()
                .getValue())
              && !sameDay(
                listrd1_lastrd.getWorkDate().getValue(),
                listrd2_firstrd.getWorkDate().getValue())) {

              Date d = listrd2_firstrd.getWorkDate().getValue();
              Calendar cal = Calendar.getInstance();
              cal.setTime(d);
              cal.set(Calendar.HOUR_OF_DAY, 0);
              cal.set(Calendar.MINUTE, 0);

              ExtTimecardResultData dummyrd = new ExtTimecardResultData();
              dummyrd.initField();
              dummyrd.setWorkFlag(ExtTimecardUtils.WORK_FLG_DUMMY);
              dummyrd.setWorkDate(cal.getTime());

              listrd1.addExtTimecardResultData(dummyrd);
            }
          }
        }
      }
      return list;
    } catch (Exception e) {
      logger.error("[ExtTimecardSummaryListSelectData]", e);
      return null;
    }
  }

  /**
   * 指定した2つの日付を比較する．
   *
   * @param date1
   * @param date2
   * @param checkTime
   *          時間まで比較する場合，true．
   * @return 等しい場合，0. date1>date2の場合, 1. date1 <date2の場合, 2.
   */
  private boolean sameDay(Date date1, Date date2) {
    Calendar cal1 = Calendar.getInstance();
    Calendar cal2 = Calendar.getInstance();
    cal1.setTime(date1);
    cal2.setTime(date2);

    int date1Year = cal1.get(Calendar.YEAR);
    int date1Month = cal1.get(Calendar.MONTH) + 1;
    int date1Day = cal1.get(Calendar.DATE);
    int date2Year = cal2.get(Calendar.YEAR);
    int date2Month = cal2.get(Calendar.MONTH) + 1;
    int date2Day = cal2.get(Calendar.DATE);

    if (date1Year == date2Year
      && date1Month == date2Month
      && date1Day == date2Day) {
      return true;
    }
    return false;
  }

  /**
   * userListを設定する．
   *
   */
  public void setuserList(String target_group_name) {
    this.target_group_name = target_group_name;
    if ((this.target_group_name != null)
      && (!this.target_group_name.equals(""))
      && (!this.target_group_name.equals("all"))
      && (!this.target_group_name.equals("only"))) {
      userList = ALEipUtils.getMainPostUsers(target_group_name);
    } else if (this.target_group_name.equals("all")
      || this.target_group_name.equals("only")) {
      userList = getUserList(Integer.parseInt(userid));
    } else {
      userList = ALEipUtils.getUsers("LoginUser");
    }
  }

  /**
   * 表示切り替え時に指定するグループ名
   *
   * @return
   */
  public String getTargetGroupName() {
    return target_group_name;
  }

  /**
   * 表示切り替え時に指定するユーザ ID
   *
   * @return
   */
  public String getTargetUserId() {
    return target_user_id;
  }

  /**
   * 指定グループに属するユーザの一覧を取得する．
   *
   * @param groupname
   * @return
   */
  public List<ALEipUser> getUsers() {
    if (hasAclSummaryOther || hasAclSummaryAdmin) {
      return userList;
    } else {
      try {
        List<ALEipUser> users = new ArrayList<ALEipUser>();
        users.add(ALEipUtils.getALEipUser(Integer.parseInt(userid)));
        return users;
      } catch (Exception e) {
        return null;
      }
    }
  }

  /**
   * 部署の一覧を取得する．
   *
   * @return
   */
  public Map<Integer, ALEipPost> getPostMap() {
    if (hasAclSummaryAdmin) {
      return ALEipManager.getInstance().getPostMap();
    } else if (hasAclSummaryOther) {
      // return ALEipManager.getInstance().getPostMap();
      // 自分が所属する部署のIDリスト
      List<Integer> list =
        ALEipUtils.getPostIdList(Integer.valueOf(this.userid));
      // すべての部署リスト
      Map<Integer, ALEipPost> map = ALEipManager.getInstance().getPostMap();
      Map<Integer, ALEipPost> tmp = new LinkedHashMap<Integer, ALEipPost>();
      for (Integer i : list) {
        tmp.put(i, map.get(i));
      }
      return tmp;
    } else {
      return null;
    }
  }

  /**
   * My グループの一覧を取得する．
   *
   * @return
   */
  public List<ALEipGroup> getMyGroupList() {
    if (hasAclSummaryOther || hasAclSummaryAdmin) {
      return myGroupList;
    } else {
      return null;
    }
  }

  /**
   * ログインユーザの ID を取得する．
   *
   * @return
   */
  public String getUserId() {
    return userid;
  }

  /**
   *
   * @param id
   * @return
   */
  public boolean isMatch(int id1, long id2) {
    return id1 == (int) id2;
  }

  /**
   * @return
   */

  public String getNowTime() {
    return nowtime;
  }

  /**
   *
   * @param minute
   * @return
   */
  @SuppressWarnings("unused")
  private String minuteToHour(long minute) {
    BigDecimal decimal = new BigDecimal(minute / 60.0);
    DecimalFormat dformat = new DecimalFormat("##.#");
    String str =
      dformat.format(decimal.setScale(1, BigDecimal.ROUND_FLOOR).doubleValue());
    return str;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  /**
   * 表示開始日時を取得します。
   *
   * @return
   */
  public ALDateTimeField getViewStart() {
    return viewStart;
  }

  /**
   * 表示終了日時を取得します。
   *
   * @return
   */
  public ALDateTimeField getViewEnd() {
    return viewEnd;
  }

  /**
   * 表示終了日時 (Criteria) を取得します。
   *
   * @return
   */
  public ALDateTimeField getViewEndCrt() {
    return viewEndCrt;
  }

  /**
   * 前の月を取得します。
   *
   * @return
   */
  public ALDateTimeField getPrevMonth() {
    return prevMonth;
  }

  /**
   * 次の月を取得します。
   *
   * @return
   */
  public ALDateTimeField getNextMonth() {
    return nextMonth;
  }

  /**
   * 今月を取得します。
   *
   * @return
   */
  public ALDateTimeField getCurrentMonth() {
    return currentMonth;
  }

  /**
   * 現在の月を取得します。
   *
   * @return
   */
  public ALDateTimeField getViewMonth() {
    return viewMonth;
  }

  /**
   * 今日を取得します。
   *
   * @return
   */
  public ALDateTimeField getToday() {
    return today;
  }

  public boolean hasAclXlsExport() {
    return hasAclXlsExport;
  }

  public String getMode() {
    return MODE;
  }

  public String getLoginUserName() {
    try {
      ALEipUser user = ALEipUtils.getALEipUser(Integer.valueOf(userid));
      return user.getAliasName().toString();
    } catch (Exception ignore) {
      return null;
    }
  }

  /**
   * タイムカードの承認権限があるか
   *
   * @return
   */
  public boolean getHasAclSummaryOther() {
    return this.hasAclSummaryOther;
  }

  /**
   * タイムカードの全社員の承認権限があるか
   *
   * @return
   */
  public boolean getHasAclSummaryAdmin() {
    return this.hasAclSummaryAdmin;
  }

  /**
   * 当月の締め処理が行われているか
   */
  public boolean getExtTimecardAggregate() {
	  return ExtTimecardUtils.getExtTimecardAggregate(viewMonth.toString("yyyy-MM"));
  }

  /**
   * 締め処理済みの人数
   */
  public int getTimecardAggregateUserCount() {

     String yyyymm = viewMonth.toString("yyyy-MM");
	 SelectQuery<ExtTimecardAggregateUser> query = Database.query(ExtTimecardAggregateUser.class);
	 Expression exp1 =
			 ExpressionFactory.matchExp(ExtTimecardAggregateUser.AGGREGATE_YEARS_PROPERTY, yyyymm);
	 query.setQualifier(exp1);

	 return query.getCount();
  }

  /**
   *
   */
  public int getMemberCount() {
	  return current_user_count;
  }
}
