/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.event;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.ecs.xhtml.object;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.log.output.io.rotate.RevolvingFileStrategy;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALStringField;
//import com.aimluck.eip.cayenne.om.portlet.OriTMeyasu;
import com.aimluck.eip.cayenne.om.portlet.OriTEvent;
import com.aimluck.eip.cayenne.om.portlet.OriTEventMap;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.event.util.EventUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * イベントのの検索データを管理するクラスです。 <BR>
 *
 */

public class EventSelectData extends ALAbstractSelectData<OriTEvent, OriTEvent>
    implements ALData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(EventSelectData.class.getName());

  /** サブメニュー（作成） */
  public static final String SUBMENU_CREATED = "created";

  /** サブメニュー（未回答） */
  public static final String SUBMENU_UNANSWERED = "unanswered";

  /** サブメニュー（回答済み） */
  public static final String SUBMENU_ANSWERED = "answered";

  /** 現在選択されているサブメニュー */
  private String currentSubMenu;

  /**  イベントの総数 */
  private int eventSum;

  /** メモ一覧 */
  //private List<MeyasuLiteResultData> memoLiteList;

  /** イベント一覧 */
  private List<EventResultData> eventLiteList;

  /** イベント参加ユーザ一覧 */
  private List<OriTEventMap> eventMapData;

  /** 検索用 */
  private ALStringField target_keyword;

  /** ログインユーザID */
  private int log_in_user_id;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    String sort = ALEipUtils.getTemp(rundata, context, LIST_SORT_STR);
    if (sort == null || sort.equals("")) {
      ALEipUtils.setTemp(rundata, context, LIST_SORT_STR, ALEipUtils
        .getPortlet(rundata, context)
        .getPortletConfig()
        .getInitParameter("p2a-sort"));
    }

    String subMenuParam = rundata.getParameters().getString("submenu");
    currentSubMenu = ALEipUtils.getTemp(rundata, context, "submenu");
    if (subMenuParam == null && currentSubMenu == null) {
      ALEipUtils.setTemp(rundata, context, "submenu", SUBMENU_UNANSWERED);
      currentSubMenu = SUBMENU_UNANSWERED;
    } else if (subMenuParam != null) {
      ALEipUtils.setTemp(rundata, context, "submenu", subMenuParam);
      currentSubMenu = subMenuParam;
    }


    // ログインユーザーIDを取得
    log_in_user_id = ALEipUtils.getUserId(rundata);
  }


  /**
   * 詳細データを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */

@Override
protected OriTEvent selectDetail(RunData rundata, Context context)
		throws ALPageNotFoundException, ALDBErrorException {
    try {
        //String eventid = rundata.getParameters().getString("eventid"); //ALEipUtils.getTemp(rundata, context, "eventid");

        OriTEvent event = EventUtils.getOriTEvent(rundata, context);

        // 参加者を取得
        eventMapData = EventUtils.getOriTEventMap(rundata, context);


        if (event == null || eventMapData == null) {
          return null;
        }

        return event;

      } catch (ALPageNotFoundException pageNotFound) {
        throw pageNotFound;
      } catch (Exception ex) {
        logger.error("Exception", ex);
        return null;
      }
}


  /**
   *
   * @param rundata
   * @param context
   */

  public void loadEventIdList(RunData rundata, Context context) {
    try {

    	// ログインユーザーIDを取得
        log_in_user_id = ALEipUtils.getUserId(rundata);

        String subMenuParam = rundata.getParameters().getString("submenu");
        currentSubMenu = ALEipUtils.getTemp(rundata, context, "submenu");

        if (subMenuParam == null) {
          ALEipUtils.setTemp(rundata, context, "submenu", SUBMENU_UNANSWERED);
          currentSubMenu = SUBMENU_UNANSWERED;
        } else {
          currentSubMenu = rundata.getParameters().getString("submenu");
        }

      // イベント一覧
    	eventLiteList = new ArrayList<EventResultData>();

      SelectQuery<OriTEvent> query = getSelectQuery(rundata, context);
      List<OriTEvent> list = query.fetchList();

      for (OriTEvent model : list) {
        EventResultData rd = new EventResultData();
        rd.initField();
        rd.setEventId(model.getOriTEventId().toString());
        rd.setCreateId(model.getCreateId());
        rd.setCreateUserName(ALEipUtils.getUserFullName(model.getCreateId()));
        rd.setEventName(model.getEventName());
        rd.setClosingDate(model.getClosingDate());
        rd.setCreateDate(model.getCreateDate());
        rd.setUpdateDate(model.getUpdateDate());


        for (int j = 0; j < model.getOriTEventMap().size(); j++) {
        	OriTEventMap map = (OriTEventMap)model.getOriTEventMap().get(j);
        	if (getLogInUserId() == map.getUserId()) {
        		rd.setUserStatus(map.getStatus());
        	}
        }

        // 締切フラグ
        Date now = Calendar.getInstance().getTime();
        int diff = now.compareTo(model.getClosingDate());
        if (diff > 0) {
      	  rd.setClosingFlg("1");
        } else{
      	  rd.setClosingFlg("0");
        }

        eventLiteList.add(rd);

      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }
  }


  /**
   * 一覧データを取得します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */

  @Override
  public ResultList<OriTEvent> selectList(RunData rundata, Context context) {
    try {

      SelectQuery<OriTEvent> query = getSelectQuery(rundata, context);
      buildSelectQueryForListView(query);
      buildSelectQueryForListViewSort(query, rundata, context);

      ResultList<OriTEvent> list = query.getResultList();
      // Event の総数をセットする．
      eventSum = list.getTotalCount();

      return list;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }


  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   *
   * @param rundata
   * @param context
   * @return
   */

  private SelectQuery<OriTEvent> getSelectQuery(RunData rundata, Context context) {

	  SelectQuery<OriTEvent> query = Database.query(OriTEvent.class);

      // 受信したもの
	  Expression exp =
			  ExpressionFactory.matchExp(OriTEvent.ORI_TEVENT_MAP_PROPERTY
					  + "."
					  + OriTEventMap.USER_ID_PROPERTY,
					  Integer.valueOf(ALEipUtils.getUserId(rundata)));
	  // 未回答のもの
	  Expression exp1 =
			  ExpressionFactory.matchExp(OriTEvent.ORI_TEVENT_MAP_PROPERTY
					  + "."
					  + OriTEventMap.STATUS_PROPERTY, EventUtils.STATUS_UNSELECTED);
	  // 回答済み
	  String[] strs = { EventUtils.STATUS_PARTICIPATION, EventUtils.STATUS_NONPARTICIPATION, EventUtils.STATUS_HOLD, EventUtils.STATUS_HOLD };
	  Expression exp2 =
			  ExpressionFactory.inExp(OriTEvent.ORI_TEVENT_MAP_PROPERTY
					  + "."
					  + OriTEventMap.STATUS_PROPERTY, strs);

	  // 作成したもの
	  Expression exp3 =
			  ExpressionFactory.matchExp(OriTEvent.CREATE_ID_PROPERTY, Integer.valueOf(ALEipUtils.getUserId(rundata)));


	  // 作成したもの
	  if (SUBMENU_CREATED.equals(currentSubMenu)){
		  query.setQualifier(exp3);
	  } else if (SUBMENU_UNANSWERED.equals(currentSubMenu)) {
	  // 未回答のもの
		  query.setQualifier(exp.andExp(exp1));
	  } else if (SUBMENU_ANSWERED.equals(currentSubMenu)) {
	  // 回答済み
		  query.setQualifier(exp.andExp(exp2));
	  } else {
		  query.setQualifier(exp.andExp(exp1));
	  }

    query.orderDesending(OriTEvent.CREATE_DATE_PROPERTY);

    return buildSelectQueryForFilter(query, rundata, context);
  }


  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   *
   * @param obj
   * @return
   */

  @Override
  protected Object getResultData(OriTEvent record) {
    try {
      EventResultData rd = new EventResultData();
      rd.initField();
      rd.setEventId(record.getOriTEventId().toString());
      rd.setCreateId(record.getCreateId());
      rd.setEventName(record.getEventName());
      rd.setClosingDate(record.getClosingDate());
      rd.setCreateDate(record.getCreateDate());
      rd.setUpdateDate(record.getUpdateDate());
      rd.setCreateUserName(ALEipUtils.getUserFullName(record.getCreateId()));


      for (int i = 0; i < record.getOriTEventMap().size(); i++) {
      OriTEventMap map = (OriTEventMap)record.getOriTEventMap().get(i);
        if (getLogInUserId() == map.getUserId()) {
          rd.setUserStatus(map.getStatus());
        }
      }

      // 締切フラグ
      Date now = Calendar.getInstance().getTime();
      int diff = now.compareTo(record.getClosingDate());
      if (diff > 0) {
    	  rd.setClosingFlg("1");
      } else{
    	  rd.setClosingFlg("0");
      }

      return rd;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * ResultData に値を格納して返します。（詳細データ） <BR>
   *
   * @param obj
   * @return
   */

  @Override
  protected Object getResultDataDetail(OriTEvent record) {
    try {
      // morimoto
      List<EventMapResultData> eventMapResultData = new ArrayList<EventMapResultData>();
      for(int i = 0; i < eventMapData.size(); i++){
        EventMapResultData maprd = new EventMapResultData();
        maprd.initField();
        maprd.setId(eventMapData.get(i).getOriTEventMapId());
        maprd.setEventId(eventMapData.get(i).getEventId());
        maprd.setUserId(eventMapData.get(i).getUserId());
        maprd.setStatus(eventMapData.get(i).getStatus());
        maprd.setUserName(ALEipUtils.getUserFullName(eventMapData.get(i).getUserId()));

        ALStringField userComment = new ALStringField();
        userComment.setValue(eventMapData.get(i).getComment());
        maprd.setUserComment(userComment);
        eventMapResultData.add(maprd);
      }

      EventResultData rd = new EventResultData();
      rd.initField();
      rd.setEventId(record.getOriTEventId().toString());
      rd.setEventName(record.getEventName());
      rd.setCreateId(record.getCreateId());
      rd.setComment(record.getComment());
      rd.setClosingDate(record.getClosingDate());
      rd.setCreateDate(record.getCreateDate());
      rd.setUpdateDate(record.getUpdateDate());
      rd.setCreateUserName(ALEipUtils.getUserFullName(record.getCreateId()));
      // morimoto 追加
      rd.setEventMapResultData(eventMapResultData);

      // 締切フラグ
      Date now = Calendar.getInstance().getTime();
      int diff = now.compareTo(record.getClosingDate());
      if (diff > 0) {
    	  rd.setClosingFlg("1");
      } else{
    	  rd.setClosingFlg("0");
      }

      return rd;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }


  /**
   * Event の総数を返す． <BR>
   *
   * @return
   */

  public int getEventSum() {
    return eventSum;
  }


  /**
   * @return
   *
   */

  @Override
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();

    map.putValue("event_id", OriTEvent.EVENT_ID_PK_COLUMN);
    map.putValue("create_id", OriTEvent.CREATE_ID_PROPERTY);
    map.putValue("event_name", OriTEvent.EVENT_NAME_PROPERTY);
    map.putValue("comment", OriTEvent.COMMENT_PROPERTY);
    map.putValue("closing_date", OriTEvent.CLOSING_DATE_PROPERTY);
    map.putValue("create_date", OriTEvent.CREATE_DATE_PROPERTY);
    map.putValue("update_date", OriTEvent.UPDATE_DATE_PROPERTY);

    return map;
  }


  public List<EventResultData> getEventLiteList() {
    return eventLiteList;
  }



/**
 * @return log_in_user_id
 */
public int getLogInUserId() {
	return log_in_user_id;
}

  /**
   * @return target_keyword
   */
  /*
  public ALStringField getTargetKeyword() {
    return target_keyword;
  }
  */

public String getCurrentSubMenu() {
    return this.currentSubMenu;
  }


}
