﻿//#define DEBUG
//#define DAT

// For .NET Framework v3.5

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.IO;
using System.Text;
using System.Net;

/****************  00:00:00.00 集計ツール  ****************/

[assembly: System.Reflection.AssemblyTitle("00:00:00.00 集計ツール")]
[assembly: System.Reflection.AssemblyFileVersion("0.6.0")]

/***** USAGE *****
 * ex. url => http://hayabusa.2ch.net/test/read.cgi/news4vip/1234567890/
 * > 00.00.00.00 http://hayabusa.2ch.net/test/read.cgi/news4vip/1234567890/
 * > 00.00.00.00 1234567890 hayabusa.2ch.net
 */

namespace Vip {
    static class Rank00 {
        const string VERSION = "0.6.0";

        static string dat, server = "hayabusa.2ch.net", board = "news4vip";
        static string url;
        static int range = 1, rankMax = 20, from = 1, count, countRange = 5;
        static string[] ngId;
        static bool local = false;
		static string localPath;

        static List<Res> resList = new List<Res>();

        static readonly Encoding Shift_JIS = Encoding.GetEncoding("Shift_JIS");

        struct Res {
            public int Number;
            public TimeSpan Distance;
            public string Time;
            public string ID;

            public long Duration {
                get { return this.Distance.Duration().Ticks; }
            }

            public Res(int number, TimeSpan distance, string time, string id) {
                this.Number = number;
                this.Distance = distance;
                this.Time = time;
                this.ID = id;
            }
        }

        static void Main(string[] args) {
            // ********** パラメータ取得 **********

            // (Usage) 引数一覧を表示して終了
            if (Array.IndexOf<string>(args, "-h") != -1
                || Array.IndexOf<string>(args, "-?") != -1
                || Array.IndexOf<string>(args, "--help") != -1) {
                Console.WriteLine("00:00:00.00 集計ツール Ver." + VERSION);
                Console.WriteLine();
                ShowUsage(); return;
            }

            int index;

            // * 開始レス番号
            // >>(from) ～ 最後までを判定に利用
            // (デフォルト値: 1)
            index = Array.IndexOf<string>(args, "--from");
            if (index != -1) {
                if (args.Length == index + 1 || !int.TryParse(args[index + 1], out from) || from < 1) {
                    Console.Error.WriteLine("Error: --from の値が正しくありません");
                    return;
                }
                args[index + 1] = "-";
            }

            // * NGID
            // 除外したい ID:*** をカンマ区切りで指定
            // ex. --ngid oBJccH+E0,7DnAJlQ/P
            index = Array.IndexOf<string>(args, "-n");
            if (index == -1) index = Array.IndexOf<string>(args, "--ngid");
            if (index != -1) {
                if (args.Length == index + 1) {
                    Console.Error.WriteLine("Error: -n / --ngid の値が正しくありません");
                    return;
                }
                ngId = args[index + 1].Split(',');
                args[index + 1] = "-";
            }

            // * 集計対象 / 参加人数の対象範囲
            // ±(range)秒以内を集計対象に (デフォルト値: 1)
            // ±(countRange)秒以内を参加人数の対象に (デフォルト値: 5)
            index = Array.IndexOf<string>(args, "-r");
            if (index == -1) index = Array.IndexOf<string>(args, "--range");
            if (index != -1) {
                if (args.Length == index + 1) {
                    Console.Error.WriteLine("Error: -r / --range の値が正しくありません");
                    return;
                }
                if (args[index + 1].Contains(",")) {
                    string[] pair = args[index + 1].Split(',');
                    if (pair.Length != 2
                        || !int.TryParse(pair[0], out range) || range < 1
                        || !int.TryParse(pair[1], out countRange) || countRange < range) {
                        Console.Error.WriteLine("Error: -r / --range の値が正しくありません");
                        return;
                    }
                } else {
                    if (!int.TryParse(args[index + 1], out range) || range < 1) {
                        Console.Error.WriteLine("Error: -r / --range の値が正しくありません");
                        return;
                    }
                    if (countRange < range) countRange = range;
                }
                args[index + 1] = "-";
            }

            // * 最大順位
            // 表示する最大の順位を指定
            index = Array.IndexOf<string>(args, "-m");
            if (index == -1) index = Array.IndexOf<string>(args, "--max");
            if (index != -1) {
                if (args.Length == index + 1 || !int.TryParse(args[index + 1], out rankMax) || rankMax < 1) {
                    Console.Error.WriteLine("Error: -m / --max の値が正しくありません");
                    return;
                }
                args[index + 1] = "-";
            }

            // * ローカルdat利用
            // ローカルのdatを利用する
            // (指定されない場合は、datをダウンロードする)
            index = Array.IndexOf<string>(args, "--local");
            if (index != -1) {
                if (args.Length == index + 1) {
                    Console.Error.WriteLine("Error: --local の値が正しくありません");
                    return;
                }
                local = true;
                localPath = args[index + 1];
                args[index + 1] = "-";
            }

            args = Array.FindAll<string>(args, str => str[0] != '-');
            //args = Array.FindAll<string>(args, str => !str.StartsWith("-"));
            if (args.Length == 0) {
                Console.Error.WriteLine("Error: dat または url が指定されていません");
                ShowUsage(); return;
            }


            // ********** URL 解析 **********

            Match m = Regex.Match(args[0], @"^h?ttp://([-\w.]+(?:/[-\w]+)*?)/test/read\.cgi/(\w+)/(\d{10})/", RegexOptions.ECMAScript);
            if (m.Success) {
                server = m.Groups[1].Value;
                board = m.Groups[2].Value;
                dat = m.Groups[3].Value;
                url = args[0];
            } else {
                dat = args[0];
                if (args.Length != 1) {
                    server = args[1];
                    if (args.Length != 2) {
                        board = args[2];
                    }
                }
                url = string.Format("http://{0}/test/read.cgi/{1}/{2}/", server, board, dat);
            }

#if DEBUG
            Console.Write("[DEBUG] ");
            Console.WriteLine("Dat:{0}, Server:{1}, Board:{2}", dat, server, board);
            Console.WriteLine("Range:{0}, From:{1}", range, from);
#endif
            Console.WriteLine(url);


            // ********** datの読込 **********

            string[] resLines;

            // ローカルdatを利用する場合
            if (local) {
                string path = localPath.Replace("{server}", server.Replace('/', '_'))
                                       .Replace("{board}", board)
                                       .Replace("{dat}", dat);

                if (!File.Exists(path)) {
                    Console.Error.WriteLine("Error: ファイルが存在しません");
                    return;
                }

                // datの読込
                resLines = File.ReadAllLines(path, Shift_JIS);
            }
            // datをダウンロードする場合
            else {
                WebClient client = new WebClient();
                client.Encoding = Shift_JIS;
                client.Headers.Set(HttpRequestHeader.UserAgent, "Monazilla/1.00 (Rank00/1.0)");

                string path = string.Format("http://{0}/{1}/dat/{2}.dat", server, board, dat);

                Console.Title = "datをダウンロード中… " + path;

                try {
                    resLines = client.DownloadString(path).Split('\n');
                }
                catch (WebException e) {
                    if (e.Status == WebExceptionStatus.ProtocolError) {
                        switch (((HttpWebResponse)e.Response).StatusCode) {
                            case HttpStatusCode.NotFound:
                                Console.Error.WriteLine("Error: ファイルが見つかりません(dat落ちかも) (404)");
                                break;

                            case HttpStatusCode.Forbidden:
                                Console.Error.WriteLine("Error: アクセスが拒否されました(バーボンかも) (403)");
                                break;

                            default:
                                Console.Error.WriteLine("Error: ダウンロード失敗 ({0})", ((HttpWebResponse)e.Response).StatusDescription);
                                break;
                        }
                    } else if (e.Status == WebExceptionStatus.NameResolutionFailure) {
                        Console.Error.WriteLine("Error: DNSエラー URLを確認してください (NameResolutionFailure)");
                    } else {
                        Console.Error.WriteLine("Error: ダウンロード中にエラーが発生しました ({0})", e.Status);
                    }
                    return;
                }
                Console.Title = "datをダウンロードしました";
            }

#if DAT
            Console.WriteLine("---------- DAT ----------");
            //Console.WriteLine(File.ReadAllText(path, Encoding.GetEncoding("Shift_JIS")));
            Console.WriteLine(File.ReadAllLines(path, Encoding.GetEncoding("Shift_JIS"))[0]);
            return;
#endif

            // ********** datの解析 & レス抽出 **********

            Regex regex = new Regex(@"(\d\d:\d\d:\d\d.\d\d) ID:([0-9a-zA-Z\/+.]{8,})", RegexOptions.Compiled);
            string[] separator = { "<>" };
            TimeSpan oneDay = new TimeSpan(TimeSpan.TicksPerDay);

            // >>(from) ～ 最後までを判定する
            for (int i = from - 1, length = resLines.Length; i < length; ++i) {

                if (resLines[i].Length == 0) continue;

                // 日付/ID の部分を利用
                m = regex.Match(resLines[i].Split(separator, 4, StringSplitOptions.None)[2]);

                if (m.Success) {
                    TimeSpan distance = TimeSpan.Parse(m.Groups[1].Value);

                    // 12:00:00 以降なら、24時間だけ前にずらす
                    // ex. 23:50:00 -> -00:10:00
                    if (distance.Hours >= 12)
                        distance -= oneDay;

                    double sec = Math.Abs(distance.TotalSeconds);

                    // 参加人数の対象になる場合 (±countRange 秒以内)
                    if (sec <= countRange) {
                        count++;  // [TODO] レス数ではなくID数にするか？

                        // 集計対象 (±range 秒以内) かつ NGID でない
                        if (sec <= range &&
                            (ngId == null || Array.IndexOf<string>(ngId, m.Groups[2].Value) == -1)) {

                            resList.Add(new Res(i + 1, distance, m.Groups[1].Value, m.Groups[2].Value));
#if DEBUG
                            Console.WriteLine(distance);
#endif
                        }
                    }
                }
            }

            if (resList.Count == 0) {
                Console.Error.WriteLine("±{0}秒以内のレスがありませんでした", range);
                return;
            }

#if DEBUG
            Console.WriteLine("--------");
#endif

            // ひっくり返すことで、安定ソートっぽくなる模様
            resList.Reverse();
            // 近い順に並び替える
            resList.Sort((a, b) => a.Duration.CompareTo(b.Duration));
#if DEBUG
            foreach (Res res in resList) {
                Console.WriteLine(res.Distance);
            }
            Console.WriteLine();
#endif

            Console.WriteLine("　　　　　{0:yyyy年MM月dd日} 00:00:00.00 ランキング{1}{2}",
                              DateTime.Today,
                              from > 1 ? (" (>>" + from + "～)") : null,
                              ngId != null ? " (除外有)" : null);

            long lastDuration = 0;
            string rankString;

            for (int i = 0, length = resList.Count; i < length; ++i) {
                Res res = resList[i];

                long duration = res.Duration;
                if (duration == 0)
                    rankString = "ネ申.";
                else if (duration == lastDuration)
                    rankString = " 　 　 ";
                else {
                    if (i >= rankMax) break;  // 最大順位オーバー
                    rankString = (i < 9 ? ". " : null) + (i + 1) + "位 ";
                    lastDuration = duration;
                }

                string device = null;
                switch (res.ID[res.ID.Length - 1]) {
                    case '0':
                    case 'o':
                        device = "　(PC)";
                        break;
                    case 'O':
                    case 'Q':
                        device = "　(携帯)";
                        break;
                    case 'P':
                        device = "　(p2)";
                        break;
                    case 'I':
                    case 'i':
                        device = "　(iPhone)";
                        break;
                }

                Console.WriteLine("　{0}{1:+0.00sec ;-0.00sec ;　　　　　　}{2} >>{3} ID:{4}{5}{6}",
                    rankString, res.Distance.TotalSeconds, res.Time, res.Number, res.ID, device,
                    duration == 0 ? "　おめでとう！" : null);
            }
                /*
　　　　#{Time.now.strftime("%Y年%m月%d日")} 00:00:00.00 ランキング#{" (#{dat_arr.size}スレ合算)" if dat_arr.size>1}#{" (>>#{from}～)" unless from.zero?}#{" (除外有)" unless ng_id.empty?}
　　　±1秒以内に入った人： #{list.size} 人　勢い： -----.--
"　#{rank}#{arr[1].distance_s}#{arr[1].to_s} #{"(" if arr[3]}>>#{arr[0]}#{")" if arr[3]} " +
  "#{arr[2]}　(#{device})#{"　おめでとう！" if arr[1].zero?}"
                */

            Console.WriteLine("±{0}秒以内に入った人： {1} 人　　本日の参加者： {2} 人 （±{3}秒）",
                              range, resList.Count,
                              count, countRange);

        }

        private static void ShowUsage() {
            Console.Error.WriteLine(@"引数: 00.00.00.00 (<dat> [<server> [<board>]] | <url>) [--from <n>]
                  [(-n | --ngid) <id>[,<id>...]] [(-r | --range) <sec>[,<sec2>]]
                  [(-m | --max) <rank>] [--local <path> | --dl] [-h | -?]

    <dat>       URLの10桁の数字
    <server>    URLのドメインの部分
    <board>     URLの板ID
    <url>       URLを直接指定 (http://<server>/test/read.cgi/<board>/<dat>/)

    --from      <n>レス目以降を判定に利用
    -n, --ngid  除外したい ID:<id> を , で区切って指定 (ex. oBJccH+E0,7DnAJlQ/P)
    -r, --range 集計対象となる範囲を±<sec>秒で指定 (デフォルト値: 1)
                オプションで参加人数の対象になる範囲を±<sec2>秒で指定できる
                (デフォルト値: 5)  ※ <sec> は <sec2> 以下であること
    -m, --max   表示する最大の順位(<rank>位まで)を指定
    --local     ローカルのdatを利用する datへのパスを<path>で指定
    --dl        datをダウンロードして利用する (両方未指定なら --dl を指定したと
                して扱う)
    -h, -?      この引数一覧を表示する

Example: 00.00.00.00 1234567890
         00.00.00.00 1234567890 xxx.2ch.net news4vip
         00.00.00.00 http://xxx.2ch.net/test/read.cgi/news4vip/1234567890/");
            //Console.Error.WriteLine("引数: <dat>[,<dat>...] [<server> [<board>]] [-res] [-from <n>] [-ng <id>[,<id>...]]");
        }
    }
}