package main;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.StringTokenizer;

/**
 *
 * @author oshino 数独パズルをモデル化したクラス パズルのセルを2次元配列で管理し、
 * そのセルにアクセスするためのメソッドからなる。
 * セルには1-9の値が入り、未確定値は0で表す。
 */
public class Puzzle {
	private int[][] cells = new int[9][9];

	public Puzzle() {

	}

	public Puzzle(String sudokuFilePath) {
		// 数独問題CSVファイル読み込み
		try {
			FileReader in = new FileReader(sudokuFilePath);
			BufferedReader br = new BufferedReader(in);
			String line;
			int rowIndex = 0;
			while ((line = br.readLine()) != null) {
				StringTokenizer st = new StringTokenizer(line, ",");
				int columnIndex = 0;
				while (st.hasMoreTokens()) {
					int value = Integer.parseInt(st.nextToken());
					cells[rowIndex][columnIndex] = value;
					columnIndex++;
				}
				rowIndex++;
			}
			br.close();
			in.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * セルの値を取得する。
	 *
	 * @param rowIndex
	 *            行インデックス
	 * @param columnIndex
	 *            列インデックス
	 * @return (rowIndex, columnIndex)の値
	 */
	public int getCell(int rowIndex, int columnIndex) {
		return cells[rowIndex][columnIndex];
	}

	/**
	 * セルに値を代入する。
	 *
	 * @param rowIndex
	 *            行インデックス
	 * @param columnIndex
	 *            列インデックス
	 * @param value
	 *            代入する値
	 */
	public void setCell(int rowIndex, int columnIndex, int value) {
		cells[rowIndex][columnIndex] = value;
	}

	/**
	 * @param index
	 *            行インデックス
	 * @return index行目のセルの配列
	 */
	public int[] getRow(int index) {
		return cells[index];
	}

	/**
	 * @param index
	 *            列インデックス
	 * @return index列目のセルの配列
	 */
	public int[] getColumn(int index) {
		int[] column = new int[9];
		for (int i = 0; i < 9; i++) {
			column[i] = cells[i][index];
		}
		return column;
	}

	/**
	 * @param rowIndex
	 *            行インデックス
	 * @param columnIndex
	 *            列インデックス
	 * @return (rowIndex, columnIndex)のセルが所属するブロック
	 */
	public int[] getBlock(int rowIndex, int columnIndex) {
		int[] rows = new int[3];
		if (rowIndex <= 2) {
			rows[0] = 0;
			rows[1] = 1;
			rows[2] = 2;
		} else if (2 < rowIndex && rowIndex <= 5) {
			rows[0] = 3;
			rows[1] = 4;
			rows[2] = 5;
		} else if (5 < rowIndex && rowIndex <= 8) {
			rows[0] = 6;
			rows[1] = 7;
			rows[2] = 8;
		}
		int[] columns = new int[3];
		if (columnIndex <= 2) {
			columns[0] = 0;
			columns[1] = 1;
			columns[2] = 2;
		} else if (2 < columnIndex && columnIndex <= 5) {
			columns[0] = 3;
			columns[1] = 4;
			columns[2] = 5;
		} else if (5 < columnIndex && columnIndex <= 8) {
			columns[0] = 6;
			columns[1] = 7;
			columns[2] = 8;
		}

		int[] block = new int[9];
		int index = 0;
		for (int i = 0; i < 3; i++) {
			for (int j = 0; j < 3; j++) {
				block[index] = getCell(rows[i], columns[j]);
				index++;
			}
		}
		return block;
	}

	/**
	 * パズルが解かれているかどうか
	 *
	 * @return 全てのセルが確定値(1-9)であればtrue, 一つでも未確定値(0)のセルがあればfalse。
	 *
	 */
	public boolean isSolved() {
		for (int[] rows : cells) {
			for (int value : rows) {
				if (value == 0)
					return false; // 未確定値が1つでもあればfalseを返して終了
			}
		}
		return true; // 全ての値が確定値であればtrueを返して終了
	}

	public String toString() {
		StringBuffer sb = new StringBuffer();
		int i, j;
		for (i = 0; i < 9; i++) {
			for (j = 0; j < 9; j++) {
				sb.append(getCell(i, j));
				if (j % 3 == 2)
					sb.append("|");
			}
			if (i < 8) {
				sb.append("\n");
				if (i % 3 == 2)
					sb.append("------------\n");
			}
		}
		return sb.toString();
	}
}
