package main;

import main.Puzzle;

/**
 *
 * @author oshino
 * 数独パズルを解くためのメインとなるクラス。
 */
public class Main {
	public static void main(String[] args) {
		String sudokuFilePath = "sample/sudoku_most_difficult.csv";
		Puzzle puzzle = new Puzzle(sudokuFilePath);

		System.out.println("--- 問題 ---");
		System.out.println(puzzle);

		sudokuSolve(puzzle);

		System.out.println("\n--- 解答 ---");
		System.out.println(puzzle);
	}

	/**
	 * 数独パズルを解く。 引数として渡したpuzzleオブジェクトのセルの値を更新する。
	 *
	 * アルゴリズムはバックトラックを使用。 未確定のセルに代入可能な値を仮定し、矛盾が生じた時点でもとに戻す。
	 * (純粋に全パターンを走査し、探索空間の枝刈りはなし。)
	 *
	 * @param puzzle
	 *            解く対象となるパズル。
	 * @return 解が存在すればtrue、存在しないならばfalse。
	 */
	public static boolean sudokuSolve(Puzzle puzzle) {
		if (puzzle.isSolved()) {
			return true;
		}
		// 数独の全てのセルをチェック
		for (int row = 0; row < 9; row++) {
			for (int column = 0; column < 9; column++) {
				if (puzzle.getCell(row, column) > 0)
					continue; // 確定値のセルは場合はとばす
				// 未確定値のセルに代入可能な値を探し、見つかれば代入する
				for (int value = 1; value <= 9; value++) {
					if (canSetValue(value, puzzle.getRow(row))
							&& canSetValue(value, puzzle.getColumn(column))
							&& canSetValue(value, puzzle.getBlock(row, column))) {
						puzzle.setCell(row, column, value);
						if (sudokuSolve(puzzle)) // 数値を仮定し、再帰的に解く
							return true; // 仮定の結果、解が得られればtrueを返して終了
						puzzle.setCell(row, column, 0); // 矛盾が生じたら未確定値に戻す
					}
				}
				return false; // 候補値が存在しなくなったら矛盾。falseを返して終了。
			}
		}
		return puzzle.isSolved(); // 全てのセルをチェックした時点で終了
	}

	/**
	 * 対象の行、列またはブロック中に対象の値を置くことができるか
	 *
	 * @param value
	 *            置くことができるかチェックする対象となる値
	 * @param cells
	 *            通常は行、列またはブロックのセル配列
	 * @return 置くことができるならばtrue, 置けなければfalse。
	 */
	private static boolean canSetValue(int value, int[] cells) {
		for (int cell : cells) {
			if (cell == value) {
				return false; // すでに置かれていた場合はfalse
			}
		}
		return true; // その値がまだ使用されていなければtrue。
	}
}
