#
# = api_client.rb
#
# Lingr API client
#
# Modified by Gimite Ichikawa based on:
# $Revision: 24 $
# $Date: 2007-02-24 01:37:04 +0900 (土, 24  2 2007) $
#

$KCODE = 'u'
require 'jcode'
require 'kconv'
require 'net/http'
require 'cgi'

# Ruby client for the Lingr[http://www.lingr.com] API.  For more details and tutorials, see the 
# {Lingr API Reference}[http://wiki.lingr.com/dev/show/API+Reference] pages on the {Lingr Developer Wiki}[http://wiki.lingr.com].
#
# All methods return a hash with two keys:
# * :succeeded - <tt>true</tt> if the method succeeded, <tt>false</tt> otherwise
# * :response - a Hash version of the response document received from the server
#

module Lingr
  
  class ApiClient
    attr_accessor :api_key
    # 0 = quiet, 1 = some debug info, 2 = more debug info
    attr_accessor :verbosity
    attr_accessor :session
    attr_accessor :timeout

    def initialize(api_key, verbosity=0, hostname='www.lingr.com')
      @api_key = api_key
      @host = hostname
      @verbosity = verbosity
      @timeout = 120
    end

    # Create a new API session
    #
    def create_session(client_type='automaton')
      if @session
        p "already in a session", 1
        @error_info = nil
        return { :succeeded => false }
      end

      ret = do_api :post, 'session/create', { :api_key => @api_key, :client_type => client_type }, false
      @session = ret[:response]["session"] if ret[:succeeded]
      ret
    end
    
    # Verify a session id.  If no session id is passed, verifies the current session id for this ApiClient
    #
    def verify_session(session_id=nil)
      do_api :get, 'session/verify', { :session => session_id || @session }
    end

    # Destroy the current API session
    # 
    def destroy_session
      ret = do_api :post, 'session/destroy', { :session => @session }
      @session = nil
      ret
    end

    # Get a list of the currently hot rooms
    #
    def get_hot_rooms(count=nil)
      do_api :get, 'explore/get_hot_rooms', { :api_key => @api_key }.merge(count ? { :count => count} : {}), false
    end

    # Get a list of the newest rooms
    #
    def get_new_rooms(count=nil)
      do_api :get, 'explore/get_new_rooms', { :api_key => @api_key }.merge(count ? { :count => count} : {}), false
    end

    # Get a list of the currently hot tags
    #
    def get_hot_tags(count=nil)
      do_api :get, 'explore/get_hot_tags', { :api_key => @api_key }.merge(count ? { :count => count} : {}), false
    end

    # Get a list of all tags 
    #
    def get_all_tags(count=nil)
      do_api :get, 'explore/get_all_tags', { :api_key => @api_key }.merge(count ? { :count => count} : {}), false
    end

    # Search room name, description, and tags for keywords.  Keywords can be a String or an Array.
    #
    def search(keywords)
      do_api :get, 'explore/search', { :api_key => @api_key, :q => keywords.is_a?(Array) ? keywords.join(',') : keywords }, false
    end

    # Search room tags. Tagnames can be a String or an Array.
    #
    def search_tags(tagnames)
      do_api :get, 'explore/search_tags', { :api_key => @api_key, :q => tagnames.is_a?(Array) ? tagnames.join(',') : tagnames }, false
    end

    # Authenticate a user within the current API session
    #
    def login(email, password)
      do_api :post, 'auth/login', { :session => @session, :email => email, :password => password }
    end

    # Log out the currently-authenticate user in the session, if any
    def logout
      do_api :post, 'auth/logout', { :session => @session }
    end

    # Get information about the currently-authenticated user
    #
    def get_user_info
      do_api :get, 'user/get_info', { :session => @session }
    end

    # Start observing the currently-authenticated user
    #
    def start_observing_user
      do_api :post, 'user/start_observing', { :session => @session }
    end

    # Observe the currently-authenticated user, watching for profile changes
    #
    def observe_user(ticket, counter)
      do_api :get, 'user/observe', { :session => @session, :ticket => ticket, :counter => counter }
    end

    # Stop observing the currently-authenticated user
    #
    def stop_observing_user(ticket)
      do_api :post, 'user/stop_observing', { :session => @session, :ticket =>ticket }
    end

    # Get information about a chatroom, including room description, current occupants, recent messages, etc.
    # 
    def get_room_info(room_id, counter=nil, password=nil)
      params = { :api_key => @api_key, :id => room_id }
      params.merge!({ :counter => counter }) if counter
      params.merge!({ :password => password }) if password
      do_api :get, 'room/get_info', params, false
    end

    # Enter a chatroom
    #
    def enter_room(room_id, nickname=nil, password=nil, idempotent=false)
      params = { :session => @session, :id => room_id }
      params.merge!({ :nickname => nickname }) if nickname
      params.merge!({ :password => password }) if password
      params.merge!({ :idempotent => 1 }) if idempotent
      do_api :post, 'room/enter', params
    end

    # Poll for messages in a chatroom
    #
    def get_messages(ticket, counter)
      do_api :get, 'room/get_messages', { :session => @session, :ticket => ticket, :counter => counter }
    end

    # Observe a chatroom, waiting for events to occur in the room
    #
    def observe_room(ticket, counter)
      do_api :get, 'room/observe', { :session => @session, :ticket => ticket, :counter => counter }
    end

    # Set your nickname in a chatroom
    #
    def set_nickname(ticket, nickname)
      do_api :post, 'room/set_nickname', { :session => @session, :ticket => ticket, :nickname => nickname }
    end

    # Say something in a chatroom
    #
    def say(ticket, msg)
      do_api :post, 'room/say', { :session => @session, :ticket => ticket, :message => msg }
    end

    # Exit a chatroom
    #
    def exit_room(ticket)
      do_api :post, 'room/exit', { :session => @session, :ticket => ticket }
    end

    private

    def do_api(method, path, parameters, require_session=true)
      if require_session and !@session
        p "not in a session", 1
        return { :succeeded => false }
      end
      
      u_parameters= {}
      parameters.each(){ |k, v| u_parameters[k]= kcode_to_utf8(v.to_s()) }

      response = json_to_hash(self.send(method, url_for(path), u_parameters.merge({ :format => 'json' })))
      ret = success?(response)
      if ret
        p "#{path} succeeded", 1
      else
        p "#{path} failed : #{response and response['error'] ? response['error']['message'] : 'socket timeout'}", 0
      end

      { :succeeded => ret, :response => response }
    end

    def url_for(method)
      "http://#{@host}/#{@@PATH_BASE}#{method}"
    end

    def get(url, params)
      uri = URI.parse(url)
      path = uri.path
      q = params.inject("?") {|s, p| s << "#{p[0].to_s}=#{CGI.escape(p[1].to_s)}&"}.chop
      path << q if q.length > 0

      Net::HTTP.start(uri.host, uri.port) { |http| 
        http.read_timeout = @timeout
        req = Net::HTTP::Get.new(path)
        req.basic_auth(uri.user, uri.password) if uri.user
        begin
          parse_result http.request(req)
        rescue Exception
          p "exception on HTTP GET: #{$!}", 2
          nil
        end
      }
    end

    def post(url, params)
      begin
        parse_result Net::HTTP.post_form(URI.parse(url), params)
      rescue Exception
        p "exception on HTTP POST: #{$!}", 2
        nil
      end
    end

    def parse_result(result)
      return nil if !result || result.code != '200' || (!result['Content-Type'] || result['Content-Type'].index('text/javascript') != 0)
      result.body
    end

    def success?(response)
      return false if !response
      p response.inspect, 2
      response["status"] and response["status"] == 'ok'
    end

    def p(msg, level=0)
      puts msg if level <= @verbosity
    end

    def json_to_hash(json)
      return nil if !json
      null = nil; 
      eval(utf8_to_kcode(json).gsub(/(["'])\s*:\s*(['"0-9tfn\[{])/n){"#{$1}=>#{$2}"}.gsub(/\#\{/n, '\#{'))
    end
    
    def utf8_to_kcode(str)
      case $KCODE
        when "SJIS"; return str.kconv(Kconv::SJIS, Kconv::UTF8)
        when "EUC"; return str.kconv(Kconv::EUC, Kconv::UTF8)
        else; return str
      end
    end
    
    def kcode_to_utf8(str)
      case $KCODE
        when "SJIS"; return str.kconv(Kconv::UTF8, Kconv::SJIS)
        when "EUC"; return str.kconv(Kconv::UTF8, Kconv::EUC)
        else; return str
      end
    end

    @@PATH_BASE = 'api/'
  end
  
end
