﻿/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
  <file> Program.cs </file>
  <brief>
    Tokenize and dump the html5 file.
  </brief>
  <author>
    Stanley Hong <link2next@gmail.com>, Dec. 2013.
  </author>
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using uls.polaris;
using uls.collection;

namespace uls
{
    namespace tests
    {
        namespace Html5Toks
        {
            public delegate int uls_getopts_proc(int opt, String optarg);

            public class Util
            {
                // <brief>
                // Get the basename of file-path!
                // Returns 'basename.c when the 'filepath' is '\a\b\basename.c'.
                // </brief>
                // <parm name="filepath">input path string</parm>
                public static String uls_filename(String filepath)
                {
                    String fname;
                    int i;

                    if ((i = filepath.LastIndexOf('\\')) < 0)
                    {
                        fname = filepath;
                    }
                    else
                    {
                        fname = filepath.Substring(i + 1);
                    }

                    return fname;
                }

                // <brief>
                // This appends the line-feed character '\n' at the message formatted by 'fmtstr'.
                // </brief>
                public static void err_log(String fmtstr, params object[] objs)
                {
                    String dotstr;
                    dotstr = String.Format(fmtstr, objs);
                    Console.WriteLine(dotstr);
                }

                // <brief>
                // This calls err_log() above and terminates the program to abnormal state.
                // </brief>
                public static void err_panic(String fmtstr, params object[] objs)
                {
                    err_log(fmtstr, objs);
                    Environment.Exit(1);
                }
            }

            class Program
            {
                static String PACKAGE_NAME = "Html5Toks";

                static void usage()
                {
                    Util.err_log("usage({0}): dumping the tokens", PACKAGE_NAME);
                    Util.err_log("\t{0} <filepath>", PACKAGE_NAME);
                }

                static void Version()
                {
                }

                static int html5toks_options(int opt, String optarg)
                {
                    int stat = 0;

                    switch (opt)
                    {
                        case 'v':
                            break;
                        case 'h':
                            usage();
                            stat = -1;
                            break;
                        case 'V':
                            //			Version();
                            stat = -1;
                            break;
                        default:
                            Util.err_log("undefined option -%c", opt);
                            stat = -1;
                            break;
                    }

                    return stat;
                }

                static void dumpTokens(Html5Lex html5lex)
                {
                    for ( ; ; )
                    {
                        if (html5lex.getTok() == Html5Lex.EOI) break;

                        int t = html5lex.TokNum;
                        String tstr = html5lex.TokStr;

                        if (t == Html5Lex.TAGBEGIN)
                        {
                            html5lex.dumpTok("\t", "TAG", tstr, "\n");
                        }
                        else if (t == Html5Lex.TAGEND)
                        {
                            html5lex.dumpTok("\t", "/TAG", tstr, "\n");
                        }
                        else if (t == Html5Lex.TEXT)
                        {
                            html5lex.dumpTok("\t", "TEXT", tstr, "\n");
                        }
                        else
                        {
                            html5lex.dumpTok("\t", "\n");
                        }
                    }
                }

                static void Main(String[] args)
                {
                    int i0;

                    String config_name = "html5.ulc";
                    Html5Lex html5lex;
                    String input_file;

                    using (UlsContext run_context = new UlsContext("vhV"))
                    {

                        UlsContext.DelegateOptProc opt_proc = new UlsContext.DelegateOptProc(html5toks_options);
                        if ((i0 = run_context.parseOpts(args, opt_proc)) != 0)
                        {
                            usage();
                            return;
                        }

                        if (i0 < args.Length)
                        {
                            input_file = args[i0];
                        }
                        else
                        {
                            usage();
                            return;
                        }

                        try
                        {
                            html5lex = new Html5Lex(config_name);

                            if (html5lex.setFile(input_file) < 0)
                            {
                                Util.err_log("%s: file open error", input_file);
                            }
                            else
                            {
                                dumpTokens(html5lex);
                            }
                        }
                        catch (Exception e)
                        {
                            Console.WriteLine("An exception caught: {0}", e);
                        }
                    }
                }
            }
        }
    }
}
