/*
 *                 Sun Public License Notice
 * 
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 * 
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2001 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.examples.modules.minicomposer;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.beans.PropertyChangeEvent;
import java.io.IOException;
import java.util.*;

import org.openide.ErrorManager;
import org.openide.TopManager;
import org.openide.actions.*;
import org.openide.nodes.*;
import org.openide.util.actions.SystemAction;
import org.openide.util.*;
import org.openide.util.datatransfer.ExTransferable;
import org.openide.util.datatransfer.PasteType;

/** Node representing one note in a score.
 */
public class NoteNode extends AbstractNode implements NodeListener {
    private final Score.Note note;
    private boolean addedCookieListener;
    public NoteNode(Score.Note note) {
        super(Children.LEAF);
        this.note = note;
        super.setName(Score.TONES_SHORT[note.getTone()]);
        setShortDescription(NbBundle.getMessage(NoteNode.class, "HINT_note_node",
                            Score.TONES_LONG[note.getTone()],
                            Score.OCTAVES_LONG[note.getOctave()],
                            Score.DURATIONS_LONG[note.getDuration()]));
        setIconBase("/org/netbeans/examples/modules/minicomposer/NoteIcon");
    }
    protected SystemAction[] createActions() {
        return new SystemAction[] {
            SystemAction.get(MoveUpAction.class),
            SystemAction.get(MoveDownAction.class),
            null,
            SystemAction.get(CutAction.class),
            SystemAction.get(CopyAction.class),
            SystemAction.get(PasteAction.class),
            null,
            SystemAction.get(DeleteAction.class),
            SystemAction.get(RenameAction.class),
            null,
            SystemAction.get(PropertiesAction.class),
            SystemAction.get(ToolsAction.class),
        };
    }
    public Node.Cookie getCookie(Class c) {
        if (c == Index.class || c == Index.Support.class) {
            return null;
        }
        Node parent = getParentNode();
        if (parent == null) return null;
        if (!addedCookieListener) {
            addedCookieListener = true;
            parent.addNodeListener(WeakListener.node(this, parent));
        }
        Node.Cookie cookie = parent.getCookie(c);
        return cookie;
    }
    public void propertyChange(PropertyChangeEvent evt) {
        if (Node.PROP_COOKIE.equals(evt.getPropertyName())) {
            fireCookieChange();
        }
    }
    public void childrenRemoved(NodeMemberEvent ev) {
        // ignore
    }
    public void childrenReordered(NodeReorderEvent ev) {
        // ignore
    }
    public void childrenAdded(NodeMemberEvent ev) {
        // ignore
    }
    public void nodeDestroyed(NodeEvent ev) {
        // ignore
    }
    public HelpCtx getHelpCtx() {
        return new HelpCtx("org.netbeans.examples.modules.minicomposer.explorer");
    }
    public boolean canRename() {
        return true;
    }
    public void setName(String nue) throws IllegalArgumentException {
        if (nue.equals(getName())) return;
        int v;
        for (v = 0; v < Score.TONES_SHORT.length; v++) {
            if (nue.equals(Score.TONES_SHORT[v])) break;
        }
        if (v == Score.TONES_SHORT.length) {
            IllegalArgumentException iae = new IllegalArgumentException("bad tone: " + nue);
            TopManager.getDefault().getErrorManager().annotate(iae, ErrorManager.WARNING, null,
                NbBundle.getMessage(NoteNode.class, "EXC_bad_tone", nue), null, null);
            throw iae;
        }
        ScoreCookie cookie = (ScoreCookie)getCookie(ScoreCookie.class);
        try {
            Score s = cookie.getScore();
            int size = s.getSize();
            List notes = new ArrayList(Math.max(size, 1));
            for (int i = 0; i < size; i++) {
                Score.Note n = s.getNote(i);
                if (n == note) {
                    n = new Score.Note(v, n.getOctave(), n.getDuration());
                }
                notes.add(n);
            }
            cookie.setScore(new Score(notes));
        } catch (IOException ioe) {
            TopManager.getDefault().getErrorManager().notify(ioe);
        }
    }
    public boolean canDestroy() {
        return true;
    }
    public void destroy() throws IOException {
        ScoreCookie cookie = (ScoreCookie)getCookie(ScoreCookie.class);
        if (cookie == null) throw new IOException();
        Score s = cookie.getScore();
        int size = s.getSize();
        List notes = new ArrayList(Math.max(size - 1, 1));
        for (int i = 0; i < size; i++) {
            Score.Note n = s.getNote(i);
            if (n != note) {
                notes.add(n);
            }
        }
        if (notes.size() != size - 1) throw new IOException("did not find note " + note + " among " + notes);
        cookie.setScore(new Score(notes));
    }
    public boolean canCopy() {
        return true;
    }
    public boolean canCut() {
        return true;
    }
    public Transferable clipboardCopy() throws IOException {
        Transferable deflt = super.clipboardCopy();
        ExTransferable enriched = ExTransferable.create(deflt);
        enriched.put(new ExTransferable.Single(NoteTransfer.SCORE_FLAVOR) {
            protected Object getData() {
                return new Score(Collections.singletonList(note));
            }
        });
        return enriched;
    }
    public Transferable clipboardCut() throws IOException {
        destroy();
        return clipboardCopy();
    }
    protected void createPasteTypes(Transferable t, List s) {
        super.createPasteTypes(t, s);
        PasteType p = NoteTransfer.createNotePasteType(t, (ScoreCookie)getCookie(ScoreCookie.class), note);
        if (p != null) {
            s.add(p);
        }
    }
    protected Sheet createSheet() {
        Sheet s = super.createSheet();
        Sheet.Set ss = Sheet.createPropertiesSet();
        ScoreCookie cookie = (ScoreCookie)getCookie(ScoreCookie.class);
        ss.put(new NoteProperty(cookie, note, 0));
        ss.put(new NoteProperty(cookie, note, 1));
        ss.put(new NoteProperty(cookie, note, 2));
        s.put(ss);
        return s;
    }
}
