/*
 *                 Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 *
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2001 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.examples.modules.minicomposer;

import java.io.*;
import java.lang.ref.*;
import java.util.*;
import javax.swing.SwingUtilities;
import javax.swing.event.*;
import javax.swing.text.*;

import org.openide.ErrorManager;
import org.openide.TopManager;
import org.openide.cookies.EditorCookie;
import org.openide.loaders.DataObject;
import org.openide.text.*;
import org.openide.util.*;

public class ScoreSupport implements ScoreCookie, Runnable, DocumentListener, ChangeListener {
    private final ErrorManager err;
    private final DataObject obj;
    private final EditorCookie edit;
    private Task prepareTask = null;
    private final Set listeners = new HashSet();
    private Score score = null;
    private IOException parseException = null;
    private boolean addedEditorListener = false;
    private Reference lastUsedDocument = null; // Reference<Document>
    public ScoreSupport(DataObject obj, EditorCookie edit) {
        this.obj = obj;
        err = TopManager.getDefault().getErrorManager().
            getInstance("org.netbeans.examples.modules.minicomposer.ScoreSupport." + // NOI18N
                        obj.getPrimaryFile().getPackageName('.'));
        this.edit = edit;
    }
    public void stateChanged(ChangeEvent ev) {
        err.log("Editor state changed");
        invalidate();
    }
    public synchronized void addChangeListener(ChangeListener l) {
        listeners.add(l);
    }
    public synchronized void removeChangeListener(ChangeListener l) {
        listeners.remove(l);
    }
    protected synchronized void fireChange() {
        final ChangeListener[] ls = (ChangeListener[])listeners.toArray(new ChangeListener[listeners.size()]);
        if (ls.length == 0) return;
        final ChangeEvent ev = new ChangeEvent(this);
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                err.log("firing change");
                for (int i = 0; i < ls.length; i++) {
                    ls[i].stateChanged(ev);
                }
            }
        });
    }
    public synchronized Task prepare() {
        if (prepareTask == null) {
            err.log("preparing to parse");
            prepareTask = RequestProcessor.postRequest(this);
        }
        return prepareTask;
    }
    public void run() {
        err.log("run");
        if (!obj.isValid()) {
            err.log("object invalidated");
            setScoreAndParseException(null, new FileNotFoundException(obj.getPrimaryFile().getPackageNameExt('/', '.')));
            return;
        }
        edit.prepareDocument().waitFinished();
        final Document doc = edit.getDocument();
        if (doc == null) {
            // Should not happen:
            err.log(ErrorManager.WARNING, "WARNING: Doc was null!");
            return;
        }
        if (!addedEditorListener) {
            err.log("adding editor listener");
            addedEditorListener = true;
            if (edit instanceof CloneableEditorSupport) {
                ((CloneableEditorSupport)edit).addChangeListener(WeakListener.change(this, edit));
            } else {
                err.log(ErrorManager.WARNING, "Warning: no CloneableEditorSupport found");
            }
        }
        doc.render(new Runnable() {
            public void run() {
                try {
                    setScoreAndParseException(parse(doc), null);
                } catch (IOException ioe) {
                    setScoreAndParseException(score, ioe);
                } catch (BadLocationException ble) {
                    IOException ioe = new IOException(ble.toString());
                    err.annotate(ioe, ble);
                    setScoreAndParseException(score, ioe);
                }
            }
        });
        Document lastDoc = null;
        if (lastUsedDocument != null) {
            lastDoc = (Document)lastUsedDocument.get();
        }
        if (lastDoc != doc) {
            if (lastDoc != null) {
                err.log("removing listener from old document");
                lastDoc.removeDocumentListener(this);
            }
            err.log("adding fresh document listener");
            doc.addDocumentListener(this);
            lastUsedDocument = new WeakReference(doc);
        }
    }
    private synchronized void setScoreAndParseException(Score s, IOException e) {
        if (err.isLoggable(ErrorManager.INFORMATIONAL)) {
            err.log("parsed; exception=" + e + "; score size=" + s.getSize());
        }
        score = s;
        parseException = e;
        fireChange();
    }
    public boolean isValid() {
        return parseException == null;
    }
    public void changedUpdate(DocumentEvent ev) {
    }
    public void insertUpdate(DocumentEvent ev) {
        invalidate();
    }
    public void removeUpdate(DocumentEvent ev) {
        invalidate();
    }
    protected synchronized void invalidate() {
        err.log("invalidated");
        if (prepareTask != null) {
            prepareTask = null;
            fireChange();
        }
    }
    public synchronized void setScore(final Score s) throws IOException {
        final Score oldScore = score;
        if (s.equals(oldScore)) {
            return;
        }
        err.log("setScore");
        prepareTask = Task.EMPTY;
        score = s;
        parseException = null;
        final StyledDocument doc = edit.openDocument();
        final BadLocationException[] e = new BadLocationException[] {null};
        try {
            NbDocument.runAtomic(doc, new Runnable() {
                public void run() {
                    doc.removeDocumentListener(ScoreSupport.this);
                    err.log("removed doc listener");
                    try {
                        generate(s, oldScore, doc);
                    } catch (BadLocationException ble) {
                        e[0] = ble;
                    } finally {
                        err.log("readded doc listener");
                        doc.addDocumentListener(ScoreSupport.this);
                    }
                }
            });
            if (e[0] != null) throw e[0];
        } catch (BadLocationException ble) {
            IOException ioe = new IOException(ble.toString());
            err.annotate(ioe, ble);
            throw ioe;
        }
        fireChange();
    }
    public Score getScore() throws IOException {
        prepare().waitFinished();
        synchronized (this) {
            if (parseException != null && !obj.isModified()) {
                throw parseException;
            } else if (score != null) {
                return score;
            } else if (parseException != null) {
                throw parseException;
            } else {
                // Should not happen:
                throw new IOException("parse did not finish as expected");
            }
        }
    }
    protected Score parse(Document doc) throws IOException, BadLocationException {
        String text = doc.getText(0, doc.getLength());
        return Score.parse(new StringReader(text));
    }
    protected void generate(Score s, Score oldScore, Document doc) throws BadLocationException {
        CharArrayWriter wr = new CharArrayWriter();
        try {
            Score.generate(s, wr);
        } catch (IOException ioe) {
            // Should not happen.
            err.notify(ioe);
            return;
        }
        doc.remove(0, doc.getLength());
        doc.insertString(0, wr.toString(), null);
    }
}
