/**
 * @file mml.c
 * @author Shinichiro Nakamura
 * @copyright
 * ===============================================================
 * A tiny MML parser (Version 0.2.0)
 * ===============================================================
 * Copyright (c) 2014-2015 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntlibc.h"
#include "mml.h"

#define MML_INIT_DONE_CODE	(0x1234)
#define MML_SETUP_DONE_CODE (0x5678)

typedef enum {
	MML_CHAR_TYPE_NOTE,
	MML_CHAR_TYPE_SHARP,
	MML_CHAR_TYPE_FLAT,
	MML_CHAR_TYPE_REST,
	MML_CHAR_TYPE_OCTAVE,
	MML_CHAR_TYPE_LENGTH,
	MML_CHAR_TYPE_VOLUME,
	MML_CHAR_TYPE_TEMPO,
	MML_CHAR_TYPE_OCTAVE_UP,
	MML_CHAR_TYPE_OCTAVE_DOWN,
	MML_CHAR_TYPE_NUMBER,
	MML_CHAR_TYPE_TIE1,
	MML_CHAR_TYPE_UNKNOWN,
} MML_CHAR_TYPE;

static MML_RESULT skip_whitespace(MML *handle)
{
	while (((*handle->tp) == ' ') || ((*handle->tp) == '\r') || ((*handle->tp) == '\n') || ((*handle->tp) == '\t')) {
		handle->tp++;
	}
	return MML_RESULT_OK;
}

static MML_RESULT get_note_number(MML *handle, char *text, int *number)
{
	static const char *note_distance = "C D EF G A B";
	char *p;

	p = ntlibc_strchr(note_distance, text[0]);
	if (p == 0) {
		return MML_RESULT_ILLEGAL_NOTE_TEXT;
	}
	*number = p - note_distance;

	p = &text[1];
	while (*p) {
		if (*p == '+') {
			*number = *number + 1;
		}
		if (*p == '-') {
			*number = *number - 1;
		}
		p++;
	}

	*number += (handle->option.octave * 12);

	return MML_RESULT_OK;
}

static int convert_note_length_to_ticks(MML *handle, int note_length)
{
	int bticks = handle->option.bticks;

	switch (note_length) {
		case 643:
			/*
			 * 1/64 note triplet.
			 */
			return (bticks / 8) / 3;
		case 64:
			return (bticks / 16);
		case 323:
			/*
			 * 1/32 note triplet.
			 */
			return (bticks / 4) / 3;
		case 32:
			return (bticks / 8);
		case 163:
			/*
			 * 1/16 note triplet.
			 */
			return (bticks / 2) / 3;
		case 16:
			return (bticks / 4);
		case 83:
			/*
			 * 1/8 note triplet.
			 */
			return (bticks * 1) / 3;
		case 8:
			return (bticks / 2);
		case 43:
			/*
			 * 1/4 note triplet.
			 */
			return (bticks * 2) / 3;
		case 4:
			return (bticks * 1);
		case 23:
			/*
			 * 1/2 note triplet.
			 */
			return (bticks * 4) / 3;
		case 2:
			return (bticks * 2);
		case 1:
			return (bticks * 4);
	}

	return 0;
}

static MML_RESULT get_note_ticks(MML *handle, char *text, int *ticks)
{
	int note_length = ntlibc_atoi(text);
	char *p;
	*ticks = convert_note_length_to_ticks(handle, note_length);
	p = &text[1];
	while (*p) {
		if (*p == '.') {
			note_length *= 2;
			*ticks += convert_note_length_to_ticks(handle, note_length);
		} if (*p == '^' || *p == '&') {
			if(*p == '&') p++;
			while(*p != 0) {
				p++;
				char pp[8];
				int x;
				for(x = 0;*p <= '9' && *p >= '0';*p++,x++)pp[x] = *p;
				pp[x] = 0;
				note_length = ntlibc_atoi(pp);
				*ticks += convert_note_length_to_ticks(handle, note_length);
			}
		} else {
			return MML_RESULT_ILLEGAL_LENGTH_VALUE;
		}
		p++;
	}
	return MML_RESULT_OK;
}

static MML_RESULT get_note_ticks_default(MML *handle, int *ticks)
{
	*ticks = convert_note_length_to_ticks(handle, handle->option.length);
	return MML_RESULT_OK;
}

static MML_CHAR_TYPE get_char_type(char c)
{
	switch (ntlibc_toupper(c)) {
		case 'C':
		case 'D':
		case 'E':
		case 'F':
		case 'G':
		case 'A':
		case 'B':
			return MML_CHAR_TYPE_NOTE;
		case '+':
			return MML_CHAR_TYPE_SHARP;
		case '-':
			return MML_CHAR_TYPE_FLAT;
		case 'R':
			return MML_CHAR_TYPE_REST;
		case 'O':
			return MML_CHAR_TYPE_OCTAVE;
		case 'L':
			return MML_CHAR_TYPE_LENGTH;
		case 'V':
			return MML_CHAR_TYPE_VOLUME;
		case 'T':
			return MML_CHAR_TYPE_TEMPO;
		case '>':
			return MML_CHAR_TYPE_OCTAVE_UP;
		case '<':
			return MML_CHAR_TYPE_OCTAVE_DOWN;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
		case '.':
			return MML_CHAR_TYPE_NUMBER;
		case '^':
		case '&':
		return MML_CHAR_TYPE_TIE1;
	}
	return MML_CHAR_TYPE_UNKNOWN;
}

static MML_RESULT get_token(MML *handle, char *buf, int siz)
{
	MML_CHAR_TYPE ct_head, ct_next;
	int i;
	int cnt = 0;

	for (i = 0; i < siz; i++) {
		buf[i] = '\0';
	}

	skip_whitespace(handle);
	if (*handle->tp == '\0') {
		return MML_RESULT_EOT;
	}

	ct_head = get_char_type(*handle->tp);
	while ((ct_next = get_char_type(*handle->tp)) == ct_head || ct_next == MML_CHAR_TYPE_TIE1) {
		*buf++ = ntlibc_toupper(*handle->tp++);
		if ((MML_CHAR_TYPE_NOTE == ct_head) && (MML_CHAR_TYPE_NOTE == ct_next)) {
			break;
		}
		cnt++;
		if (siz <= (cnt + 1)) {
			*buf++ = 0;
			return MML_RESULT_ILLEGAL_TOKEN_LENGTH;
		}
	}

	while (1) {
		MML_CHAR_TYPE ct_work;
		ct_work = get_char_type(*handle->tp);
		if ((ct_work != MML_CHAR_TYPE_SHARP) && (ct_work != MML_CHAR_TYPE_FLAT)) {
			break;
		}
		*buf++ = ntlibc_toupper(*handle->tp++);
	}

	*buf++ = 0;

	return MML_RESULT_OK;
}

static MML_RESULT is_valid_input(char *text)
{
	int i;
	int len = ntlibc_strlen(text);
	for (i = 0; i < len; i++) {
		char c = ntlibc_toupper(text[i]);
		if (ntlibc_strchr("CDEFGAB+-0123456789.LVTOR<>&^ \r\n\t", c) == 0) {
			return MML_RESULT_ILLEGAL_INPUT_TEXT;
		}
	}
	return MML_RESULT_OK;
}

MML_RESULT mml_init(MML *handle, MML_CALLBACK callback, void *extobj)
{
	handle->callback = callback;
	handle->extobj = extobj;

	handle->text_head = 0;
	handle->tp = 0;

	handle->init_done = MML_INIT_DONE_CODE;
	handle->setup_done = 0;

	return MML_RESULT_OK;
}

MML_RESULT mml_setup(MML *handle, MML_OPTION *option, char *text)
{
	MML_RESULT mr;

	if (handle->init_done != MML_INIT_DONE_CODE) {
		return MML_RESULT_ILLEGAL_INIT_FOUND;
	}

	if (option != 0) {
		handle->option.octave = option->octave;
		handle->option.length = option->length;
		handle->option.bticks = option->bticks;
	} else {
		handle->option.octave = MML_OPTION_DEFAULT_OCTAVE;
		handle->option.length = MML_OPTION_DEFAULT_LENGTH;
		handle->option.bticks = MML_OPTION_DEFAULT_BTICKS;
	}

	handle->text_head = text;
	handle->tp = handle->text_head;

	mr = is_valid_input(handle->text_head);
	if (mr == MML_RESULT_OK) {
		handle->setup_done = MML_SETUP_DONE_CODE;
	} else {
		handle->setup_done = 0;
	}

	return mr;
}

MML_RESULT mml_fetch(MML *handle)
{
	MML_RESULT mr;
	char buf1[8];
	char buf2[8];

	if (handle->init_done != MML_INIT_DONE_CODE) {
		return MML_RESULT_ILLEGAL_INIT_FOUND;
	}

	if (handle->setup_done != MML_SETUP_DONE_CODE) {
		return MML_RESULT_ILLEGAL_SETUP_FOUND;
	}

	mr = get_token(handle, buf1, sizeof(buf2));
	if (mr != MML_RESULT_OK) {
		return mr;
	}

	switch (get_char_type(buf1[0])) {
		case MML_CHAR_TYPE_NOTE:
			{
				char *p = handle->tp;
				mr = get_token(handle, buf2, sizeof(buf2));
				if (get_char_type(buf2[0]) == MML_CHAR_TYPE_NUMBER) {
					MML_INFO info;
					info.type = MML_TYPE_NOTE;
					get_note_number(handle, buf1, &(info.args.note.number));
					get_note_ticks(handle, buf2, &(info.args.note.ticks));
					handle->callback(&info, handle->extobj);
				} else {
					MML_INFO info;
					info.type = MML_TYPE_NOTE;
					get_note_number(handle, buf1, &(info.args.note.number));
					get_note_ticks_default(handle, &(info.args.note.ticks));
					handle->callback(&info, handle->extobj);
					handle->tp = p;
				}
			}
			break;
		case MML_CHAR_TYPE_REST:
			{
				char *p = handle->tp;
				mr = get_token(handle, buf2, sizeof(buf2));
				if (get_char_type(buf2[0]) == MML_CHAR_TYPE_NUMBER) {
					MML_INFO info;
					info.type = MML_TYPE_REST;
					get_note_ticks(handle, buf2, &(info.args.rest.ticks));
					handle->callback(&info, handle->extobj);
				} else {
					MML_INFO info;
					info.type = MML_TYPE_REST;
					get_note_ticks_default(handle, &(info.args.rest.ticks));
					handle->callback(&info, handle->extobj);
					handle->tp = p;
				}
			}
			break;
		case MML_CHAR_TYPE_TEMPO:
			{
				mr = get_token(handle, buf2, sizeof(buf2));
				if (get_char_type(buf2[0]) == MML_CHAR_TYPE_NUMBER) {
					MML_INFO info;
					info.type = MML_TYPE_TEMPO;
					info.args.tempo.value = ntlibc_atoi(buf2);
					handle->callback(&info, handle->extobj);
				} else {
					return MML_RESULT_ILLEGAL_TEMPO_VALUE;
				}
			}
			break;
		case MML_CHAR_TYPE_VOLUME:
			{
				mr = get_token(handle, buf2, sizeof(buf2));
				if (get_char_type(buf2[0]) == MML_CHAR_TYPE_NUMBER) {
					MML_INFO info;
					info.type = MML_TYPE_VOLUME;
					info.args.volume.value = ntlibc_atoi(buf2);
					handle->callback(&info, handle->extobj);
				} else {
					return MML_RESULT_ILLEGAL_VOLUME_VALUE;
				}
			}
			break;
		case MML_CHAR_TYPE_LENGTH:
			{
				mr = get_token(handle, buf2, sizeof(buf2));
				if (get_char_type(buf2[0]) == MML_CHAR_TYPE_NUMBER) {
					MML_INFO info;
					info.type = MML_TYPE_LENGTH;
					info.args.length.value = ntlibc_atoi(buf2);
					handle->option.length = info.args.length.value;
					handle->callback(&info, handle->extobj);
				} else {
					return MML_RESULT_ILLEGAL_LENGTH_VALUE;
				}
			}
			break;
		case MML_CHAR_TYPE_OCTAVE:
			{
				mr = get_token(handle, buf2, sizeof(buf2));
				if (get_char_type(buf2[0]) == MML_CHAR_TYPE_NUMBER) {
					MML_INFO info;
					info.type = MML_TYPE_OCTAVE;
					info.args.octave.value = ntlibc_atoi(buf2);
					handle->option.octave = info.args.octave.value;
					handle->callback(&info, handle->extobj);
				} else {
					return MML_RESULT_ILLEGAL_OCTAVE_VALUE;
				}
			}
			break;
		case MML_CHAR_TYPE_OCTAVE_UP:
			{
				MML_INFO info;
				char *p = buf1;
				while (*p++) {
					if (handle->option.octave < MML_OCTAVE_MAX) {
						handle->option.octave++;
					}
				}
				info.type = MML_TYPE_OCTAVE_UP;
				info.args.octave_up.value = handle->option.octave;
				handle->callback(&info, handle->extobj);
			}
			break;
		case MML_CHAR_TYPE_OCTAVE_DOWN:
			{
				MML_INFO info;
				char *p = buf1;
				while (*p++) {
					if (MML_OCTAVE_MIN < handle->option.octave) {
						handle->option.octave--;
					}
				}
				info.type = MML_TYPE_OCTAVE_DOWN;
				info.args.octave_down.value = handle->option.octave;
				handle->callback(&info, handle->extobj);
			}
			break;
		case MML_CHAR_TYPE_NUMBER:
		case MML_CHAR_TYPE_SHARP:
		case MML_CHAR_TYPE_FLAT:
		case MML_CHAR_TYPE_UNKNOWN:
			{
				/*
				 * This is a illegal case of the parsing because these characters are needed a command.
				 */
				return MML_RESULT_ILLEGAL_SEQUENCE_FOUND;
			}
			break;
	}

	return mr;
}

