package ts.query;

import ts.tester.UnitTest;
import ts.util.*;
import ts.util.resource.*;
import java.io.*;
import java.util.*;

public class QueryTransactionManager_2Test extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryTransactionManager_2Test.class, args);
  }

  static StringWriter SW = null;
  static PrintWriter PW = null;

  static String OUTPUT_LOG() {
    return SW.toString();
  }
  static void CLEAR_LOG() {
    if (PW != null) PW.close();
    SW = new StringWriter();
    PW = new PrintWriter(SW);
  }

  static boolean NOT_THROW_AT_ROLLBACK = false;
  static ReasonedException THROW0 = null;
  static ReasonedRuntimeException THROW1 = null;
  static RuntimeException THROW2 = null;

  static enum Error { ERR }

  static class MyConnection implements IQueryConnection {
    private final QueryConnectionConfig config;
    private IQueryHistory history = new QueryHistory();
    private long limitTimeMillis = 0L;
    private boolean isOpened = false;
    public MyConnection(QueryConnectionConfig cfg) { this.config = cfg; }
    public MyConnection(QueryConnectionConfig cfg, IQueryTransaction tran) {
      this.config = cfg;
      this.limitTimeMillis = tran.getLimitTimeMillis();
    }
    @Override
    public String getConnectionId() { return this.config.getConnectionId(); }
    @Override
    public long getLimitTimeMillis() { return this.limitTimeMillis; }
    public IQueryHistory getQueryHistory() { return this.history; }
    @Override
    public boolean isOpened() { return this.isOpened; }
    @Override
    public boolean isClosed() { return ! this.isOpened; }
    @Override
    public void open() throws ReasonedException {
      this.isOpened = true;
      PW.print("[" + getClass().getSimpleName() + " is opened]");
      if (THROW0 != null) throw THROW0;
      if (THROW1 != null) throw THROW1;
      if (THROW2 != null) throw THROW2;
    }
    @Override
    public void commit() throws ReasonedException {
      PW.print("[" + getClass().getSimpleName() + " is committed]");
      if (THROW0 != null) throw THROW0;
      if (THROW1 != null) throw THROW1;
      if (THROW2 != null) throw THROW2;
    }
    @Override
    public void rollback() throws ReasonedException {
      PW.print("[" + getClass().getSimpleName() + " is rollbacked]");
      if (! NOT_THROW_AT_ROLLBACK) {
        if (THROW0 != null) throw THROW0;
        if (THROW1 != null) throw THROW1;
        if (THROW2 != null) throw THROW2;
      }
    }
    @Override
    public void close() throws ReasonedException {
      this.isOpened = false;
      PW.print("[" + getClass().getSimpleName() + " is closed]");
      if (THROW0 != null) throw THROW0;
      if (THROW1 != null) throw THROW1;
      if (THROW2 != null) throw THROW2;
    }
  }
  static class MyConnection0 extends MyConnection {
    public MyConnection0(QueryConnectionConfig cfg) { super(cfg); }
    public MyConnection0(QueryConnectionConfig cfg, IQueryTransaction tran)
    { super(cfg, tran); }
  }
  static class MyConnection1 extends MyConnection {
    public MyConnection1(QueryConnectionConfig cfg) { super(cfg); }
    public MyConnection1(QueryConnectionConfig cfg, IQueryTransaction tran)
    { super(cfg, tran); }
  }
  static class MyConnection2 extends MyConnection {
    public MyConnection2(QueryConnectionConfig cfg) { super(cfg); }
    public MyConnection2(QueryConnectionConfig cfg, IQueryTransaction tran)
    { super(cfg, tran); }
  }

  static class MyExecution implements IQueryExecution {
    private QueryExecutionConfig config;
    public MyExecution(QueryExecutionConfig cfg, IQueryConnection conn)
    { this.config = cfg; }
    @Override
    public String getExecutionId() { return this.config.getExecutionId(); }
    @Override
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException { return null; }
    @Override
    public void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException {}
  }
  static class MyExecution0 extends MyExecution {
    public MyExecution0(QueryExecutionConfig cfg, IQueryConnection conn)
    { super(cfg, conn); }
  }
  static class MyExecution1 extends MyExecution {
    public MyExecution1(QueryExecutionConfig cfg, IQueryConnection conn)
    { super(cfg, conn); }
  }
  static class MyExecution2 extends MyExecution {
    public MyExecution2(QueryExecutionConfig cfg, IQueryConnection conn)
    { super(cfg, conn); }
  }

  @Override
  protected void preInvocation(String method)
  {
    CLEAR_LOG();
    THROW0 = null;
    THROW1 = null;
    THROW2 = null;
    NOT_THROW_AT_ROLLBACK = false;
  }

  public void constructor()
  {
    MSG("デフォルト・コンストラクタ。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), QueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);
    NOTNULL(tran.getQueryHistory());

    tran.end();
  }

  public void setTimeoutMillis_millis_Commit()
  {
    MSG("タイムアウト時間を設定するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), QueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);
    NOTNULL(tran.getQueryHistory());

    tran.setTimeoutMillis(123L);

    EQUAL(tran.getState(), QueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 123L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    tran.end();

    EQUAL(tran.getState(), QueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());
  }

  public void setTimeoutMillis_millis_Rollback()
  {
    MSG("タイムアウト時間を設定するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), QueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);
    NOTNULL(tran.getQueryHistory());

    tran.setTimeoutMillis(123L);

    EQUAL(tran.getState(), QueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 123L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    tran.rollback();

    EQUAL(tran.getState(), QueryTransaction.State.Rollbacked);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Rollbacked);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    tran.end();

    EQUAL(tran.getState(), QueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());

    try {
      tran.setTimeoutMillis(999L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    EQUAL(tran.getState(), QueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 123L);
    POSITIVE(tran.getBeginTimeMillis());
    EQUAL(tran.getLimitTimeMillis(), tran.getBeginTimeMillis() + 123L);
    NOTNULL(tran.getQueryHistory());
  }

  public void getQueryConnection_connId()
  {
    MSG("IQueryConnectionオブジェクトを取得するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(OUTPUT_LOG(), "");
    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    tran.end();
    EQUAL(OUTPUT_LOG(), "");
    CLEAR_LOG();

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryConnection conn0 = null, conn1 = null, conn2 = null;

    tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    try {
      conn0 = tran.getQueryConnection(CID0);
      EQUAL(OUTPUT_LOG(), "[MyConnection0 is opened]");
      TRUE (conn0.isOpened());
      FALSE(conn0.isClosed());

      EQUAL(tran.getQueryConnection(CID0), conn0);
      EQUAL(OUTPUT_LOG(), "[MyConnection0 is opened]");
      TRUE (conn0.isOpened());
      FALSE(conn0.isClosed());

      EQUAL(tran.getQueryConnection(CID0), conn0);
      EQUAL(OUTPUT_LOG(), "[MyConnection0 is opened]");
      TRUE (conn0.isOpened());
      FALSE(conn0.isClosed());

      tran.commit();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened][MyConnection0 is committed]");
      TRUE (conn0.isOpened());
      FALSE(conn0.isClosed());
    } catch (Exception e) {
      tran.rollback();
      NG(e.toString());
    } finally {
      tran.end();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection0 is committed]" +
        "[MyConnection0 is rollbacked]" +
        "[MyConnection0 is closed]" +
      "");
    }

    CLEAR_LOG();

    tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    try {
      conn0 = tran.getQueryConnection(CID0);
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
      "");
      TRUE (conn0.isOpened());
      FALSE(conn0.isClosed());

      conn1 = tran.getQueryConnection(CID1);
      NOTEQUAL(conn1, conn0);
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
      "");
      TRUE (conn1.isOpened());
      FALSE(conn1.isClosed());

      conn2 = tran.getQueryConnection(CID2);
      NOTEQUAL(conn2, conn0);
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
        "[MyConnection2 is opened]" +
      "");
      TRUE (conn2.isOpened());
      FALSE(conn2.isClosed());

      EQUAL(tran.getQueryConnection(CID0), conn0);
      EQUAL(tran.getQueryConnection(CID1), conn1);
      EQUAL(tran.getQueryConnection(CID2), conn2);

      tran.commit();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
        "[MyConnection2 is opened]" +
        "[MyConnection0 is committed]" +
        "[MyConnection1 is committed]" +
        "[MyConnection2 is committed]" +
      "");

      EQUAL(tran.getQueryConnection(CID0), conn0);
      EQUAL(tran.getQueryConnection(CID1), conn1);
      EQUAL(tran.getQueryConnection(CID2), conn2);

    } catch (Exception e) {
      tran.rollback();
      NG(e.toString());
    } finally {
      tran.end();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
        "[MyConnection2 is opened]" +
        "[MyConnection0 is committed]" +
        "[MyConnection1 is committed]" +
        "[MyConnection2 is committed]" +
        "[MyConnection0 is rollbacked]" +
        "[MyConnection1 is rollbacked]" +
        "[MyConnection2 is rollbacked]" +
        "[MyConnection0 is closed]" +
        "[MyConnection1 is closed]" +
        "[MyConnection2 is closed]" +
      "");
    }
  }

  public void getQueryConnection_connId_Null()
  {
    MSG("引数がヌルの場合。");
 
    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    try {
      tran.getQueryConnection(null);
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionIdIsNullOrEmpty);
    } finally {
      tran.end();
    }
  }

  public void getQueryConnection_connId_NoConnectionConfig()
  {
    MSG("引数のIDに対応する接続設定ファイルが存在しない場合。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    try {
      tran.getQueryConnection("XXX");
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionConfigFileNotFound);
    } finally {
      tran.end();
    }
  }

  public void getQueryConnection_connId_IllegalState()
  {
    MSG("トランザクションの状態が不正な場合。");

    String CID0 = "QueryTransactionTest_findConnection_connId_0";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    try {
      tran.getQueryConnection(CID0);
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    } finally {
      tran.end();
    }
  }

  public void getQueryExecution_execId()
  {
    MSG("IQueryExecutionオブジェクトを取得するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(OUTPUT_LOG(), "");
    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    tran.end();
    EQUAL(OUTPUT_LOG(), "");
    CLEAR_LOG();

    String EID0 = "QueryTransactionManager_2Test_getQueryExecution_execId_0";
    String EID1 = "QueryTransactionManager_2Test_getQueryExecution_execId_1";
    String EID2 = "QueryTransactionManager_2Test_getQueryExecution_execId_2";

    IQueryExecution exec0 = null, exec1 = null, exec2 = null;

    tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    try {
      exec0 = tran.getQueryExecution(EID0);
      EQUAL(OUTPUT_LOG(), "[MyConnection0 is opened]");

      NOTEQUAL(tran.getQueryExecution(EID0), exec0);
      EQUAL(OUTPUT_LOG(), "[MyConnection0 is opened]");

      tran.commit();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened][MyConnection0 is committed]");
    } catch (Exception e) {
      tran.rollback();
      NG(e.toString());
      e.printStackTrace(System.out);
    } finally {
      tran.end();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection0 is committed]" +
        "[MyConnection0 is rollbacked]" +
        "[MyConnection0 is closed]" +
      "");
    }

    CLEAR_LOG();

    tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(OUTPUT_LOG(), "");
    try {
      exec0 = tran.getQueryExecution(EID0);
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
      "");

      exec1 = tran.getQueryExecution(EID1);
      NOTEQUAL(exec1, exec0);
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
      "");

      exec2 = tran.getQueryExecution(EID2);
      NOTEQUAL(exec2, exec0);
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
        "[MyConnection2 is opened]" +
      "");

      NOTEQUAL(tran.getQueryExecution(EID0), exec0);
      NOTEQUAL(tran.getQueryExecution(EID1), exec1);
      NOTEQUAL(tran.getQueryExecution(EID2), exec2);

      tran.commit();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
        "[MyConnection2 is opened]" +
        "[MyConnection0 is committed]" +
        "[MyConnection1 is committed]" +
        "[MyConnection2 is committed]" +
      "");

      NOTEQUAL(tran.getQueryExecution(EID0), exec0);
      NOTEQUAL(tran.getQueryExecution(EID1), exec1);
      NOTEQUAL(tran.getQueryExecution(EID2), exec2);

    } catch (Exception e) {
      tran.rollback();
      NG(e.toString());
    } finally {
      tran.end();
      EQUAL(OUTPUT_LOG(),
        "[MyConnection0 is opened]" +
        "[MyConnection1 is opened]" +
        "[MyConnection2 is opened]" +
        "[MyConnection0 is committed]" +
        "[MyConnection1 is committed]" +
        "[MyConnection2 is committed]" +
        "[MyConnection0 is rollbacked]" +
        "[MyConnection1 is rollbacked]" +
        "[MyConnection2 is rollbacked]" +
        "[MyConnection0 is closed]" +
        "[MyConnection1 is closed]" +
        "[MyConnection2 is closed]" +
      "");
    }
  }

  public void getQueryExecution_execId_Null()
  {
    MSG("引数がヌルか空文字列の場合。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }

    try {
      tran.getQueryExecution(null);
      NG();
    } catch (ReasonedException e) {
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionIdIsNullOrEmpty);
    }

    try {
      tran.getQueryExecution("");
      NG();
    } catch (ReasonedException e) {
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryExecutionConfig.Error.ExecutionIdIsNullOrEmpty);
    }

    tran.end();
  }

  public void getQueryExecution_execId_NoConnectionConfig()
  {
    MSG("引数のIDに対応する接続設定ファイルが存在しない場合。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }

    try {
      tran.getQueryExecution("XXX");
      NG();
    } catch (ReasonedException e) {
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryExecutionConfig.Error.ExecutionConfigFileNotFound);
    } finally {
      tran.end();
    }
  }

  public void getQueryExecution_execId_IllegalState()
  {
    MSG("トランザクションの状態が不正な場合。");

    String EID0 = "QueryTransactionManager_2Test_getQueryExecution_execId_0";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);

    try {
      tran.getQueryExecution(EID0);
      NG();
    } catch (ReasonedException e) {
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    try {
      tran.begin();
      NOTNULL(tran.getQueryExecution(EID0));
      tran.commit();
      NOTNULL(tran.getQueryExecution(EID0));
      tran.rollback();
      NOTNULL(tran.getQueryExecution(EID0));
    } catch (ReasonedException e) {
      NG(e);
    }

    tran.end();

    try {
      tran.getQueryExecution(EID0);
      NG();
    } catch (ReasonedException e) {
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    } finally {
      tran.end();
    }
  }

  public void getQuery_queryId()
  {
    MSG("IQueryオブジェクトを取得するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    tran.end();

    String QID0 = "QueryTransactionManager_2Test_getQuery_queryId_0";
    String QID1 = "QueryTransactionManager_2Test_getQuery_queryId_1";
    String QID2 = "QueryTransactionManager_2Test_getQuery_queryId_2";

    IQuery query0 = null, query1 = null, query2 = null;

    tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
      query0 = tran.getQuery(QID0);
      NOTEQUAL(query0, tran.getQuery(QID0));
      NOTEQUAL(query0, tran.getQuery(QID0));
      tran.commit();
    } catch (Exception e) {
      tran.rollback();
    } finally {
      tran.end();
    }

    tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
      query0 = tran.getQuery(QID0);
      query1 = tran.getQuery(QID1);
      query2 = tran.getQuery(QID2);
      NOTEQUAL(query0, tran.getQuery(QID0));
      NOTEQUAL(query1, tran.getQuery(QID1));
      NOTEQUAL(query2, tran.getQuery(QID2));
      tran.commit();
    } catch (Exception e) {
      tran.rollback();
    } finally {
      tran.end();
    }
  }

  public void getQuery_queryId_Null()
  {
    MSG("引数がヌルの場合。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    try {
      tran.getQuery(null);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.QueryIdIsNullOrEmpty);
    } catch (ReasonedException e) {
      tran.rollback();
    } finally {
      tran.end();
    }
  }

  public void getQuery_queryId_NoConnectionConfig()
  {
    MSG("引数のIDに対応する接続設定ファイルが存在しない場合。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    try {
      tran.getQuery("XXX");
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.QueryConfigFileNotFound);
    } finally {
      tran.end();
    }
  }

  public void getQuery_queryId_IllegalState()
  {
    MSG("トランザクションの状態が不正な場合。");

    String QID0 = "QueryTransactionManager_2Test_getQuery_queryId_0";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    try {
      tran.getQuery(QID0);
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    } finally {
      tran.end();
    }
  }

  public void begin()
  {
    MSG("トランザクションを開始するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 0L);
    long begin = tran.getBeginTimeMillis();
    POSITIVE(begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.begin();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.begin();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.begin();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }
  }

  public void begin_TimeoutIsSet()
  {
    MSG("トランザクションを開始するメソッドの確認。");
    MSG("- タイムアウト時間を設定した場合");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.setTimeoutMillis(123L);
    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 123L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 123L);
    long begin = tran.getBeginTimeMillis();
    POSITIVE(begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 123L);

    try {
      tran.begin();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 123L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 123L);

    try {
      tran.begin();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 123L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 123L);

    try {
      tran.begin();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }
  }

  public void commit()
  {
    MSG("トランザクションの結果を確定するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.commit();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 0L);
    long begin = tran.getBeginTimeMillis();
    POSITIVE(begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.rollback();
    EQUAL(tran.getState(), IQueryTransaction.State.Rollbacked);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.commit();
      NG();
    } catch (ReasonedException e) {
      NG(e);
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }
  }

  public void rollback()
  {
    MSG("トランザクションの結果を取り消すメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.rollback();
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 0L);
    long begin = tran.getBeginTimeMillis();
    POSITIVE(begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.rollback();
    EQUAL(tran.getState(), IQueryTransaction.State.Rollbacked);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.rollback();
    EQUAL(tran.getState(), IQueryTransaction.State.Rollbacked);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    try {
      tran.rollback();
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.IllegalState);
    }
  }

  public void rollback_ThrowReasonedException()
  {
    MSG("コネクションの1つがエラーになっても全てのコネクションのロールバックが実行されることを確認。");

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);

        // rollback実行時にエラーになるように設定
        THROW0 = new ReasonedException(Error.ERR);

        throw new Exception();
      } catch (Exception e) {
        EQUAL(tran.getState(), IQueryTransaction.State.Begined);
        tran.rollback();
        NG();
      } finally {
        tran.end();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), THROW0.getReason());
      EQUAL(e.getCause(), THROW0);
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }

  public void rollback_ThrowReasonedRuntimeException()
  {
    MSG("コネクションの1つがエラーになっても全てのコネクションのロールバックが実行されることを確認。");

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);

        // rollback実行時にエラーになるように設定
        THROW1 = new ReasonedRuntimeException(Error.ERR);

        throw new Exception();
      } catch (Exception e) {
        EQUAL(tran.getState(), IQueryTransaction.State.Begined);
        tran.rollback();
        NG();
      } finally {
        tran.end();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), THROW1.getReason());
      NULL(e.getCause());
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }

  public void rollback_ThrowException()
  {
    MSG("コネクションの1つがエラーになっても全てのコネクションのロールバックが実行されることを確認。");

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);

        // rollback実行時にエラーになるように設定
        THROW2 = new RuntimeException();

        throw new Exception();
      } catch (Exception e) {
        EQUAL(tran.getState(), IQueryTransaction.State.Begined);
        tran.rollback();
        NG();
      } finally {
        tran.end();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.FailToRollback);
      EQUAL(e.getCause(), THROW2);
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }

  public void end()
  {
    MSG("トランザクションを終了するメソッドの確認。");

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.setTimeoutMillis(1000L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    long begin = tran.getBeginTimeMillis();
    POSITIVE(begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    try {
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.setTimeoutMillis(1000L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    begin = tran.getBeginTimeMillis();
    POSITIVE(begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    try {
      tran.getQueryConnection(CID0);
      tran.getQueryConnection(CID1);
      tran.getQueryConnection(CID2);
      tran.commit();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);
  }

  public void end_ConnectionIsClosed()
  {
    MSG("コネクションの1つが既にクローズされている場合。");

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    EQUAL(tran.getState(), IQueryTransaction.State.Created);
    EQUAL(tran.getTimeoutMillis(), 0L);
    EQUAL(tran.getBeginTimeMillis(), 0L);
    EQUAL(tran.getLimitTimeMillis(), 0L);

    tran.setTimeoutMillis(1000L);

    try {
      tran.begin();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Begined);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    long begin = tran.getBeginTimeMillis();
    POSITIVE(begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    try {
      IQueryConnection conn0 = tran.getQueryConnection(CID0);
      IQueryConnection conn1 = tran.getQueryConnection(CID1);
      IQueryConnection conn2 = tran.getQueryConnection(CID2);
      tran.commit();

      conn1.close();
    } catch (ReasonedException e) {
      NG(e);
    }
    EQUAL(tran.getState(), IQueryTransaction.State.Committed);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);

    tran.end();
    EQUAL(tran.getState(), IQueryTransaction.State.Ended);
    EQUAL(tran.getTimeoutMillis(), 1000L);
    EQUAL(tran.getBeginTimeMillis(), begin);
    EQUAL(tran.getLimitTimeMillis(), begin + 1000L);
  }

  public void end_ThrowsReasonedExceptionAtRollback()
  {
    MSG("コネクションの1つがend中のrollback時に例外をスローする場合。");

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);
      } catch (Exception e) {
        tran.rollback();
      } finally {
        // rollback実行時にエラーになるように設定
        THROW0 = new ReasonedException(Error.ERR);
        tran.end();
        NG();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), THROW0.getReason());
      EQUAL(e.getCause(), THROW0);
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }

  public void end_ThrowsReasonedRuntimeExceptionAtRollback()
  {
    MSG("コネクションの1つがend中のrollback時に例外をスローする場合。");

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);
      } catch (Exception e) {
        tran.rollback();
      } finally {
        // rollback実行時にエラーになるように設定
        THROW1 = new ReasonedRuntimeException(Error.ERR);
        tran.end();
        NG();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), THROW1.getReason());
      NULL(e.getCause());
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }

  public void end_ThrowsExceptionAtRollback()
  {
    MSG("コネクションの1つがend中のrollback時に例外をスローする場合。");

    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);
      } catch (Exception e) {
        tran.rollback();
      } finally {
        // rollback実行時にエラーになるように設定
        THROW2 = new RuntimeException();
        tran.end();
        NG();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.FailToRollback);
      EQUAL(e.getCause(), THROW2);
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }
  
  public void end_ThrowsReasonedException()
  {
    MSG("コネクションの1つがend中のclose時に例外をスローする場合。");
 
    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);
      } catch (Exception e) {
        tran.rollback();
      } finally {
        NOT_THROW_AT_ROLLBACK = true;
        // rollback実行時にエラーになるように設定
        THROW0 = new ReasonedException(Error.ERR);
        tran.end();
        NG();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), THROW0.getReason());
      EQUAL(e.getCause(), THROW0);
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }
 
  public void end_ThrowsReasonedRuntimeException()
  {
    MSG("コネクションの1つがend中のclose時に例外をスローする場合。");
 
    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);
      } catch (Exception e) {
        tran.rollback();
      } finally {
        NOT_THROW_AT_ROLLBACK = true;
        // rollback実行時にエラーになるように設定
        THROW1 = new ReasonedRuntimeException(Error.ERR);
        tran.end();
        NG();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), THROW1.getReason());
      NULL(e.getCause());
    }

    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }
 
  public void end_ThrowsException()
  {
    MSG("コネクションの1つがend中のclose時に例外をスローする場合。");
 
    String CID0 = "QueryTransactionManager_2Test_getQueryConnection_connId_0";
    String CID1 = "QueryTransactionManager_2Test_getQueryConnection_connId_1";
    String CID2 = "QueryTransactionManager_2Test_getQueryConnection_connId_2";

    IQueryTransaction tran =
      IQueryTransactionManager.INSTANCE.createThreadLocalTransaction();

    try {
      try {
        tran.begin();
        tran.getQueryConnection(CID0);
        tran.getQueryConnection(CID1);
        tran.getQueryConnection(CID2);
      } catch (Exception e) {
        tran.rollback();
      } finally {
        NOT_THROW_AT_ROLLBACK = true;
        // rollback実行時にエラーになるように設定
        THROW2 = new RuntimeException();
        tran.end();
        NG();
      }
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), IQueryTransaction.Error.FailToEnd);
      EQUAL(e.getCause(), THROW2);
    }
 
    EQUAL(OUTPUT_LOG(),
      "[MyConnection0 is opened]" +
      "[MyConnection1 is opened]" +
      "[MyConnection2 is opened]" +
      "[MyConnection0 is rollbacked]" +
      "[MyConnection1 is rollbacked]" +
      "[MyConnection2 is rollbacked]" +
      "[MyConnection0 is closed]" +
      "[MyConnection1 is closed]" +
      "[MyConnection2 is closed]" +
    "");
  }
} 
