/*
 * QueryTransaction class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.LinkedHashMap;

/**
 * トランザクションを表すクラス。
 *
 * @author 佐藤隆之
 * @version $Id: QueryTransaction.java,v 1.24 2012-02-24 18:02:50 tayu Exp $
 */
public class QueryTransaction implements IQueryTransaction
{
  /** トランザクションの状態。 */
  private State state = State.Created;

  /** トランザクションの開始時刻 [msec]。 */
  private long beginTimeMillis = 0L;

  /** トランザクションの制限時刻 [msec]。 */
  private long limitTimeMillis = 0L;

  /** トランザクションの制限時間 [msec]。 */
  private long timeoutMillis = 0L;

  /** {@link IQueryConnection}オブジェクトを格納するマップ。 */
  private final Map<String,IQueryConnection> connectionMap = newConnectionMap();

  /** クエリの実行履歴。 */
  private final QueryHistory history = newQueryHistory();

  /**
   * デフォルト・コンストラクタ。
   * <br>
   * このクラスのインスタンスは{@link QueryTransactionManager}クラスによって
   * 生成されることを意図しているため、アクセス指定子を<tt>protected</tt>に
   * している。
   */
  protected QueryTransaction()
  {
    this.state = State.Created;
  }

  /**
   * {@link IQueryConnection}オブジェクトを格納するマップを作成する。
   *
   * @return {@link IQueryConnection}オブジェクトを格納するマップ。
   */
  protected Map<String,IQueryConnection> newConnectionMap()
  {
    return new LinkedHashMap<String,IQueryConnection>();
  }

  /**
   * クエリの実行履歴を作成する。
   *
   * @return クエリの実行履歴を格納する{@link QueryHistory}オブジェクト。
   */
  protected QueryHistory newQueryHistory()
  {
    return new QueryHistory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getBeginTimeMillis()
  {
    return this.beginTimeMillis;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getLimitTimeMillis()
  {
    if (this.beginTimeMillis <= 0 || this.timeoutMillis <= 0) {
      return 0L;
    }
    else {
      return (this.beginTimeMillis + this.timeoutMillis);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getTimeoutMillis()
  {
    return this.timeoutMillis;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setTimeoutMillis(long millis)
  {
    checkState(new State[]{ State.Created });
    this.timeoutMillis = millis;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryHistory getQueryHistory()
  {
    return this.history;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public State getState()
  {
    return this.state;
  }

  /**
   * トランザクションの状態を変更する。
   * <br>
   * 現在の状態が、第一引数に指定された状態に含まれていない場合は、例外をスロー
   * する。
   *
   * @param froms 許される現在の状態。
   * @param to 変更後の状態。
   * @throws ReasonedRuntimeException 現在の状態が、第一引数に指定された状態の
   *   いずれにも含まれていない場合。
   */
  protected final void changeState(State[] froms, State to)
    throws ReasonedRuntimeException
  {
    checkState(froms);
    this.state = to;
  }

  /**
   * 現在の状態が、指定された状態のいずれかに含まれるかどうかを判定する。
   *
   * @param allows 許される現在の状態。
   * @throws ReasonedRuntimeException 現在の状態が、指定された状態のいずれにも
   *   含まれていない場合。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected final void checkState(State[] allows)
    throws ReasonedRuntimeException
  {
    assert (allows != null) : "@param:allows is null.";

    for (State state : allows) {
      if (state == getState()) {
        return;
      }
    }

    throw new ReasonedRuntimeException(IQueryTransaction.Error.IllegalState,
      "[state=" + getState().name() + "]");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void begin() throws ReasonedException, ReasonedRuntimeException
  {
    changeState(new State[]{ State.Created }, State.Begining);

    this.beginTimeMillis = System.currentTimeMillis();

    if (getTimeoutMillis() > 0L) {
      this.limitTimeMillis = this.beginTimeMillis + getTimeoutMillis();
    }
    else {
      this.limitTimeMillis = 0L;
    }

    this.state = State.Begined;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void commit() throws ReasonedException, ReasonedRuntimeException
  {
    changeState(new State[]{ State.Begined, State.Committed, State.Rollbacked },
      State.Committing);

    for (IQueryConnection conn : this.connectionMap.values()) {
      conn.commit();
    }

    this.state = State.Committed;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void rollback() throws ReasonedRuntimeException
  {
    changeState(new State[]{ State.Begined, State.Committed, State.Rollbacked,
      State.Rollbacking, State.Rollbacked }, State.Rollbacking);

    ReasonedRuntimeException exc = null;

    for (IQueryConnection conn : this.connectionMap.values()) {
      try {
        conn.rollback();
      }
      catch (ReasonedRuntimeException e) {
        if (exc == null) {
          exc = e;
        }
      }
      catch (ReasonedException e) {
        if (exc == null) {
          exc = new ReasonedRuntimeException(e.getReason(),
            "[connection Id=" + conn.getConnectionId() + "]", e);
        }
      }
      catch (Exception e) {
        if (exc == null) {
          exc = new ReasonedRuntimeException(
            IQueryTransaction.Error.FailToRollback, 
            "[connection Id=" + conn.getConnectionId() + "]", e);
        }
      }
    }

    if (exc != null) {
      throw exc;
    }

    this.state = State.Rollbacked;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void end() throws ReasonedRuntimeException
  {
    State state = this.state;
    this.state = State.Ending;

    List<IQueryConnection> connLst = new ArrayList<IQueryConnection>(
      this.connectionMap.values());

    this.connectionMap.clear();

    ReasonedRuntimeException exc = null;

    if (state != State.Created && state != State.Ended) {
      for (IQueryConnection conn : connLst) {
        if (conn.isClosed()) {
          continue;
        }

        try {
          conn.rollback();
        }
        catch (ReasonedRuntimeException e) {
          if (exc == null) {
            exc = e;
          }
        }
        catch (ReasonedException e) {
          if (exc == null) {
            exc = new ReasonedRuntimeException(e.getReason(),
              "[connection Id=" + conn.getConnectionId() + "]", e);
          }
        }
        catch (Exception e) {
          if (exc == null) {
            exc = new ReasonedRuntimeException(
              IQueryTransaction.Error.FailToRollback,
              "[connection Id=" + conn.getConnectionId() + "]", e);
          }
        }
      }
    }

    for (IQueryConnection conn : connLst) {
      if (conn.isClosed()) {
        continue;
      }

      try {
        conn.close();
      }
      catch (ReasonedRuntimeException e) {
        if (exc == null) {
          exc = e;
        }
      }
      catch (ReasonedException e) {
        if (exc == null) {
          exc = new ReasonedRuntimeException(e.getReason(),
            "[connection Id=" + conn.getConnectionId() + "]", e);
        }
      }
      catch (Exception e) {
        if (exc == null) {
          exc = new ReasonedRuntimeException(
            IQueryTransaction.Error.FailToEnd,
            "[connection Id=" + conn.getConnectionId() + "]", e);
        }
      }
    }

    if (exc != null) {
      throw exc;
    }

    this.state = State.Ended;
  }

  /**
   * 指定された接続先IDに対応する{@link IQueryConnection}オブジェクトを取得
   * する。
   * <br>
   * 対応する{@link IQueryConnection}オブジェクトがこのトランザクション内に存在
   * したら、それを返す。
   * 存在しなかった場合は新規に作成し、このトランザクションに登録して返す。
   *
   * @param connId 接続先ID。
   * @return {@link IQueryConnection}オブジェクト。
   * @throws ReasonedException {@link IQueryConnection}オブジェクトの開始処理に
   *   失敗した場合。
   * @throws ReasonedRuntimeException {@link IQueryConnection}オブジェクトの
   *   作成に失敗した場合。
   */
  protected IQueryConnection findConnection(String connId)
    throws ReasonedException, ReasonedRuntimeException
  {
    IQueryConnection conn = this.connectionMap.get(connId);
    if (conn == null) {
      QueryConnectionConfig config = new QueryConnectionConfig(connId);
      conn = config.create();
      conn.open();
      this.connectionMap.put(config.getConnectionId(), conn);
    }
    return conn;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryConnection getQueryConnection(String connId)
    throws ReasonedException, ReasonedRuntimeException
  {
    checkState(new State[]{ State.Begined, State.Committed, State.Rollbacked });

    return findConnection(connId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryExecution getQueryExecution(String execId)
    throws ReasonedException, ReasonedRuntimeException
  {
    checkState(new State[]{ State.Begined, State.Committed, State.Rollbacked });

    QueryExecutionConfig execCfg = new QueryExecutionConfig(execId);
    return execCfg.create(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQuery getQuery(String queryId)
    throws ReasonedException, ReasonedRuntimeException
  {
    checkState(new State[]{ State.Begined, State.Committed, State.Rollbacked });

    QueryConfig queryCfg = new QueryConfig(queryId);
    return queryCfg.create(this);
  }
}
