/*
 * QueryHistory class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.table.Header;
import ts.util.table.Table;
import ts.util.table.Index;
import ts.util.table.MapComparator;
import ts.util.table.MapIterator;
import ts.util.table.ArrayListTable;
import ts.util.table.DefaultMapComparator;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.io.Serializable;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;

/**
 * 実行履歴を格納するクラス。
 * <br>
 * クエリ・トランザクションやクエリ・コネクションによって使用され、その開始から
 * 終了までの間に処理されたクエリの実行履歴を格納する。
 * 実行履歴として、以下の情報を保持する：
 * <ul>
 *  <li>クエリID</li>
 *  <li>接続先ID</li>
 *  <li>実行ID</li>
 *  <li>実行成否 (true/false)</li>
 *  <li>失敗時にスローされた例外オブジェクト</li>
 *  <li>開始時刻 [msec]</li>
 *  <li>終了時刻 [msec]</li>
 *  <li>処理時間 [msec]</li>
 *  <li>照会件数</li>
 *  <li>更新件数</li>
 *  <li>実行内容 (実行クラスにより異なる)</li>
 * </ul>
 * さらに、これらの情報をクエリID、接続先ID、実行ID、実行成否、例外オブジェクト
 * をキーに検索して取得することができる。
 *
 * @author 佐藤隆之
 * @version $Id: QueryHistory.java,v 1.4 2012-03-09 16:12:35 tayu Exp $
 */
public class QueryHistory implements IQueryHistory
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 1063227293342192945L;

  /** 実行履歴を格納するテーブル。 */
  protected final Table<Item,Serializable> historyTable = newHistoryTable();

  /** クエリIDをキーとする実行履歴テーブルのインデックス。 */
  protected final Index<Item,Serializable> historyIndexForQidCidEid =
    historyTable.getIndex(Item.QueryId, Item.ConnectionId, Item.ExecutionId);

  /** 接続先IDをキーとする実行履歴テーブルのインデックス。 */
  protected final Index<Item,Serializable> historyIndexForCidEid =
    historyTable.getIndex(Item.ConnectionId, Item.ExecutionId);

  /** 実行IDをキーとする実行履歴テーブルのインデックス。 */
  protected final Index<Item,Serializable> historyIndexForEid =
    historyTable.getIndex(Item.ExecutionId);

  /** 例外オブジェクトをキーとする実行履歴テーブルのインデックス。 */
  protected final Index<Item,Serializable> historyIndexForExc =
    historyTable.getIndex(Item.Exception);

  /** テーブル検索時のレコードのソートに使用する{@link MapComparator}。 */
  protected final MapComparator<Item,Serializable> sorter =
    new DefaultMapComparator<Item,Serializable>(Item.EntryDateTime);

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryHistory()
  {}

  /**
   * 実行履歴を格納するテーブルを作成する。
   *
   * @return 実行履歴を格納するテーブル。
   */
  protected Table<Item,Serializable> newHistoryTable()
  {
    Header<Item> header = new ArrayListTable.Header<Item>(12);
    header.addColumn(Item.QueryId);
    header.addColumn(Item.ConnectionId);
    header.addColumn(Item.ExecutionId);
    header.addColumn(Item.IsSuccess);
    header.addColumn(Item.Exception);
    header.addColumn(Item.BeginTime);
    header.addColumn(Item.EndTime);
    header.addColumn(Item.SpentTime);
    header.addColumn(Item.FetchCount);
    header.addColumn(Item.UpdateCount);
    header.addColumn(Item.Content);
    header.addColumn(Item.EntryDateTime);
    return new ArrayListTable<Item,Serializable>(header);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<Item,Serializable> appendNew()
  {
    return this.historyTable.appendNew();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getAll()
  {
    Map<Item,Serializable> cond = Collections.emptyMap();
    List<Map<Item,Serializable>> recL = this.historyTable.select(cond);
    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getByQueryId(String queryId)
  {
    List<Map<Item,Serializable>> recL =
      this.historyIndexForQidCidEid.select(Item.QueryId, queryId);
    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getByConnectionId(String connId)
  {
    List<Map<Item,Serializable>> recL =
      this.historyIndexForCidEid.select(Item.ConnectionId, connId);
    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getByConnectionId(String queryId,
    String connId)
  {
    Map<Item,Serializable> cond = new HashMap<Item,Serializable>();
    cond.put(Item.QueryId, queryId);
    cond.put(Item.ConnectionId, connId);

    List<Map<Item,Serializable>> recL =
      this.historyIndexForQidCidEid.select(cond);
    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getByExecutionId(String execId)
  {
    List<Map<Item,Serializable>> recL =
      this.historyIndexForEid.select(Item.ExecutionId, execId);
    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getByExecutionId(String connId,
    String execId)
  {
    Map<Item,Serializable> cond = new HashMap<Item,Serializable>();
    cond.put(Item.ConnectionId, connId);
    cond.put(Item.ExecutionId, execId);

    List<Map<Item,Serializable>> recL =
      this.historyIndexForCidEid.select(cond);
    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getByExecutionId(String queryId,
    String connId, String execId)
  {
    Map<Item,Serializable> cond = new HashMap<Item,Serializable>();
    cond.put(Item.QueryId, queryId);
    cond.put(Item.ConnectionId, connId);
    cond.put(Item.ExecutionId, execId);

    List<Map<Item,Serializable>> recL =
      this.historyIndexForQidCidEid.select(cond);
    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Map<Item,Serializable>> getByIsSuccess(boolean isSuccess)
  {
    List<Map<Item,Serializable>> recL =
      new LinkedList<Map<Item,Serializable>>();

    if (isSuccess) {
      MapIterator<Item,Serializable> recIt = this.historyTable.records();
      while (recIt.hasNext()) {
        Map<Item,Serializable> rec = recIt.next();
        if (Boolean.TRUE.equals(rec.get(Item.IsSuccess))) {
          recL.add(rec);
        }
      }
    }
    else {
      MapIterator<Item,Serializable> recIt = this.historyTable.records();
      while (recIt.hasNext()) {
        Map<Item,Serializable> rec = recIt.next();
        if (Boolean.FALSE.equals(rec.get(Item.IsSuccess))) {
          recL.add(rec);
        }
      }
    }

    Collections.sort(recL, this.sorter);
    return recL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<Item,Serializable> getByException(ReasonedException exc)
  {
    List<Map<Item,Serializable>> recL =
      this.historyIndexForCidEid.select(Item.Exception, exc);
    if (recL.isEmpty()) {
      return null;
    }
    else {
      return recL.get(0);
    }
  }
}
