/*
 * QueryExecutionContent class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import ts.util.table.Table;
import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;

/**
 * {@link QueryExecution}オブジェクトが実行する内容を表すクラス。
 * <br>
 * 実行内容を表す標準の情報として、このクラスは以下の情報を取得するメソッドを
 * 用意している。
 * <ul>
 *  <li>接続先ID</li>
 *  <li>実行ID</li>
 *  <li>入力パラメータ・リスト</li>
 *  <li>出力パラメータ・セット</li>
 *  <li>結果データ・テーブルの有無</li>
 *  <li>結果データ・テーブル</li>
 *  <li>処理件数</li>
 *  <li>実行に失敗した場合にスローされた例外オブジェクト</li>
 *  <li>終了時刻の制限値</li>
 *  <li>取得データ件数の制限値</li>
 * </ul>
 * <br>
 * このクラスは{@link java.util.Map}インターフェイスを継承しており、{@link
 * Enum}型のキーを使って任意のデータを上記以外の属性として保持することができる。
 * また、{@link ts.util.AbstractTypedGetter}インターフェイスを継承しているため、
 * {@link Enum}型のキーを使った属性値を<tt>int</tt>型などの基本データ型や日時を
 * 表す{@link ts.util.DateTime}オブジェクトに変換して取得することができる。
 * 
 * @author 佐藤隆之。
 * @version $Id: QueryExecutionContent.java,v 1.7 2012-03-14 07:49:20 tayu Exp $
 */
public class QueryExecutionContent
  extends AbstractTypedGetter<Enum<?>,Serializable> implements Serializable
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 2970137509789869821L;

  /** 接続先ID。 */
  private String connectionId = "";

  /** 実行ID。 */
  private String executionId = "";

  /** 入力パラメータ・リスト。 */
  private final List<QueryExecutionParam> paramLst = newParamList();

  /** 出力パラメータ・セット。 */
  private final Set<QueryExecutionOutput> outputSet = newOutputSet();

  /** 実行内容に応じた属性を格納するマップ。 */
  private Map<Enum<?>,Serializable> attributeMap = null;

  /** 結果テーブル。 */
  private Table<String,Serializable> resultTable = null;

  /** 取得データ件数。 */
  private int fetchCount = 0;
 
  /** 更新データ件数。 */
  private int updateCount = 0;

  /** 終了時刻の制限値 [msec]。値がゼロ以下の場合は制限なしとする。 */
  private long limitTimeMillis = 0L;

  /** 取得データ件数の制限値。値がゼロ以下の場合は制限なしとする。 */
  private int limitFetchCount = 0;

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryExecutionContent()
  {}

  /**
   * 入力パラメータ・リストを作成する。
   *
   * @return 入力パラメータ・リスト。
   */
  protected List<QueryExecutionParam> newParamList()
  {
    return new LinkedList<QueryExecutionParam>();
  }

  /**
   * 出力パラメータ・セットを作成する。
   *
   * @return 出力パラメータ・セット。
   */
  protected Set<QueryExecutionOutput> newOutputSet()
  {
    return new LinkedHashSet<QueryExecutionOutput>();
  }

  /**
   * 属性を格納するマップを作成する。
   *
   * @return 属性を格納するマップ。
   */
  protected Map<Enum<?>,Serializable> newAttributeMap()
  {
    return new HashMap<Enum<?>,Serializable>();
  }

  /**
   * 指定されたキーに結びつけられた属性値を取得する。
   *
   * @param key 属性キー。
   * @return 引数のキーに結びつけられた属性値。
   * @throws AssertionError 引数のキーがヌルの場合（デバッグ・モードのみ）。
   */
  @Override
  public Serializable get(Enum<?> key)
  {
    assert (key != null) : "@param;key is null.";

    return (this.attributeMap == null) ? null : this.attributeMap.get(key);
  }

  /**
   * 指定されたキーに対する属性値を設定する。
   *
   * @param key 属性キー。
   * @param value 属性値。 
   */
  public Serializable put(Enum<?> key, Serializable value)
  {
    assert (key != null) : "@param:key is null.";

    if (this.attributeMap == null) {
      this.attributeMap = newAttributeMap();
    }

    if (value == null) {
      return this.attributeMap.remove(key);
    }
    else {
      return this.attributeMap.put(key, value);
    }
  }

  /**
   * 実行時に使用される入力パラメータのリストを取得する。
   * <br>
   * 実行内容の中で同じ入力パラメータが複数回使用される場合は、その重複分も
   * 含めて取得される。
   *
   * @return 実行時に使用される入力パラメータのリスト。
   */
  public List<QueryExecutionParam> listParams()
  {
    return this.paramLst;
  }

  /**
   * 実行時に使用される入力パラメータの数を取得する。
   * <br>
   * 実行内容の中で同じ入力パラメータが複数回使用される場合は、その重複分も
   * 含めて数える。
   *
   * @return 実行時に使用される入力パラメータの数。
   */
  public int countParams()
  {
    return this.paramLst.size();
  }

  /**
   * 入力パラメータを追加する。
   *
   * @param param 入力パラメータ。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addParam(QueryExecutionParam param)
  {
    assert (param != null) : "@param:param is null.";

    this.paramLst.add(param);
  }

  /**
   * 実行結果の出力項目を取得する。
   * <br>
   * 実行内容の中で同じ出力項目が複数回現れても、その重複は除かれて取得される。
   *
   * @return 実行結果の出力項目のコレクション。
   */
  public Collection<QueryExecutionOutput> getOutputs()
  {
    return this.outputSet;
  }

  /**
   * 実行結果の出力項目の数を取得する。
   * <br>
   * 実行内容の中で同じ出力項目が複数回現れても、その重複は除かれて数えられる。
   *
   * @return 実行結果の出力項目の数。
   */
  public int countOutputs()
  {
    return this.outputSet.size();
  }

  /**
   * 出力項目を追加する。
   *
   * @param output 出力項目。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addOutput(QueryExecutionOutput output)
  {
    assert (output != null) : "@param:output is null.";

    this.outputSet.add(output);
  }

  /**
   * 接続先IDを設定する。
   *
   * @param connId 接続先ID。
   */
  public void setConnectionId(String connId)
  {
    assert (connId != null) : "@param:connId is null.";

    this.connectionId = connId;
  }

  /**
   * 接続先IDを取得する。
   *
   * @return 接続先ID。
   */
  public String getConnectionId()
  {
    return this.connectionId;
  }

  /**
   * 実行IDを設定する。
   *
   * @param execId 実行ID。
   */
  public void setExecutionId(String execId)
  {
    assert (execId != null) : "@param:execId is null.";

    this.executionId = execId;
  }

  /**
   * 実行IDを取得する。
   *
   * @return 実行ID。
   */
  public String getExecutionId()
  {
    return this.executionId;
  }

  /**
   * このオブジェクトの実行内容を実行した後に、結果テーブルを持つかどうかを
   * 判定する。
   *
   * @return 結果テーブルを持つ場合は<tt>true</tt>。
   */
  public boolean hasResultTable()
  {
    return (countOutputs() > 0) ? true : false;
  }

  /**
   * 結果テーブルを設定する。
   *
   * @param table 結果テーブル。
   */
  public void setResultTable(Table<String,Serializable> table)
  {
    this.resultTable = table;
  }

  /**
   * 結果テーブルを取得する。
   *
   * @return 結果テーブル。
   */
  public Table<String,Serializable> getResultTable()
  {
    return this.resultTable;
  }

  /**
   * 取得データ件数を設定する。
   *
   * @param count 取得データ件数。
   */
  public void setFetchCount(int count)
  {
    assert (count >= 0) : "@param:count is negative.";

    this.fetchCount = count;
  }

  /**
   * 取得データ件数を取得する。
   *
   * @return 取得データ件数。
   */
  public int getFetchCount()
  {
    return this.fetchCount;
  }

  /**
   * 更新データ件数を設定する。
   *
   * @param count 更新データ件数。
   */
  public void setUpdateCount(int count)
  {
    assert (count >= 0) : "@param:count is negative.";

    this.updateCount = count;
  }

  /**
   * 更新データ件数を取得する。
   *
   * @return 更新データ件数。
   */
  public int getUpdateCount()
  {
    return this.updateCount;
  }

  /**
   * 終了時刻の制限値を設定する。
   * <br>
   * 値がゼロ以下の場合は制限なしとする。
   *
   * @param limitTimeMillis 終了時刻の制限値 [msec]。
   */
  public void setLimitTimeMillis(long limitTimeMillis)
  {
    this.limitTimeMillis = limitTimeMillis;
  }

  /**
   * 終了時刻の制限値を取得する。
   * <br>
   * 値がゼロ以下の場合は制限なしとする。
   *
   * @return 終了時刻の制限値 [msec]。
   */
  public long getLimitTimeMillis()
  {
    return this.limitTimeMillis;
  }

  /**
   * 取得データ件数の制限値を設定する。
   * <br>
   * 値がゼロ以下の場合は制限なしとする。
   *
   * @param limitCount 取得データ件数の制限値 [msec]。
   */
  public void setLimitFetchCount(int limitCount)
  {
    this.limitFetchCount = limitCount;
  }

  /**
   * 取得データ件数の制限値を取得する。
   * <br>
   * 値がゼロ以下の場合は制限なしとする。
   *
   * @return 取得データ件数の制限値 [msec]。
   */
  public int getLimitFetchCount()
  {
    return this.limitFetchCount;
  }
}
