package ts.query;

import ts.tester.UnitTest;
import ts.util.*;
import ts.util.file.FileOperation;
import java.io.*;
import java.nio.channels.*;
import java.util.*;

public class QueryEnvironmentConfigTest extends UnitTest
{
  public static void main(String args[])
  {
    run(QueryEnvironmentConfigTest.class, args);
  }

  @Override
  protected void preInvocation(String method)
  {
    ResourceBundle.clearCache();
  }

  File DATA_DIR = new File("test/data");
  File SAMPLE_DIR = new File(DATA_DIR, "ts/query/QueryEnvironmentConfig");

  public void constructor()
  {
    MSG("コンストラクタの確認。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    NOTNULL(cfg);
  }

  public void getProperty_key()
  {
    MSG("プロパティを取得するメソッドの確認。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();

    EQUAL(cfg.getProperty("ts-query.environment.connection.config.directory"),
      "test/data/ts/query/configs/connection");

    EQUAL(cfg.getProperty("ts-query.environment.execution.config.directory"),
      "test/data/ts/query/configs/execution");

    EQUAL(cfg.getProperty("ts-query.environment.query.config.directory"),
      "test/data/ts/query/configs/query");

    EQUAL(cfg.getProperty("xxx.yyy.zzz"), "AAA");
    EQUAL(cfg.getProperty("aaa.bbb"), "");
  }

  public void getProperty_key_Null()
  {
    MSG("引数がヌルの場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();

    try {
      cfg.getProperty(null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryEnvironmentConfig.Error.FailToGetEnvironmentProperty);
      EQUAL(e.getMessage(), "[base=ts.query.QueryEnvironmentConfig][key=null]");
      EQUAL(e.getCause().getClass(), NullPointerException.class);
    }
  }

  public void getProperty_key_EmptyKey()
  {
    MSG("空のプロパティ・キーを指定した場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();

    try {
      cfg.getProperty("");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryEnvironmentConfig.Error.PropertyKeyNotFound);
      EQUAL(e.getMessage(), "[base=ts.query.QueryEnvironmentConfig][key=]");
      EQUAL(e.getCause().getClass(), MissingResourceException.class);
    }
  }

  public void getProperty_key_KeyNotExist()
  {
    MSG("存在しないプロパティ・キーを指定した場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();

    try {
      cfg.getProperty("a.b.c");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryEnvironmentConfig.Error.PropertyKeyNotFound);
      EQUAL(e.getMessage(),"[base=ts.query.QueryEnvironmentConfig][key=a.b.c]");
      EQUAL(e.getCause().getClass(), MissingResourceException.class);
    }
  }

  public void getProperty_key_PropertyFileNotFound()
  {
    MSG("プロパティ・ファイルが見つからない場合。");

    File file0 =new File(DATA_DIR,"ts/query/QueryEnvironmentConfig.org");
    File file1 =new File(DATA_DIR,"ts/query/QueryEnvironmentConfig.properties");

    try {
      if (file1.exists()) FileOperation.delete(file1);
      ResourceBundle.clearCache();
    }
    catch (Exception e) {
      NG(e);
    }

    QueryEnvironmentConfig env = new QueryEnvironmentConfig();

    try {
      env.getProperty("ts-query.environment.connection.config.directory");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.getCause().getMessage());
      EQUAL(e.getReason(),
        QueryEnvironmentConfig.Error.FailToLoadEnvironmentConfig);
      EQUAL(e.getMessage(), "[base=ts.query.QueryEnvironmentConfig]");
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      try {
        FileOperation.copy(file0, file1);
      }
      catch (Exception e) {
        NG(e);
      }
    }
  }

  public void getProperty_key_FailToLoadConfigFile()
  {
    MSG("プロパティ・ファイルがロードできない場合。");
    MSG("（ファイルにロックをかけてロードできないようにしている）");

    File file = new File(DATA_DIR,"ts/query/QueryEnvironmentConfig.properties");

    RandomAccessFile raf = null;
    FileLock lock = null;

    try {
      raf = new RandomAccessFile(file, "rw");
      FileChannel channel = raf.getChannel();
      lock = channel.lock(0, Long.MAX_VALUE, false);
    }
    catch (Exception e) {
      NG(e);
    }

    ResourceBundle.clearCache();

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();

    try {
      cfg.getProperty("ts-query.environment.connection.config.directory");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryEnvironmentConfig.Error.FailToLoadEnvironmentConfig);
      EQUAL(e.getMessage(), "[base=ts.query.QueryEnvironmentConfig]");
    }
    catch (Throwable e) {
      NG(e);
    }
    finally {
      try {
        if (lock != null) try { lock.release(); } catch (Exception e) {}
        if (raf != null) try { raf.close(); } catch (Exception e) {}
      }
      catch (Exception e) {
        NG(e);
      }
    }
  }

  public void getConnectionConfigDirectory()
  {
    MSG("接続設定ファイル格納ディレクトリを取得するメソッドの確認。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.getConnectionConfigDirectory(),
      "test/data/ts/query/configs/connection");
  }

  public void getExecutionConfigDirectory()
  {
    MSG("実行設定ファイル格納ディレクトリを取得するメソッドの確認。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.getExecutionConfigDirectory(),
      "test/data/ts/query/configs/execution");
  }

  public void getQueryConfigDirectory()
  {
    MSG("クエリ設定ファイル格納ディレクトリを取得するメソッドの確認。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.getQueryConfigDirectory(), "test/data/ts/query/configs/query");
  }

  public void outputSample_printWriter()
  {
    MSG("環境設定ファイルのサンプルを出力するメソッドの確認。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();

    File sampleFile = new File(SAMPLE_DIR,
      "QueryEnvironmentConfig.properties.sample");

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        sampleFile), "MS932"));
      cfg.outputSample(pw);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file = new File(SAMPLE_DIR, "QueryEnvironmentConfig.properties");
      if (file.exists()) FileOperation.delete(file);
      FileOperation.copy(sampleFile, file);

      ResourceBundle rb = ResourceBundle.getBundle(
        "ts.query.QueryEnvironmentConfig.QueryEnvironmentConfig");

      EQUAL(rb.getString("ts-query.environment.connection.config.directory"),
        "...");
      EQUAL(rb.getString("ts-query.environment.execution.config.directory"),
        "...");
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSample_printWriter_Null()
  {
    MSG("引数がヌルの場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();

    PrintWriter pw = null;
    try {
      cfg.outputSample(null);
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }
  }

  public void executeCommand_args()
  {
    MSG("環境設定ファイルのサンプルを出力するメソッドの確認。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[]{"pgm",
      "test/data/ts/query/QueryEnvironmentConfig", "MS932"}), 0);
  }

  public void executeCommand_args_Null()
  {
    MSG("引数がヌルの場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(null), 1);
  }

  public void executeCommand_args_ZeroArgs()
  {
    MSG("引数の配列の要素数がゼロの場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[0]), 1);
  }

  public void executeCommand_args_IllegalArgsCount()
  {
    MSG("引数の配列の要素数が不正な場合。");

    QueryEnvironmentConfig cfg;
    
    cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[]{"pgm"}), 1);

    cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[]{"pgm","test/data"}), 1);

    cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[]{"pgm","test/data","MSG932","x"}), 1);
  }

  public void executeCommand_args_IllegalEncoding()
  {
    MSG("指定されたエンコーディングが不正な場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[]{
      "pgm","test/data/ts/query/QueryEnvironmentConfig/", "XXX"}), 2);
  }

  public void executeCommand_args_DirectoryNotFound()
  {
    MSG("指定されたディレクトリが存在しない場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[]{
      "pgm","test/data/ts/query/QueryEnvironmentConfigXXX/", "MS932"}), 3);
  }

  public void executeCommand_args_IllegalDirectory()
  {
    MSG("指定されたディレクトリが通常のファイルだった場合。");

    QueryEnvironmentConfig cfg = new QueryEnvironmentConfig();
    EQUAL(cfg.executeCommand(new String[]{"pgm","test/data/ts/query/QueryEnvironmentConfig/QueryEnvironmentConfig.properties","MS932"}), 3);
  }
}
