package ts.query;

import ts.tester.UnitTest;
import ts.tester.function.ObjectInspector;
import ts.util.*;
import ts.util.file.*;
import ts.util.resource.*;
import java.io.*;
import java.nio.channels.*;
import java.util.*;

public class QueryConnectionConfigTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryConnectionConfigTest.class, args);
  }

  File DATA_DIR = new File("test/data");
  File CFG_DIR = new File(DATA_DIR, "ts/query/configs/connection/");
  File SAMPLE_DIR = new File(DATA_DIR, "ts/query/QueryConnectionConfig");

  static class MyConnection implements IQueryConnection {
    static final long serialVersionUID = -1L;
    private QueryConnectionConfig cfg;
    private boolean isClosed = true;
    public MyConnection(QueryConnectionConfig cfg) { this.cfg = cfg; }
    public MyConnection(QueryConnectionConfig cfg, IQueryTransaction tran)
    { this(cfg); }
    @Override
    public String getConnectionId() { return cfg.getConnectionId(); }
    @Override
    public long getLimitTimeMillis() { return 0L; }
    @Override
    public IQueryHistory getQueryHistory() { return null; }
    @Override
    public void open() throws ReasonedException {
      this.isClosed = false;
    }
    @Override
    public void commit() throws ReasonedException {}
    @Override
    public void rollback() throws ReasonedException {}
    @Override
    public void close() throws ReasonedException { this.isClosed = true; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public boolean isOpened() { return ! this.isClosed; }
  }

  static class MyConnection2 extends MyConnection {
    public MyConnection2(QueryConnectionConfig cfg) { super(cfg); }
    public MyConnection2(QueryConnectionConfig cfg, IQueryTransaction tran)
    { super(cfg, tran); }
  }

  static class MyConnection3 extends MyConnection {
    public MyConnection3() { super(new QueryConnectionConfig()); }
  }

  static class MyConnection4 extends MyConnection {
    public MyConnection4(QueryConnectionConfig cfg) {
      super(cfg);
      throw new RuntimeException("ERR1");
    }
    public MyConnection4(QueryConnectionConfig cfg, IQueryTransaction tran) {
      super(cfg, tran);
      throw new RuntimeException("ERR2");
    }
  }


  public void constructor()
  {
    MSG("デフォルト・コンストラクタ。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    EQUAL(config.getConnectionId(), "");
    NOTNULL(config.getResource());
    NULL(AbstractResource.class.cast(config.getResource()).getPath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getConnectionClass(), "");
    EQUAL(config.getLimitSpentTime(), 0L);
  }

  public void constructor_connId()
  {
    MSG("接続IDを引数にとるコンストラクタの確認。");

    String CID = "QueryConnectionConfigTest_constructor_connId";

    QueryConnectionConfig config = new QueryConnectionConfig(CID);
    EQUAL(config.getConnectionId(), CID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, CID + ".xml").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getConnectionClass(),
      "ts.query.QueryConnectionConfigTest$MyConnection2");
    EQUAL(config.getLimitSpentTime(), 20000L);
    EQUAL(config.typedGetter().getString("ts-query.connection.class"),
      "ts.query.QueryConnectionConfigTest$MyConnection2");
    EQUAL(config.typedGetter().getString("ts-query.connection.limit.spenttime"),
      "20000");
    EQUAL(config.typedGetter().getLong("ts-query.connection.limit.spenttime"),
      20000L);
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.method"),
      "driver2");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.driver"), 
      "jdbc.driver.Class2");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.userid"), 
      "db_user2");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.passwd"), 
      "db_pass2");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.url"), 
      "jdbc:url:2");
    List<?> lst = config.typedGetter().getList("ts-query.connection.jdbc.*");
    EQUAL(lst.get(0), "driver2");
    EQUAL(lst.get(1), "jdbc.driver.Class2");
    EQUAL(lst.get(2), "db_user2");
    EQUAL(lst.get(3), "db_pass2");
    EQUAL(lst.get(4), "jdbc:url:2");
  }

  public void constructor_connId_Null()
  {
    MSG("引数がヌルや空文字列の場合。");

    try {
      new QueryConnectionConfig(null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionIdIsNullOrEmpty);
      EQUAL(e.getMessage(), "[connection Id=null]");
    }

    try {
      new QueryConnectionConfig("");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionIdIsNullOrEmpty);
      EQUAL(e.getMessage(), "[connection Id=]");
    }
  }

  public void loadResource_LoadXmlFile()
  {
    MSG("XML形式の設定ファイルのロードの確認。");

    String CID = "QueryConnectionConfigTest_loadResource_LoadXmlFile";

    QueryConnectionConfig config = new QueryConnectionConfig(CID);
    EQUAL(config.getConnectionId(), CID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, CID + ".xml").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getConnectionClass(),
      "ts.query.QueryConnectionConfigTest$MyConnection3");
    EQUAL(config.getLimitSpentTime(), 30000L);
    EQUAL(config.typedGetter().getString("ts-query.connection.class"),
      "ts.query.QueryConnectionConfigTest$MyConnection3");
    EQUAL(config.typedGetter().getString("ts-query.connection.limit.spenttime"),
      "30000");
    EQUAL(config.typedGetter().getLong("ts-query.connection.limit.spenttime"),
      30000L);
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.method"),
      "driver3");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.driver"), 
      "jdbc.driver.Class3");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.userid"), 
      "db_user3");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.passwd"), 
      "db_pass3");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.url"), 
      "jdbc:url:3");
    List<?> lst = config.typedGetter().getList("ts-query.connection.jdbc.*");
    EQUAL(lst.get(0), "driver3");
    EQUAL(lst.get(1), "jdbc.driver.Class3");
    EQUAL(lst.get(2), "db_user3");
    EQUAL(lst.get(3), "db_pass3");
    EQUAL(lst.get(4), "jdbc:url:3");
  }

  public void loadResource_LoadPropFile()
  {
    MSG("Javaプロパティ形式の設定ファイルのロードの確認。");

    String CID = "QueryConnectionConfigTest_loadResource_LoadPropFile";

    QueryConnectionConfig config = new QueryConnectionConfig(CID);
    EQUAL(config.getConnectionId(), CID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, CID + ".properties").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getConnectionClass(),
      "ts.query.QueryConnectionConfigTest$MyConnection4");
    EQUAL(config.getLimitSpentTime(), 40000L);
    EQUAL(config.typedGetter().getString("ts-query.connection.class"),
      "ts.query.QueryConnectionConfigTest$MyConnection4");
    EQUAL(config.typedGetter().getString("ts-query.connection.limit.spenttime"),
      "40000");
    EQUAL(config.typedGetter().getLong("ts-query.connection.limit.spenttime"),
      40000L);
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.method"),
      "driver4");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.driver"), 
      "jdbc.driver.Class4");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.userid"), 
      "db_user4");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.passwd"), 
      "db_pass4");
    EQUAL(config.typedGetter().getString("ts-query.connection.jdbc.url"), 
      "jdbc:url:4");
    List<?> lst = config.typedGetter().getList("ts-query.connection.jdbc.*");
    EQUAL(lst.get(0), "driver4");
    EQUAL(lst.get(1), "jdbc.driver.Class4");
    EQUAL(lst.get(2), "db_user4");
    EQUAL(lst.get(3), "db_pass4");
    EQUAL(lst.get(4), "jdbc:url:4");
  }

  public void loadResource_FailToLoadConfig_Xml()
  {
    MSG("XML形式の設定ファイルのロードに失敗した場合。");

    String CID = "QueryConnectionConfigTest_loadResource_FailToLoadConfig_Xml";

    try {
      new QueryConnectionConfig(CID);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.FailToLoadConnectionConfigFile);
    }
  }

  public void loadResource_FailToLoadConfig_Prop()
  {
    MSG("Javaプロパティ形式の設定ファイルのロードに失敗した場合。");

    String CID = "QueryConnectionConfigTest_loadResource_FailToLoadConfig_Prop";
    File file = new File(CFG_DIR, CID + ".properties");

    RandomAccessFile raf = null;
    FileLock lock = null;

    try {
      raf = new RandomAccessFile(file, "rw");
      FileChannel channel = raf.getChannel();
      lock = channel.lock(0, Long.MAX_VALUE, false);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      new QueryConnectionConfig(CID);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.FailToLoadConnectionConfigFile);
    }
    catch (Throwable e) {
      NG(e);
    }
    finally {
      try {
        if (lock != null) try { lock.release(); } catch (Exception e) {}
        if (raf  != null) try { raf.close(); } catch (Exception e) {}
      }
      catch (Exception e) {
        NG(e);
      }
    }
  }

  public void loadResource_ConfigFileNotFound()
  {
    MSG("設定ファイルがXML形式でもJavaプロパティ形式でも存在しなかった場合。");

    String CID = "QueryConnectionConfigTest_loadResource_ConfigFileNotFound";

    try {
      new QueryConnectionConfig(CID);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionConfigFileNotFound);
    }
  }

  public void getConnectionClass()
  {
    MSG("コネクション・クラス名の設定と取得の確認。");

    QueryConnectionConfig config;

    config = new QueryConnectionConfig();
    EQUAL(config.getConnectionClass(), "");

    config.getResource().setFirstValue("ts-query.connection.class", "X");
    EQUAL(config.getConnectionClass(), "X");

    config.getResource().setFirstValue("ts-query.connection.class", "Y");
    EQUAL(config.getConnectionClass(), "Y");

    config.getResource().removeChildren("ts-query.connection.class");
    EQUAL(config.getConnectionClass(), "");
  }

  public void getLimitSpentTime()
  {
    MSG("制限時間の設定と取得の確認。");

    QueryConnectionConfig config;

    config = new QueryConnectionConfig();
    EQUAL(config.getLimitSpentTime(), 0L);

    config.getResource().setFirstValue("ts-query.connection.limit.spenttime",
      "1000");
    EQUAL(config.getLimitSpentTime(), 1000L);

    config.getResource().setFirstValue("ts-query.connection.limit.spenttime",
      "9999");
    EQUAL(config.getLimitSpentTime(), 9999L);

    config.getResource().setFirstValue("ts-query.connection.limit.spenttime",
      "-111");
    EQUAL(config.getLimitSpentTime(), -111L);

    config.getResource().setFirstValue("ts-query.connection.limit.spenttime",
      "-0");
    EQUAL(config.getLimitSpentTime(), 0L);

    config.getResource().setFirstValue("ts-query.connection.limit.spenttime",
      "");
    EQUAL(config.getLimitSpentTime(), 0L);

    config.getResource().setFirstValue("ts-query.connection.limit.spenttime",
      "   ");
    EQUAL(config.getLimitSpentTime(), 0L);

    config.getResource().removeChildren("ts-query.connection.limit.spenttime");
    EQUAL(config.getLimitSpentTime(), 0L);
  }

  public void getLimitSpentTime_IllegalLimitSpentTime()
  {
    MSG("制限時間に不正な値を設定した場合。");

    QueryConnectionConfig config = new QueryConnectionConfig(); 
    config.getResource().setFirstValue("ts-query.connection.limit.spenttime",
      "abc");

    try {
      config.getLimitSpentTime();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConnectionConfig.Error.IllegalLimitSpentTime);
    }
  }

  public void create()
  {
    MSG("IQueryConnectionオブジェクトを作成するメソッドの確認。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection");

    try {
      MyConnection conn = config.create();
      NOTNULL(conn);
      EQUAL(conn.getConnectionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      IQueryConnection conn = config.create();
      NOTNULL(conn);
      EQUAL(conn.getConnectionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_ClassNameIsEmpty()
  {
    MSG("クラス名が空の場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConnectionConfig.Error.ConnectionClassNotFound);
    }
  }

  public void create_ClassNotFound()
  {
    MSG("設定された名前のクラスが存在しない場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfig$MyConnectionXXX");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), ClassNotFoundException.class);
      EQUAL(e.getReason(), QueryConnectionConfig.Error.ConnectionClassNotFound);
    }
  }

  public void create_ConstructorNotFound()
  {
    MSG("指定されたクラスに要求されるコンストラクタが存在しない場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection3");

    try {
      config.create();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionConstructorNotFound);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_FailToCreate()
  {
    MSG("IQueryConnectionオブジェクトの作成に失敗した場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection4");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConnectionConfig.Error.FailToCreateConnection);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_ClassNameIsDifferent()
  {
    MSG("設定されたクラスと戻り値のクラスが異なる場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection");

    try {
      MyConnection2 conn = config.create();
      NG();
    }
    catch (ClassCastException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran()
  {
    MSG("IQueryConnectionオブジェクトを作成するメソッドの確認。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection");

    try {
      MyConnection conn = config.create(tran);
      NOTNULL(conn);
      EQUAL(conn.getConnectionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      IQueryConnection conn = config.create(tran);
      NOTNULL(conn);
      EQUAL(conn.getConnectionId(), "");
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_ClassNameIsEmpty()
  {
    MSG("クラス名が空の場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConnectionConfig config = new QueryConnectionConfig();

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConnectionConfig.Error.ConnectionClassNotFound);
    }
  }

  public void create_tran_ClassNotFound()
  {
    MSG("設定された名前のクラスが存在しない場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfig$MyConnectionXXX");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), ClassNotFoundException.class);
      EQUAL(e.getReason(), QueryConnectionConfig.Error.ConnectionClassNotFound);
    }
  }

  public void create_tran_ConstructorNotFound()
  {
    MSG("指定されたクラスに要求されるコンストラクタが存在しない場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection3");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(),
        QueryConnectionConfig.Error.ConnectionConstructorNotFound);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_FailToCreate()
  {
    MSG("IQueryConnectionオブジェクトの作成に失敗した場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection4");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConnectionConfig.Error.FailToCreateConnection);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_ClassNameIsDifferent()
  {
    MSG("設定されたクラスと戻り値のクラスが異なる場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConnectionConfig config = new QueryConnectionConfig();
    config.getResource().setFirstValue("ts-query.connection.class",
      "ts.query.QueryConnectionConfigTest$MyConnection");

    try {
      MyConnection2 conn = config.create(tran);
      NG();
    }
    catch (ClassCastException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleXml_pw()
  {
    MSG("XML形式の接続設定ファイルの出力メソッドの確認。");

    String CID = "QueryConnectionConfigTest_outputSampleXml_pw";

    File file = new File(SAMPLE_DIR, CID + ".xml");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "Windows-31J")); 

      QueryConnectionConfig config = new QueryConnectionConfig();
      config.outputSampleXml(pw, "Windows-31J");
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, CID + ".xml");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConnectionConfig config = new QueryConnectionConfig(CID);
      EQUAL(config.getConnectionClass(), "...");
      EQUAL(config.getLimitSpentTime(), 0);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void outputSampleXml_pw_Null()
  {
    MSG("引数がヌルの場合。");

    PrintWriter pw = null;
    try {
      QueryConnectionConfig config = new QueryConnectionConfig();
      config.outputSampleXml(null, "Windows-31J");
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      pw = new PrintWriter(new StringWriter());
      QueryConnectionConfig config = new QueryConnectionConfig();
      config.outputSampleXml(pw, null);
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }
  }

  public void outputSampleXmlEntries_pw_Inheritance()
  {
    MSG("XMLエントリを出力するメソッドの継承の確認。");

    String CID =
      "QueryConnectionConfigTest_outputSampleXmlEntries_pw_Inheritance";

    File file = new File(SAMPLE_DIR, CID + ".xml");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "UTF-8"));

      QueryConnectionConfig config = new QueryConnectionConfig() {
        static final long serialVersionUID = -1L;
        @Override
        public void outputSampleXmlEntries(PrintWriter pw) throws IOException {
          pw.println("      <eeee>");
          pw.println("        <ffff>FFFF</ffff>");
          pw.println("      </eeee>");
        }
      };
      config.outputSampleXml(pw, "UTF-8");
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, CID + ".xml");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConnectionConfig config = new QueryConnectionConfig(CID);
      EQUAL(config.getConnectionClass(), "...");
      EQUAL(config.getLimitSpentTime(), 0);
      EQUAL(config.typedGetter().getString("ts-query.connection.eeee.ffff"),
        "FFFF");
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleProp_pw()
  {
    MSG("プロパティ形式の接続設定ファイルの出力メソッドの確認。");

    String CID = "QueryConnectionConfigTest_outputSampleProp_pw";

    File file = new File(SAMPLE_DIR, CID + ".properties");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "Windows-31J"));

      QueryConnectionConfig config = new QueryConnectionConfig();
      config.outputSampleProp(pw);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, CID + ".properties");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConnectionConfig config = new QueryConnectionConfig(CID);
      EQUAL(config.getConnectionClass(), "...");
      EQUAL(config.getLimitSpentTime(), 0);
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleProp_pw_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      QueryConnectionConfig config = new QueryConnectionConfig();
      config.outputSampleProp(null);
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSamplePropEntries_pw_Inheritance()
  {
    MSG("プロパティ・エントリを出力するメソッドの継承の確認。");

    String CID =
      "QueryConnectionConfigTest_outputSamplePropEntries_pw_Inheritance";

    File file = new File(SAMPLE_DIR, CID + ".properties");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
        "UTF-8"));

      QueryConnectionConfig config = new QueryConnectionConfig() {
        static final long serialVersionUID = -1L;
        @Override
        public void outputSamplePropEntries(PrintWriter pw) throws IOException {
          pw.println("ts-query.connection.eeee.ffff = FFFF");
        }
      };
      config.outputSampleProp(pw);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, CID + ".properties");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConnectionConfig config = new QueryConnectionConfig(CID);
      EQUAL(config.getConnectionClass(), "...");
      EQUAL(config.getLimitSpentTime(), 0);
      EQUAL(config.typedGetter().getString("ts-query.connection.eeee.ffff"),
        "FFFF");
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void executeCommand_args_config_FileTypeIsXml()
  {
    MSG("接続設定ファイルのサンプルを出力するメソッドの確認(XML)。");

    QueryConnectionConfig config = new QueryConnectionConfig();

    String CID =
      "QueryConnectionConfigTest_executeCommand_args_config_FileTypeIsXml";

    EQUAL(QueryConnectionConfig.executeCommand(new String[]{ "pgm",
      new File(SAMPLE_DIR, CID + ".xml").getPath(), "xMl", "UTF-8" },
      config), 0);
  }

  public void executeCommand_args_config_FileTypeIsProp()
  {
    MSG("接続設定ファイルのサンプルを出力するメソッドの確認(Prop)。");

    QueryConnectionConfig config = new QueryConnectionConfig();

    String CID =
      "QueryConnectionConfigTest_executeCommand_args_config_FileTypeIsProp";

    EQUAL(QueryConnectionConfig.executeCommand(new String[]{ "pgm",
      new File(SAMPLE_DIR, CID + ".properties").getPath(), "pROp", "UTF-8" },
      config), 0);
  }

  public void executeCommand_args_config_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    EQUAL(QueryConnectionConfig.executeCommand(null, config), 1);
  }

  public void executeCommand_args_config_ZeroArgs()
  {
    MSG("引数の配列の要素数がゼロの場合。");

    QueryConnectionConfig config = new QueryConnectionConfig();
    EQUAL(QueryConnectionConfig.executeCommand(new String[0], config), 1);
  }

  public void executeCommand_args_config_IllegalArgsCount()
  {
    MSG("引数の配列の要素数が不正な場合。");

    QueryConnectionConfig cfg = new QueryConnectionConfig();

    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"p"}, cfg), 1);
    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"p","q"}, cfg), 1);
    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"p","q","xml"},cfg),
      1);
    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"p","q","xml",
      "UTF8", "t"}, cfg), 1);
    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"p","q","xml",
      "UTF8", "t", "u"}, cfg), 1);
  }

  public void executeCommand_args_config_IllegalOutputFileType()
  {
    MSG("出力ファイルの形式が不正な場合。");

    QueryConnectionConfig cfg = new QueryConnectionConfig();

    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"pgm",
      new File(SAMPLE_DIR, "xxx.xml").getPath(), "XXX", "UTF-8"}, cfg), 1);
  }

  public void executeCommand_args_config_IllegalEncoding()
  {
    MSG("出力ファイルの文字エンコーディングが不正な場合。");

    QueryConnectionConfig cfg = new QueryConnectionConfig();

    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"pgm",
      new File(SAMPLE_DIR, "xxx.xml").getPath(), "XML", "UUUU"}, cfg), 2);
  }

  public void executeCommand_args_config_OutputFilePathIsBad()
  {
    MSG("出力ファイルのパスが不正な場合。");
    MSG("- 指定したパスが既存のディレクトリだった場合。");

    QueryConnectionConfig cfg = new QueryConnectionConfig();

    EQUAL(QueryConnectionConfig.executeCommand(new String[]{"pgm",
      SAMPLE_DIR.getPath(), "XML", "UTF8"}, cfg), 3);
  }
}
