/*
 * Query class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.resource.Resource;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.util.Iterator;
import java.util.Map;

/**
 * １つのまとまったクエリを表すクラス。
 * <br>
 * 任意の接続先に対する一つ又は複数の照会や更新からなるクエリを処理し、結果を
 * {@link IQueryResult}に格納して返す。
 *
 * @author 佐藤隆之
 * @version $Id: Query.java,v 1.23 2012-03-13 18:54:42 tayu Exp $
 */
public class Query implements IQuery
{
  /** クエリ設定オブジェクト。 */
  private final QueryConfig config;

  /** クエリ・トランザクション・オブジェクト。 */
  private IQueryTransaction transaction;

  /** このオブジェクトの中でトランザクション制御が必要かを表すフラグ。 */
  private final boolean isNeededToControlTransaction;

  /**
   * クエリ設定オブジェクトを引数にとるコンストラクタ。
   *
   * @param config クエリ設定オブジェクト。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected Query(QueryConfig config)
  {
    assert (config != null) : "@param:config is null.";

    this.config = config;
    this.transaction = createDefaultTransaction();
    this.isNeededToControlTransaction = true;
  }

  /**
   * クエリ設定オブジェクトとクエリ・トランザクション・オブジェクトを引数とる
   * コンストラクタ。
   *
   * @param config クエリ設定オブジェクト。
   * @param tran クエリ・トランザクション・オブジェクト。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected Query(QueryConfig config, IQueryTransaction tran)
  {
    assert (config != null && tran != null) :
      (config == null) ? "@param:config is null." : 
      (tran == null) ? "@param:tran is null." : "";

    this.config = config;
    this.transaction = tran;
    this.isNeededToControlTransaction = false;
  }

  /**
   * 外部からトランザクションを渡されなかった場合に、使用するトランザクションを
   * 自分で作成する。
   *
   * @return 自分で使用するトランザクション・オブジェクト。
   */
  protected IQueryTransaction createDefaultTransaction()
  {
    return new QueryTransaction();
  }

  /**
   * クエリ設定オブジェクトを取得する。
   *
   * @return クエリ設定オブジェクト。
   */
  protected QueryConfig getConfig()
  {
    return this.config;
  }

  /**
   * クエリ・トランザクション・オブジェクトを取得する。
   *
   * @return クエリ・トランザクション・オブジェクト。
   */
  protected IQueryTransaction getTransaction()
  {
    return this.transaction;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryHistory getQueryHistory()
  {
    return getTransaction().getQueryHistory();
  }

  /**
   * このオブジェクトの中でトランザクションの制御か必要かどうかを判定する。
   *
   * @return トランザクションの制御が必要な場合は<tt>true</tt>。
   */
  protected boolean isNeededToControlTransaction()
  {
    return this.isNeededToControlTransaction;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException
  {
    QueryResult result = new QueryResult(getConfig().getQueryId());
    result.setBeginTimeMillis(System.currentTimeMillis());

    if (isNeededToControlTransaction()) {
      controlTransaction(inputMap, result);
    }
    else {
      executeQuery(inputMap, result);
    }

    result.setEndTimeMillis(System.currentTimeMillis());
    return result;
  }

  /**
   * トランザクションの制御を行う。
   * <br>
   * トランザクションの開始から終了までの制御と、その間のクエリの実行を処理
   * する。
   *
   * @param inputMap 入力パラメータ・マップ。
   * @param result クエリ結果オブジェクト。
   * @throws ReasonedException トランザクションの制御やクエリの実行に失敗した
   *   場合。
   * @throws ReasonedRuntimeException クエリ設定に不正があった場合。
   */
  protected void controlTransaction(Map<String,Object> inputMap,
    QueryResult result) throws ReasonedException, ReasonedRuntimeException
  {
    if (this.isNeededToControlTransaction) {
      if (this.transaction.getState() == IQueryTransaction.State.Ended) {
        this.transaction = createDefaultTransaction();
      }
    }

    IQueryTransaction transaction = getTransaction();
    try {
      transaction.begin();
      executeQuery(inputMap, result);
      transaction.commit();
    }
    catch (ReasonedException e) {
      transaction.rollback();
      throw e;
    }
    catch (ReasonedRuntimeException e) {
      transaction.rollback();
      throw e;
    }
    catch (RuntimeException e) {
      transaction.rollback();
      throw e;
    }
    finally {
      transaction.end();
    }
  }

  /**
   * クエリを実行する。
   * <br>
   * このオブジェクトが保持するクエリ設定で指定されている実行IDに対応する{@link
   * IQueryExecution}オブジェクト（複数可）を処理する。
   *
   * @param inputMap 入力パラメータ・マップ。
   * @param result クエリ結果オブジェクト。
   * @throws ReasonedException {@link IQueryExecution}オブジェクトの処理に失敗
   *   した場合。
   * @throws ReasonedRuntimeException クエリ設定や実行設定が不正だった場合。
   */
  protected void executeQuery(Map<String,Object> inputMap,
    QueryResult result) throws ReasonedException, ReasonedRuntimeException
  {
    IQueryTransaction transaction = getTransaction();

    for (String execId : getConfig().getExecutionIds()) {
      QueryExecutionConfig execCfg = new QueryExecutionConfig(execId);
      IQueryExecution execution = execCfg.create(transaction);
      execution.execute(inputMap, result);
    }
  }
}
