/*
 * IQueryResult class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.TypedMap;
import ts.util.ReasonedException;
import ts.util.table.Table;
import java.io.Serializable;
import java.util.Collection;

/**
 * {@link IQuery}クラスの実行結果を格納するクラスのインターフェイス。
 * <br>
 * 実行の成否やステータス、処理時間、失敗時の例外オブジェクト、結果テーブルなど
 * を取得するためのメソッドを宣言している。
 *
 * @author 佐藤隆之
 * @version $Id: IQueryResult.java,v 1.6 2012-03-14 07:49:19 tayu Exp $
 */
public interface IQueryResult extends Serializable
{
  /** エラーの理由を示す列挙型。 */
  enum Error {
    /** 何らかの理由で実行に失敗した場合。 */
    FailToExecute,
  }

  /**
   * クエリIDを取得する。
   */
  String getQueryId();

  /**
   * 実行結果の成否を取得する。
   *
   * @return 成功の場合は<tt>true</tt>、失敗の場合は<tt>false</tt>を返す。
   */
  boolean isSuccess();

  /**
   * 開始時刻を取得する。
   *
   * @return 開始時刻 [msec]。
   */
  long getBeginTimeMillis();

  /**
   * 終了時刻を取得する。
   *
   * @return 終了時刻 [msec]。
   */
  long getEndTimeMillis();

  /**
   * 処理時間を取得する。
   *
   * @return 実行時間 [msec]。
   */
  long getSpentTimeMillis();

  /**
   * 失敗時の例外オブジェクトを取得する。
   *
   * @return 例外オブジェクト。
   */
  ReasonedException getException();

  /**
   * 全ての結果IDを格納したコレクションを取得する。
   *
   * @return 結果IDを格納したコレクション・オブジェクト。
   */
  Collection<String> getAllResultIds();

  /**
   * 指定された実行IDの結果データ・テーブルを取得する。
   * <br>
   * 同じ実行IDの処理が複数回実行されている場合は、それらの結果データを実行順
   * にまとめた一つのテーブルを返す。
   *
   * @return 指定された実行IDの結果データ・テーブル。
   */
  Table<String,Serializable> getResultTable(String rsltId);
}
