/*
 * IQueryExecution class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.util.Map;

/**
 * クエリ処理の中の一回の照会又は更新の実行を表すクラスのインターフェイス。
 * <br>
 * 一回のクエリ処理の中で、異なる接続先に対する複数の照会や更新を実行することが
 * できるが、このクラスはその一つ一つの照会／更新の実行に対応している。
 * どの照会／更新処理を実行するかは、クエリの実行設定を保持する{@link
 * QueryExecutionConfig}オブジェクトから取得する。
 * 
 * @author 佐藤隆之
 * @version $Id: IQueryExecution.java,v 1.3 2012-02-17 17:14:08 tayu Exp $
 */
public interface IQueryExecution
{
  /** この継承クラスで発生しうるエラーの列挙型。 */
  enum Error {
    /** タイムアウトが発生した場合。 */
    Timeout,

    /** 取得データ件数が制限を超えた場合。 */
    FetchCountOver,
  }

  /**
   * 実行IDを取得する。
   *
   * @return 実行ID。
   */
  String getExecutionId();

  /**
   * このオブジェクトが表す照会又は更新処理を実行する。
   * <br>
   * 実行結果は、{@link QueryResult}オブジェクトに格納して返す。
   *
   * @param inputMap 入力パラメータ・マップ。
   * @return クエリの実行結果を格納した{@link QueryResult}オブジェクト。
   * @throws ReasonedException このオブジェクトが表す処理の実行に失敗した場合。
   * @throws ReasonedRuntimeException 実行設定等が不正だった場合。
   */
  IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException;

  /**
   * このオブジェクトが表す照会又は更新処理を実行する。
   * <br>
   * 実行結果は、{@link QueryResult}オブジェクトに格納される。
   *
   * @param inputMap 入力パラメータ・マップ。
   * @param result クエリの実行結果を格納した{@link QueryResult}オブジェクト。
   * @throws ReasonedException このオブジェクトが表す処理の実行に失敗した場合。
   * @throws ReasonedRuntimeException 実行設定等が不正だった場合。
   */
  void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException;
}
