/*
 * ControllableQuery class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.TypedMap;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.resource.Resource;
import ts.util.resource.XmlResource;
import ts.util.table.MapIterator;
import ts.util.table.Table;
import ts.util.text.StringCutter;
import java.io.File;
import java.io.Serializable;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

/**
 * 入力パラメータや結果データによって、処理される実行IDの制御を行うクエリ・
 * クラス。
 * <br>
 * クエリ設定オブジェクト内の<tt>ts-query.query.execution.control</tt>要素以下
 * の要素ツリーを解釈して実行IDの求め、それに対応する{@link IQueryExecution}
 * オブジェクトを実行する。
 *
 * @author 佐藤隆之
 * @version $Id: ControllableQuery.java,v 1.6 2012-03-14 07:49:19 tayu Exp $
 */
public class ControllableQuery extends Query
{
  /** このクラスで発生しうるエラーを定義する列挙型。 */
  public enum Error {
    /** 処理される実行IDの制御ファイルが見つからなかったことを示す列挙値。 */
    QueryControlFileNotFound,

    /** 処理される実行IDの制御ファイルのロードに失敗したことを示す列挙値。 */
    FailToLoadQueryControlFile,

    /** 制御リソース中の属性の値が不正な場合。 */
    IllegalAttributeValue,
  }

  /** 処理する実行IDを制御するためのリソース要素ツリー。 */
  private Resource controlResource;

  /**
   * クエリ設定オブジェクトを引数にとるコンストラクタ。
   *
   * @param config クエリ設定オブジェクト。
   * @throws ReasonedRuntimeException 制御リソース・ファイルの読込に失敗した
   *   場合。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected ControllableQuery(QueryConfig config)
    throws ReasonedRuntimeException
  {
    super(config);
    controlResource = loadControlResource();
  }

  /**
   * クエリ設定オブジェクトとクエリ・トランザクション・オブジェクトを引数にとる
   * コンストラクタ。
   *
   * @param config クエリ設定オブジェクト。
   * @param tran クエリ・トランザクション・オブジェクト。
   * @throws ReasonedRuntimeException 制御リソース・ファイルの読込に失敗した
   *   場合。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected ControllableQuery(QueryConfig config, IQueryTransaction tran)
    throws ReasonedRuntimeException
  {
    super(config, tran);
    controlResource = loadControlResource();
  }

  /**
   * 処理される実行IDを制御するためのリソースをファイルからロードする。
   * <br>
   * クエリ設定の中に <tt>ts-query.query.execution.control.xmlfile</tt>キーに
   * 対する値が存在する場合はその値が示すパスのファイルから、存在しない場合は
   * クエリ設定の中からリソースを取得する。
   * <br>
   * なお、クエリ設定の<tt>ts-query.query.execution.control.xmlfile</tt>キーが
   * 示すパスは、環境設定の<tt>ts-query.environment.query.config.directory</tt>
   * キーが示すパスからの相対パスである。
   *
   * @return 処理される実行IDを制御するためのリソース・オブジェクト。
   */
  protected Resource loadControlResource()
  {
    QueryEnvironment env = QueryEnvironment.getInstance();
    File dir = env.getQueryConfigDirectory();

    final String KEY = "ts-query.query.execution.control.xmlfile";
    String xmlPath = getConfig().getResource().getFirstValue(KEY);

    if (xmlPath.isEmpty()) {
      return getConfig().getResource();
    }
    else {
      File xmlFile = new File(dir, xmlPath);
      if (! xmlFile.exists()) {
        throw new ReasonedRuntimeException(Error.QueryControlFileNotFound,
          "[path=" + xmlFile.getAbsolutePath() + "]");
      }

      try {
        XmlResource res = new XmlResource();
        res.setValidating(false);
        res.load(xmlFile.getCanonicalPath());
        return res;
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadQueryControlFile,
          "[path=" + xmlFile.getAbsolutePath() + "]", e);
      }
    }
  }

  /**
   * 処理する実行IDを制御するためのリソース要素ツリーを取得する。
   * 
   * @return 処理する実行IDを制御するためのリソース要素ツリー。
   */
  protected Resource getControlResource()
  {
    return this.controlResource;
  }
  
  /**
   * {@inheritDoc}
   */
  @Override
  protected void executeQuery(Map<String,Object> inputMap,
    QueryResult result) throws ReasonedException, ReasonedRuntimeException
  {
    final String KEY = "ts-query.query.execution.control.*";
    for (Resource res : getControlResource().getChildren(KEY)) {
      controlExecution(res, inputMap, result);
    }
  }

  /**
   * {@link IQueryExecution}の実行を制御する。
   *
   * @param res 制御リソース。
   * @param inputMap 入力パラメータ。
   * @param result クエリ結果オブジェクト。
   * @throws ReasonedException {@link IQueryExecution}オブジェクトの処理に失敗
   *  した場合。
   * @throws ReasonedRuntimeException クエリ制御や実行設定が不正だった場合。
   */
  protected void controlExecution(Resource res, Map<String,Object> inputMap,
    QueryResult result) throws ReasonedException, ReasonedRuntimeException
  {
    String elemName = res.getBaseElementName();
    if (elemName.equals("if")) {
      String param = res.getFirstAttribute("", "param");
      String from = res.getFirstAttribute("", "from");
      String condExists = res.getFirstAttribute("", "exists");
      String condEquals = res.getFirstAttribute("", "equals");
      String condNotEqual = res.getFirstAttribute("", "notEqual");
      String condContains = res.getFirstAttribute("", "contains");
      String condNotContain = res.getFirstAttribute("", "notContains");
      String delimiter = res.getFirstAttribute("", "delimiter");

      String value = getParamValue(param, from, inputMap, result);
      if (! checkExistsCondition(value, condExists)) return;
      if (! checkEqualsCondition(value, condEquals)) return;
      if (! checkNotEqualCondition(value, condNotEqual)) return;
      if (! checkContainsCondition(value, condContains, delimiter)) return;
      if (! checkNotContainCondition(value, condNotContain, delimiter)) return;

      for (Resource child : res.getChildren("*")) {
        controlExecution(child, inputMap, result);
      }
    }
    else if (elemName.equals("id")) {
      String execId = res.getFirstValue("");
      QueryExecutionConfig execCfg = new QueryExecutionConfig(execId);
      IQueryExecution execution = execCfg.create(getTransaction());
      execution.execute(inputMap, result);
    }
  }

  /**
   * 入力パラメータ又は結果データ・テーブルから指定された名前のパラメータ値を
   * 取得する。
   * <br>
   * 引数<tt>from</tt>が空文字列の場合は入力パラメータから、空文字列でない場合
   * は結果データ・テーブルからパラメータ値を取得する。
   * <br>
   * なお、パラメータが入力パラメータ・マップ又は結果データ・テーブルに複数値
   * （コレクション型または配列型）で登録されていた場合は最初の要素の値を返す。
   *
   * @param param パラメータ名。
   * @param from  結果データ・テーブルを識別する結果ID。
   * @param inputMap 入力パラメータ・マップ。
   * @param result クエリ結果オブジェクト。
   */
  protected String getParamValue(String param, String from,
    Map<String,Object> inputMap, QueryResult result)
  {
    Object val = null;
    if (! from.isEmpty()) {
      Table<String,Serializable> table = result.getResultTable(from);
      if (table != null && table.recordCount() > 0) {
        val = table.recordFirst().get(param);
      }
    }
    else {
      val = inputMap.get(param);
    }
    
    if (val == null) {
      return null;
    }
    else {
      final String KEY = "";
      TypedMap<String,Object> map = new TypedMap<String,Object>();
      map.put(KEY, val);
      return map.getString(KEY);
    }
  }

  /**
   * パラメータの存在チェックを行う。
   * <br>
   * 引数の<tt>condition</tt>の値が空文字列の場合は存在チェックを行わず、
   * <tt>true</tt>を返す。
   * 引数の<tt>condition</tt>の値が<tt>"true"</tt>の場合は、引数<tt>value</tt>
   * の値が非ヌルの場合は<tt>true</tt>（存在する）、ヌルの場合は<tt>false</tt>
   * （存在しない）を返す。
   * 引数の<tt>condition</tt>の値が<tt>"false"</tt>の場合は、引数<tt>value</tt>
   * の値が非ヌルの場合は<tt>false</tt>（存在する）、ヌルの場合は<tt>true</tt>
   * （存在しない）を返す。
   * 引数の<tt>condition</tt>の値が上記のいずれでもない場合は例外をスローする。
   *
   * @param value パラメータ値。
   * @param condition パラメータの存在条件。
   * @return パラメータの存在条件を満たす場合は<tt>true</tt>を返す。
   * @throws ReasonedRuntimeException パラメータの存在条件の値が不正な場合。
   */
  protected boolean checkExistsCondition(String value, String condition)
    throws ReasonedRuntimeException
  {
    if (condition.isEmpty()) {
      return true;
    }
    else if (condition.equals("true")) {
      return (value != null) ? true : false;
    }
    else if (condition.equals("false")) {
      return (value != null) ? false : true;
    }
    else {
      throw new ReasonedRuntimeException(Error.IllegalAttributeValue,
        "[exists=" + condition + "]");
    }
  }

  /**
   * パラメータの値が条件と等しいかどうかをチェックする。
   * <br>
   * 引数<tt>value</tt>の値が、引数<tt>condition</tt>の値と等しいかどうかを
   * 判定し、等しい場合は<tt>true</tt>を、等しくない場合は<tt>false</tt>を
   * 返す。
   *
   * @param value パラメータ値。
   * @param condition 条件値。
   * @return 等しい場合は<tt>true</tt>を返す。
   */
  protected boolean checkEqualsCondition(String value, String condition)
  {
    if (condition.isEmpty()) {
      return true;
    }
    else {
      return condition.equals(value) ? true : false;
    }
  }

  /** 
   * パラメータの値が条件と等しくないかどうかをチェックする。
   * <br>
   * 引数<tt>value</tt>の値が、引数<tt>condition</tt>の値と等しくないか
   * どうかを判定し、等しい場合は<tt>false</tt>を、等しくない場合は
   * <tt>true</tt>を返す。
   *
   * @param value パラメータ値。
   * @param condition 条件値。
   * @return 等しくない場合は<tt>true</tt>を返す。
   */
  protected boolean checkNotEqualCondition(String value, String condition)
  {
    if (condition.isEmpty()) {
      return true;
    }
    else {
      return condition.equals(value) ? false : true;
    }
  }

  /**
   * パラメータの値が条件に含まれるかどうかをチェックする。
   * <br>
   * 引数<tt>condition</tt>を引数<tt>delimiter</tt>で分割し、その配列に
   * 引数<tt>value</tt>の値が含まれる場合は<tt>true</tt>を、含まれない場合は
   * <tt>false</tt>を返す。
   *
   * @param value パラメータ値。
   * @param condition 条件値の連結文字列。
   * @param delimiter 条件値の区切り文字列。
   * @return 条件値に含まれる場合は<tt>true</tt>を返す。
   */
  protected boolean checkContainsCondition(String value, String condition,
    String delimiter)
  {
    if (condition.isEmpty()) {
      return true;
    }
    else if (delimiter.isEmpty()) {
      return condition.equals(value) ? true : false;
    }
    else {
      List<String> list = StringCutter.split(condition, delimiter);
      return list.contains(value) ? true : false;
    }
  }
  
  /**
   * パラメータの値が条件に含まれないかどうかをチェックする。
   * <br>
   * 引数<tt>condition</tt>を引数<tt>delimiter</tt>で分割し、その配列に
   * 引数<tt>value</tt>の値が含まれない場合は<tt>true</tt>を、含まれる場合は
   * <tt>false</tt>を返す。
   *
   * @param value パラメータ値。
   * @param condition 条件値の連結文字列。
   * @param delimiter 条件値の区切り文字列。
   * @return 条件値に含まれない場合は<tt>true</tt>を返す。
   */
  protected boolean checkNotContainCondition(String value,
    String condition, String delimiter)
  {
    if (condition.isEmpty()) {
      return true;
    }
    else if (delimiter.isEmpty()) {
      return condition.equals(value) ? false : true;
    }
    else {
      List<String> list = StringCutter.split(condition, delimiter);
      return list.contains(value) ? false : true;
    }
  }
}
