/*
 * BatchedQueryExecution class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.table.Table;
import java.io.Serializable;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;

/**
 * 一回の実行のおいて、同じ接続先に対する複数の照会／更新の実行を行う{@link
 * IQueryExecution}クラスの抽象クラス。
 *
 * @author 佐藤隆之
 * @version $Id: BatchedQueryExecution.java,v 1.5 2012-03-11 16:51:53 tayu Exp $
 */
public abstract class BatchedQueryExecution extends QueryExecution
{
  /** このクラスが使用する属性キーの列挙型。 */
  public enum Attribute {
    /** 結果テーブルと実行IDのマップを示す列挙値。 */
    ResultTableMap,

    /** 副実行IDを示す列挙型。 */
    SubExecutionId,
  }

  /**
   * 実行設定オブジェクトを引数にとるコンストラクタ。
   *
   * @param config {@link QueryExecutionConfig}オブジェクト。
   * @throws ReasonedException このオブジェクトで使用する{@link
   *   IQueryConnection}オブジェクトの作成に失敗した場合。
   * @throws ReasonedRuntimeException このオブジェクトが使用する接続設定が不正
   *   だった場合。
   * @throws AssertionError 引数がヌルの場合。
   */
  public BatchedQueryExecution(QueryExecutionConfig config)
    throws ReasonedException, ReasonedRuntimeException
  {
    super(config);
  }

  /**
   * 実行設定オブジェクトと{@link IQueryConnection}オブジェクトを引数にとる
   * コンストラクタ。
   *
   * @param config {@link QueryExecutionConfig}オブジェクト。
   * @param conn {@link IQueryConnection}オブジェクト。
   * @throws AssertionError 引数がヌルの場合。
   */
  public BatchedQueryExecution(QueryExecutionConfig config,
    IQueryConnection conn)
  {
    super(config, conn);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(Map<String,Object> inputMap, IQueryResult result)
    throws ReasonedException, ReasonedRuntimeException
  {
    assert (inputMap != null && result != null) :
      (inputMap == null) ? "@param:inputMap is null." :
      (result   == null) ? "@param:result is null." : "";

    long beginTm = System.currentTimeMillis();
    long limitTm = calcLimitTimeMillis(beginTm);
    int limitCnt = calcLimitFetchCount();

    QueryResult rslt = QueryResult.class.cast(result);
    if (rslt.getBeginTimeMillis() <= 0 || rslt.getBeginTimeMillis() > beginTm) {
      rslt.setBeginTimeMillis(beginTm);
    }

    HashMap<String,Table<String,Serializable>> tableMap =
      new HashMap<String,Table<String,Serializable>>();

    try {
      Iterator<String> itr = iterateSubExecutionIds();
      while (itr.hasNext()) {
        String subExecId = itr.next();
        QueryExecutionContent content = createContent();
        content.setLimitTimeMillis(limitTm);
        content.setLimitFetchCount(limitCnt);
        content.put(Attribute.ResultTableMap, tableMap);
        content.put(Attribute.SubExecutionId, subExecId);

        try {
          checkTimeout(content);
          prepareContent(content, inputMap);
          prepareResultTable(content);

          checkTimeout(content);
          executeContent(content);
        }
        finally {
          disposeContent(content, rslt);
        }
      }
    }
    catch (Exception e) {
      rslt.setException(e);
      throw rslt.getException();
    }
    finally {
      rslt.setEndTimeMillis(System.currentTimeMillis());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void prepareResultTable(QueryExecutionContent content)
  {
    String subExecId = content.getString(Attribute.SubExecutionId);

    Serializable obj = content.get(Attribute.ResultTableMap);
    @SuppressWarnings("unchecked")
    Map<String,Table<String,Serializable>> tableMap =
      (Map<String,Table<String,Serializable>>) obj;

    if (! content.hasResultTable()) {
      content.setResultTable(null);
      tableMap.remove(subExecId);
    }
    else {
      content.setResultTable(newResultTable(content));
      tableMap.put(subExecId, content.getResultTable());
    }
  }

  /**
   * {@inheritDoc}
   */
  protected void disposeContent(QueryExecutionContent content,
    QueryResult result)
  {
    final String KEY = "ts-query.execution.result.id";
    String rsltId = getConfig().getResource().getFirstValue(KEY);
    String subExecId = content.getString(Attribute.SubExecutionId);
    if (rsltId.equals(subExecId)) {
      super.disposeContent(content, result);
    }
  }

  /**
   * 副実行IDのイテレータを取得する。
   *
   * @return 副実行IDのイテレータ。
   */
  protected abstract Iterator<String> iterateSubExecutionIds();
}
