/*
 * TypedMapTest class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.ObjectInspector;
import ts.tester.function.print.*;
import java.math.*;
import java.util.*;
import java.text.*;

/**
 * <code>TypedMap</code>NX̋@\NXB
 *
 * @author V
 * @version $Id: TypedMapTest.java,v 1.5 2011-11-09 16:15:44 tayu Exp $
 */
public class TypedMapTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);
      
      run(TypedMapTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  protected void preTesting()
  {
    MSG("TypedMapNX̋@\sB");
  }

  public void constructor()
  {
    MSG("ftHgERXgN^̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
  }

  public void containsKey()
  {
    MSG("}bvɃL[݂邩ǂmF郁\bh̊mF");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    FALSE(map.containsKey(123));
    FALSE(map.containsKey(456));
    FALSE(map.containsKey(null));

    NULL(map.put(123, ""));
    EQUAL(map.size(), 1);
    FALSE(map.isEmpty());
    TRUE(map.containsKey(123));
    FALSE(map.containsKey(456));
    FALSE(map.containsKey(null));

    NULL(map.put(456, null));
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    TRUE(map.containsKey(123));
    TRUE(map.containsKey(456));
    FALSE(map.containsKey(null));

    EQUAL(map.put(123, ""), "");
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    TRUE(map.containsKey(123));
    TRUE(map.containsKey(456));
    FALSE(map.containsKey(null));

    NULL(map.put(456, null));
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    TRUE(map.containsKey(123));
    TRUE(map.containsKey(456));
    FALSE(map.containsKey(null));

    EQUAL(map.remove(123), "");
    EQUAL(map.size(), 1);
    FALSE(map.isEmpty());
    FALSE(map.containsKey(123));
    TRUE(map.containsKey(456));
    FALSE(map.containsKey(null));

    NULL(map.remove(456));
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    FALSE(map.containsKey(123));
    FALSE(map.containsKey(456));
    FALSE(map.containsKey(null));
  }

  public void containsValue()
  {
    MSG("}bvɒlIuWFNg݂邩ǂmF郁\bh̊mF");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    FALSE(map.containsValue(""));
    FALSE(map.containsValue(""));
    FALSE(map.containsValue(null));

    NULL(map.put(123, ""));
    EQUAL(map.size(), 1);
    FALSE(map.isEmpty());
    TRUE(map.containsValue(""));
    FALSE(map.containsValue(""));
    FALSE(map.containsValue(null));

    NULL(map.put(456, null));
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    TRUE(map.containsValue(""));
    FALSE(map.containsValue(""));
    TRUE(map.containsValue(null));

    EQUAL(map.put(123, ""), "");
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    FALSE(map.containsValue(""));
    TRUE(map.containsValue(""));
    TRUE(map.containsValue(null));

    NULL(map.put(456, null));
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    FALSE(map.containsValue(""));
    TRUE(map.containsValue(""));
    TRUE(map.containsValue(null));

    EQUAL(map.remove(123), "");
    EQUAL(map.size(), 1);
    FALSE(map.isEmpty());
    FALSE(map.containsValue(""));
    FALSE(map.containsValue(""));
    TRUE(map.containsValue(null));

    NULL(map.remove(456));
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    FALSE(map.containsValue(""));
    FALSE(map.containsValue(""));
    FALSE(map.containsValue(null));
  }

  public void get()
  {
    MSG("w肳ꂽL[ɑ΂lIuWFNg擾郁\bh̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    NULL(map.get(123));
    NULL(map.get(456));
    NULL(map.get(null));

    NULL(map.put(123, ""));
    EQUAL(map.size(), 1);
    FALSE(map.isEmpty());
    EQUAL(map.get(123), "");
    NULL(map.get(456));
    NULL(map.get(null));

    NULL(map.put(456, null));
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    EQUAL(map.get(123), "");
    NULL(map.get(456));
    NULL(map.get(null));

    EQUAL(map.put(123, ""), "");
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    EQUAL(map.get(123), "");
    NULL(map.get(456));
    NULL(map.get(null));

    NULL(map.put(456, null));
    EQUAL(map.size(), 2);
    FALSE(map.isEmpty());
    EQUAL(map.get(123), "");
    NULL(map.get(456));
    NULL(map.get(null));

    EQUAL(map.remove(123), "");
    EQUAL(map.size(), 1);
    FALSE(map.isEmpty());
    NULL(map.get(123));
    NULL(map.get(456));
    NULL(map.get(null));

    NULL(map.remove(456));
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    NULL(map.get(123));
    NULL(map.get(456));
    NULL(map.get(null));
  }

  public void putAll()
  {
    MSG("̃}bv̑SĂ̗vfǉ郁\bh̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    FALSE(map.containsKey(123));
    FALSE(map.containsKey(456));
    FALSE(map.containsKey(null));
    NULL(map.get(123));
    NULL(map.get(456));
    NULL(map.get(null));

    HashMap<Integer,String> m1 = new HashMap<Integer,String>();
    NULL(m1.put(123, "`abcdef"));
    NULL(m1.put(456, null));
    NULL(m1.put(null, "ghijklm"));

    map.putAll(m1);
    EQUAL(map.size(), 3);
    FALSE(map.isEmpty());
    TRUE(map.containsKey(123));
    TRUE(map.containsKey(456));
    TRUE(map.containsKey(null));
    EQUAL(map.get(123), "`abcdef");
    EQUAL(map.get(456), null);
    EQUAL(map.get(null), "ghijklm");
    EQUAL(map, m1);
  }

  public void clear()
  {
    MSG("}bvɂ郁\bh̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());

    map.put(123, "");
    map.put(456, null);
    map.put(null, "");
    EQUAL(map.size(), 3);
    FALSE(map.isEmpty());
    TRUE(map.containsKey(123));
    TRUE(map.containsKey(456));
    TRUE(map.containsKey(null));
    EQUAL(map.get(123), "");
    NULL(map.get(456));
    EQUAL(map.get(null), "");

    map.clear();
    EQUAL(map.size(), 0);
    TRUE(map.isEmpty());
    FALSE(map.containsKey(123));
    FALSE(map.containsKey(456));
    FALSE(map.containsKey(null));
    NULL(map.get(123));
    NULL(map.get(456));
    NULL(map.get(null));
  }

  public void keySet()
  {
    MSG("}bvɊi[ĂL[̏W擾郁\bh̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    EQUAL(map.keySet().size(), 0);

    map.put(123, "");
    map.put(456, null);
    map.put(null, "");
    EQUAL(map.size(), 3);

    Set<Integer> keySet0 = map.keySet();
    EQUAL(keySet0.size(), 3);
    List<Integer> lst0 = new LinkedList<Integer>();
    lst0.add(123);
    lst0.add(456);
    lst0.add(null);
    EQUAL(keySet0, lst0);

    map.clear();
    Set<Integer> keySet1 = map.keySet();
    EQUAL(keySet1.size(), 0);
  }

  public void values()
  {
    MSG("}bvɊi[ĂlIuWFNg̃RNV擾郁\bh̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    EQUAL(map.values().size(), 0);

    map.put(123, "");
    map.put(456, null);
    map.put(null, "");
    EQUAL(map.size(), 3);

    Collection<String> values0 = map.values();
    EQUAL(values0.size(), 3);
    List<String> lst0 = new LinkedList<String>();
    lst0.add("");
    lst0.add("");
    lst0.add(null);
    EQUAL(values0, lst0);

    map.clear();
    Collection<String> values1 = map.values();
    EQUAL(values1.size(), 0);
  }

  public void entrySet()
  {
    MSG("}bvɊi[ĂL[ƒl̑g̏W擾郁\bhB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.size(), 0);
    EQUAL(map.entrySet().size(), 0);

    map.put(123, "");
    map.put(456, null);
    map.put(null, "");
    EQUAL(map.size(), 3);

    Set<Map.Entry<Integer,String>> entrySet0 = map.entrySet();
    List<Map.Entry<Integer,String>> lst0 =
      new LinkedList<Map.Entry<Integer,String>>();
    Map<Integer,String> map0 = new HashMap<Integer,String>();
    map0.put(123,"");
    map0.put(456,null);
    map0.put(null,"");
    EQUAL(entrySet0, map0.entrySet());

    map.clear();
    EQUAL(map.entrySet().size(), 0);
  }

  public void equals_Object()
  {
    MSG("̃IuWFNgƓeǂmF郁\bh̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    HashMap<Integer,String> m0 = new HashMap<Integer,String>();
    HashMap<Integer,String> m1 = new HashMap<Integer,String>();
    TRUE(map.equals(m0));
    TRUE(m0.equals(map));
    TRUE(map.equals(m1));
    TRUE(m1.equals(map));

    m1.put(123, "");
    m1.put(null, "");
    m1.put(456, null);
    TRUE(map.equals(m0));
    TRUE(m0.equals(map));
    FALSE(map.equals(m1));
    FALSE(m1.equals(map));

    map.put(123, "");
    map.put(456, null);
    map.put(null, "");
    EQUAL(map.size(), 3);
    FALSE(map.equals(m0));
    FALSE(m0.equals(map));
    TRUE(map.equals(m1));
    TRUE(m1.equals(map));

    map.put(789, "");
    EQUAL(map.size(), 4);
    FALSE(map.equals(m0));
    FALSE(m0.equals(map));
    FALSE(map.equals(m1));
    FALSE(m1.equals(map));

    map.remove(456);
    EQUAL(map.size(), 3);
    FALSE(map.equals(m0));
    FALSE(m0.equals(map));
    FALSE(map.equals(m1));
    FALSE(m1.equals(map));

    map.remove(789);
    EQUAL(map.size(), 2);
    FALSE(map.equals(m0));
    FALSE(m0.equals(map));
    FALSE(map.equals(m1));
    FALSE(m1.equals(map));

    map.put(456, null);
    EQUAL(map.size(), 3);
    FALSE(map.equals(m0));
    FALSE(m0.equals(map));
    TRUE(map.equals(m1));
    TRUE(m1.equals(map));

    map.put(456, "AAA");
    EQUAL(map.size(), 3);
    FALSE(map.equals(m0));
    FALSE(m0.equals(map));
    FALSE(map.equals(m1));
    FALSE(m1.equals(map));

    map.clear();
    TRUE(map.equals(m0));
    TRUE(m0.equals(map));
    FALSE(map.equals(m1));
    FALSE(m1.equals(map));
  }

  public void equals_Object_null()
  {
    MSG("k̏ꍇB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    FALSE(map.equals(null));
  }
  
  public void hashCode_Void()
  {
    MSG("nbVER[h擾郁\bh̊mFB");

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.hashCode(), new HashMap<Integer,String>().hashCode());

    Map<Integer,String> m0 = new HashMap<Integer,String>();
    map.put(123, "");
    NOTEQUAL(map.hashCode(), m0.hashCode());

    Map<Integer,String> m1 = new TreeMap<Integer,String>();
    m1.put(123, "");
    EQUAL(map.hashCode(), m1.hashCode());

    m1.put(123, "");
    NOTEQUAL(map.hashCode(), m1.hashCode());
  }

  public void limit_long()
  {
    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.limit(0L, 0L, 0L), 0L);
    EQUAL(map.limit(100L, 0L, 200L), 100L);

    try {
      NG("" + map.limit(201L, 0L, 200L));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }

    try {
      NG("" + map.limit(-1L, 0L, 200L));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void limit_float()
  {
    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.limit(0.0f, 0.0f, 0.0f), 0.0f);
    EQUAL(map.limit(1.0f, 0.0f, 2.0f), 1.0f);

    try {
      NG("" + map.limit(2.1f, 0.0f, 2.0f));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }

    try {
      NG("" + map.limit(-0.1f, 0.0f, 2.0f));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void limit_double()
  {
    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.limit(0.0, 0.0, 0.0), 0.0);
    EQUAL(map.limit(1.0, 0.0, 2.0), 1.0);

    try {
      NG("" + map.limit(2.1, 0.0, 2.0));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try {
      NG("" + map.limit(-0.1, 0.0, 2.0));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void limit_BigDecimal()
  {
    BigDecimal min = new BigDecimal(-123456789L);
    BigDecimal max = new BigDecimal(123456789L);

    TypedMap<Integer,String> map = new TypedMap<Integer,String>();
    EQUAL(map.limit(new BigDecimal(0L), min, max), new BigDecimal(0L));

    try {
      NG("" + map.limit(new BigDecimal(123456790L), min, max));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
      
    try { 
      NG("" + map.limit(new BigDecimal(-123456790L), min, max));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void getBoolean()
  {
    MSG("booleanl擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    FALSE(map.getBoolean("A"));

    map.put("A", Boolean.TRUE);
    TRUE(map.getBoolean("A"));

    map.put("A", Boolean.FALSE);
    FALSE(map.getBoolean("A"));

    map.put("A", new Integer(1));
    FALSE(map.getBoolean("A"));

    map.put("A", new Integer(0));
    FALSE(map.getBoolean("A"));

    map.put("A", "1");
    FALSE(map.getBoolean("A"));

    map.put("A", "0");
    FALSE(map.getBoolean("A"));

    map.put("A", "true");
    TRUE(map.getBoolean("A"));

    map.put("A", "false");
    FALSE(map.getBoolean("A"));

    map.put("A", "TRUE");
    FALSE(map.getBoolean("A"));

    map.put("A", "FALSE");
    FALSE(map.getBoolean("A"));

    map.put("A", new Object());
    FALSE(map.getBoolean("A"));

    map.put("A", null);
    FALSE(map.getBoolean("A"));
  }

  public void addAndRemoveTrueObject()
  {
    MSG("trueƔ肷IuWFNgݒ肷郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    FALSE(map.getBoolean("A"));

    map.put("A", Boolean.TRUE);
    TRUE(map.getBoolean("A"));
    map.removeTrueObject(Boolean.TRUE);
    FALSE(map.getBoolean("A"));

    map.put("A", "true");
    TRUE(map.getBoolean("A"));
    map.removeTrueObject("true");
    FALSE(map.getBoolean("A"));

    map.put("A", "false");
    FALSE(map.getBoolean("A"));
    map.addTrueObject("false");
    TRUE(map.getBoolean("A"));

    map.put("B", "1");
    TRUE (map.getBoolean("A"));
    FALSE(map.getBoolean("B"));
    map.addTrueObject("1");
    TRUE (map.getBoolean("A"));
    TRUE (map.getBoolean("B"));

    map.removeTrueObject("1");
    map.removeTrueObject("false");
    FALSE(map.getBoolean("A"));
    FALSE(map.getBoolean("B"));

    map.put("C", null);
    FALSE(map.getBoolean("C"));
    map.addTrueObject(null);
    TRUE (map.getBoolean("C"));
    map.removeTrueObject(null);
    FALSE(map.getBoolean("C"));
  }

  public void getInteger()
  {
    MSG("intl擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    map.put("A", (byte)123);
    map.put("B", (short) 456);
    map.put("C", 789);
    map.put("D", 987L);
    map.put("E", 999.999f);
    map.put("F", 111.111111);
    map.put("G", new BigDecimal(555.666));
    map.put("H", new BigInteger("777777"));
    map.put("I", "");
    map.put("J", "123");
    map.put("K", "0000456");
    map.put("L", "-000987");
    map.put("M", "    654 ");
    map.put("N", "+  0321");
    EQUAL(map.getInteger("A"), 123);
    EQUAL(map.getInteger("B"), 456);
    EQUAL(map.getInteger("C"), 789);
    EQUAL(map.getInteger("D"), 987);
    EQUAL(map.getInteger("E"), 999);
    EQUAL(map.getInteger("F"), 111);
    EQUAL(map.getInteger("G"), 555);
    EQUAL(map.getInteger("H"), 777777);
    EQUAL(map.getInteger("I"), 0);
    EQUAL(map.getInteger("J"), 123);
    EQUAL(map.getInteger("K"), 456);
    EQUAL(map.getInteger("L"), -987);
    EQUAL(map.getInteger("M"), 654);
    EQUAL(map.getInteger("N"), 321);
  }

  public void setInteger_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    map.put("A", null);
    EQUAL(map.getInteger("A"), 0);
    EQUAL(map.getInteger("B"), 0);
  }

  public void getInteger_fromTooLargeOrSmallValues()
  {
    MSG("̌^̑傫閔͏l̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", (long)Integer.MAX_VALUE + 1L);
    map.put("B", ((long)Integer.MIN_VALUE - 1L));
    map.put("C", Long.MAX_VALUE);
    map.put("D", Long.MIN_VALUE);
    map.put("E", (float)Integer.MAX_VALUE + 1000.0f);
    map.put("F", (float)Integer.MIN_VALUE - 1000.0f);
    map.put("G", Float.MAX_VALUE);
    map.put("H", - Float.MAX_VALUE);
    map.put("I", (double)Integer.MAX_VALUE + 1.0);
    map.put("J", (double)Integer.MIN_VALUE - 1.0);
    map.put("K", Double.MAX_VALUE);
    map.put("L", - Double.MAX_VALUE);
    map.put("M", "987654321987654321");
    map.put("N", "-987654321987654321");

    try { 
      NG("" + map.getInteger("A"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("B"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("C"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("D"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("E"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("F"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("G"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("H"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("I"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("J"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("K"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("L"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("M"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("N"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void getInteger_notNumber()
  {
    MSG("llɕϊłȂꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", "abc");
    map.put("B", new Object());
    map.put("C", new ArrayList(0));

    try { 
      NG("" + map.getInteger("A"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("B"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try { 
      NG("" + map.getInteger("C"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }

    map.put("D", "123456789");
    map.put("E", "" + Integer.MAX_VALUE);
    map.put("F", "" + (Integer.MAX_VALUE - 1));
    map.put("G", "1234.56789");
    EQUAL(map.getInteger("D"), 123456789);
    EQUAL(map.getInteger("E"), 2147483647);
    EQUAL(map.getInteger("F"), 2147483646);
    EQUAL(map.getInteger("G"), 1234);
  }

  public void getLong()
  {
    MSG("longl擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    map.put("A", (byte)123);
    map.put("B", (short) 456);
    map.put("C", 789);
    map.put("D", 987L);
    map.put("E", 999.999f);
    map.put("F", 111.111111);
    map.put("G", new BigDecimal(555.666));
    map.put("H", new BigInteger("777777"));
    map.put("I", "9876");
    map.put("J", "-  5432");
    map.put("K", "0000011");
    EQUAL(map.getLong("A"), 123L);
    EQUAL(map.getLong("B"), 456L);
    EQUAL(map.getLong("C"), 789L);
    EQUAL(map.getLong("D"), 987L);
    EQUAL(map.getLong("E"), 999L);
    EQUAL(map.getLong("F"), 111L);
    EQUAL(map.getLong("G"), 555L);
    EQUAL(map.getLong("H"), 777777L);
    EQUAL(map.getLong("I"), 9876L);
    EQUAL(map.getLong("J"), -5432L);
    EQUAL(map.getLong("K"), 11L);
  }

  public void getLong_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getLong("A"), 0L);
    EQUAL(map.getLong("B"), 0L);
  }

  public void getLong_fromTooLargeOrSmallValues()
  {
    MSG("̌^̑傫閔͏l̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    
    double v0 = - ((double) Long.MAX_VALUE + Math.pow(10,4));
    BigDecimal v1 = new BigDecimal(Long.MAX_VALUE).add(new BigDecimal(1L));

    map.put("A", v0);
    map.put("B", v1);

    try {
      NG("" + map.getLong("A"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try {
      NG("" + map.getLong("B"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void getLong_notNumber()
  {
    MSG("llɕϊłȂꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", new Object());
    map.put("B", "abc");

    try {
      NG("" + map.getLong("A"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    try {
      NG("" + map.getLong("B"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }

    map.put("C", "" + Long.MAX_VALUE);
    map.put("D", "" + (Long.MAX_VALUE - 1L));
    map.put("E", "" + (Long.MAX_VALUE - 2L));
    EQUAL(map.getLong("C"), 9223372036854775807L);
    EQUAL(map.getLong("D"), 9223372036854775806L);
    EQUAL(map.getLong("E"), 9223372036854775805L);
  }

  public void getFloat()
  {
    MSG("floatl擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", (byte)123);
    map.put("B", (short) 456);
    map.put("C", 789);
    map.put("D", 987L);
    map.put("E", 999.999f);
    map.put("F", 111.111111);
    map.put("G", new BigDecimal(555.666));
    map.put("H", new BigInteger("777777"));
    map.put("I", "12.345");
    map.put("J", "- 000.067");
    map.put("K", "+0008.912");

    EQUAL(map.getFloat("A"), 123.0f);
    EQUAL(map.getFloat("B"), 456.0f);
    EQUAL(map.getFloat("C"), 789.0f);
    EQUAL(map.getFloat("D"), 987.0f);
    EQUAL(map.getFloat("E"), 999.999f);
    EQUAL(map.getFloat("F"), 111.111111f);
    EQUAL(map.getFloat("G"), 555.666f);
    EQUAL(map.getFloat("H"), 777777.0f);
    EQUAL(map.getFloat("I"), 12.345f);
    EQUAL(map.getFloat("J"), -0.067f);
    EQUAL(map.getFloat("K"),  8.912f);
  }

  public void getFloat_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getFloat("A"), 0.0f);
    EQUAL(map.getFloat("B"), 0.0f);
  }

  public void getFloat_fromTooLargeOrSmallValues()
  {
    MSG("̌^̑傫閔͏l̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    double v0 = - (double) Float.MAX_VALUE * 10.0;
    BigDecimal v1 = new BigDecimal(Float.MAX_VALUE).multiply(
      new BigDecimal(10.0));

    map.put("A", v0);
    map.put("B", v1);

    EQUAL(map.getFloat("A"), Float.NEGATIVE_INFINITY);
    EQUAL(map.getFloat("B"), Float.POSITIVE_INFINITY);
  }

  public void getFloat_notNumber()
  {
    MSG("llɕϊłȂꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", new Object());
    map.put("B", "abc");

    try {
      NG("" + map.getFloat("A"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }

    try {
      NG("" + map.getFloat("B"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void getDouble()
  {
    MSG("doublel擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", (byte)123);
    map.put("B", (short) 456);
    map.put("C", 789);
    map.put("D", 987L);
    map.put("E", 999.999f);
    map.put("F", 111.111111);
    map.put("G", new BigDecimal(555.666));
    map.put("H", new BigInteger("777777"));
    map.put("I", "9.876");
    map.put("J", "- 000.0543");
    map.put("K", "+ 002.198 ");

    EQUAL(map.getDouble("A"), 123.0);
    EQUAL(map.getDouble("B"), 456.0);
    EQUAL(map.getDouble("C"), 789.0);
    EQUAL(map.getDouble("D"), 987.0);
    EQUAL(map.getDouble("E"), (double) 999.999f);
    EQUAL(map.getDouble("F"), 111.111111);
    EQUAL(map.getDouble("G"), 555.666);
    EQUAL(map.getDouble("H"), 777777.0);
    EQUAL(map.getDouble("I"), 9.876);
    EQUAL(map.getDouble("J"), -0.0543);
    EQUAL(map.getDouble("K"), 2.198);
  }

  public void getDouble_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getDouble("A"), 0.0);
    EQUAL(map.getDouble("B"), 0.0);
  }

  public void getDouble_fromTooLargeOrSmallValues()
  {
    MSG("̌^̑傫閔͏l̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    BigDecimal v0 = new BigDecimal(Double.MAX_VALUE).multiply(
      new BigDecimal(-10.0));
    BigDecimal v1 = new BigDecimal(Double.MAX_VALUE).multiply(
      new BigDecimal(10.0));

    map.put("A", v0);
    map.put("B", v1);

    EQUAL(map.getDouble("A"), Double.NEGATIVE_INFINITY);
    EQUAL(map.getDouble("B"), Double.POSITIVE_INFINITY);
  }

  public void getDouble_notNumber()
  {
    MSG("llɕϊłȂꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", new Object());
    map.put("B", "abc");

    try {
      NG("" + map.getDouble("A"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }

    try {
      NG("" + map.getDouble("B"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void getBigDecimal()
  {
    MSG("BigDecimall擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", (byte)123);
    map.put("B", (short) 456);
    map.put("C", 789);
    map.put("D", 987L);
    map.put("E", 999.999f);
    map.put("F", 111.111111);
    map.put("G", new BigDecimal(555.666));
    map.put("H", new BigInteger("777777"));
    map.put("I", "12345");
    map.put("J", "-    12.345");
    map.put("K", "+ 000123.45 ");

    EQUAL(map.getBigDecimal("A"), new BigDecimal(123));
    EQUAL(map.getBigDecimal("B"), new BigDecimal(456));
    EQUAL(map.getBigDecimal("C"), new BigDecimal(789));
    EQUAL(map.getBigDecimal("D"), new BigDecimal(987));
    EQUAL(map.getBigDecimal("E"), new BigDecimal(999.999f));
    EQUAL(map.getBigDecimal("F"), new BigDecimal(111.111111));
    EQUAL(map.getBigDecimal("G"), new BigDecimal(555.666));
    EQUAL(map.getBigDecimal("H"), new BigDecimal("777777"));
    EQUAL(map.getBigDecimal("I"), new BigDecimal("12345"));
    EQUAL(map.getBigDecimal("J"), new BigDecimal("-12.345"));
    EQUAL(map.getBigDecimal("K"), new BigDecimal("123.45"));
  }

  public void getBigDecimal_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getBigDecimal("A"), new BigDecimal(0));
    EQUAL(map.getBigDecimal("B"), new BigDecimal(0));
  }

  public void getBigDecimal_notNumber()
  {
    MSG("llɕϊłȂꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", new Object());
    map.put("B", "abc");

    try {
      NG("" + map.getBigDecimal("A"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
    
    try {
      NG("" + map.getBigDecimal("B"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void getString()
  {
    MSG("IuWFNg擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", (byte)123);
    map.put("B", (short) 456);
    map.put("C", 789);
    map.put("D", 987L);
    map.put("E", 999.999f);
    map.put("F", 111.111111);
    map.put("G", new BigDecimal(555.666));
    map.put("H", new BigInteger("777777"));
    map.put("I", "");
    map.put("J", "");

    EQUAL(map.getString("A"), "123");
    EQUAL(map.getString("B"), "456");
    EQUAL(map.getString("C"), "789");
    EQUAL(map.getString("D"), "987");
    EQUAL(map.getString("E"), "999.999");
    EQUAL(map.getString("F"), "111.111111");
    TRUE(map.getString("G").startsWith("555.666"));
    EQUAL(map.getString("H"), "777777");
    EQUAL(map.getString("I"), "");
    EQUAL(map.getString("J"), "");
  }

  public void getString_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getString("A"), "");
    EQUAL(map.getString("B"), "");
  }

  public void getString_DateTime()
  {
    MSG("f[^̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    DateTime dttm0 = new DateTime(2011,6,5,1,22,33);
    DateTime dttm1 = new DateTime(2011,2,6,23,45,11);
    DateTime dttm2 = new DateTime(2011,1,4,11,11,11);
    DateTime dttm3 = new DateTime(2011,10,22,11,22,33);
    DateTime dttm4 = new DateTime(2011,11,22,22,33,44);

    map.put("A", dttm0);
    map.put("B", dttm1.getDate());
    map.put("C", dttm2.getSQLDate());
    map.put("D", dttm3.getSQLTime());
    map.put("E", dttm4.getSQLTimestamp());

    EQUAL(map.getString("A"), "2011-06-05 01:22:33");
    EQUAL(map.getString("B"), "2011-02-06 23:45:11");
    EQUAL(map.getString("C"), "2011-01-04 11:11:11");
    EQUAL(map.getString("D"), "2011-10-22 11:22:33");
    EQUAL(map.getString("E"), "2011-11-22 22:33:44");
  }

  public void getNumberString()
  {
    MSG("lϊ擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", (byte)123);
    map.put("B", (short) 456);
    map.put("C", 7890);
    map.put("D", 9876L);
    map.put("E", 9999.999f);
    map.put("F", 1111111.111111);
    map.put("G", new BigDecimal(444555.666));
    map.put("H", new BigInteger("777777"));

    EQUAL(map.getNumberString("A", "$#,###"), "$123");
    EQUAL(map.getNumberString("B", "$#,###"), "$456");
    EQUAL(map.getNumberString("C", "$#,###"), "$7,890");
    EQUAL(map.getNumberString("D", "$#,###"), "$9,876");
    EQUAL(map.getNumberString("E", "$#,###"), "$10,000");
    EQUAL(map.getNumberString("F", "$#,###"), "$1,111,111");
    EQUAL(map.getNumberString("G", "$#,###"), "$444,556");
    EQUAL(map.getNumberString("H", "$#,###"), "$777,777");
  }

  public void getNumberString_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getNumberString("A", "#,###"), "");
    EQUAL(map.getNumberString("B", "#,###"), "");
  }

  public void getNumberString_notNumber()
  {
    MSG("llɕϊłȂꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", new Object());
    map.put("B", "abc");

    try {
      NG("" + map.getNumberString("A", "#,###"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }

    try {
      NG("" + map.getNumberString("B", "#,###"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }

  public void getDateTimeString()
  {
    MSG("ϊ擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    DateTime dttm0 = new DateTime(2011,6,5,1,22,33);
    DateTime dttm1 = new DateTime(2011,2,6,23,45,11);
    DateTime dttm2 = new DateTime(2011,1,4,11,11,11);
    DateTime dttm3 = new DateTime(2011,10,22,11,22,33);
    DateTime dttm4 = new DateTime(2011,11,22,22,33,44);

    map.put("A", dttm0);
    map.put("B", dttm1.getDate());
    map.put("C", dttm2.getSQLDate());
    map.put("D", dttm3.getSQLTime());
    map.put("E", dttm4.getSQLTimestamp());
    map.put("F", "2011-11-22 01:23:45");
    map.put("G", "2011-11-22");
    map.put("H", "01:23:45");

    EQUAL(map.getDateTimeString("A", "yyMMdd-HHmmss"), "110605-012233");
    EQUAL(map.getDateTimeString("B", "yyMMdd-HHmmss"), "110206-234511");
    EQUAL(map.getDateTimeString("C", "yyMMdd-HHmmss"), "110104-111111");
    EQUAL(map.getDateTimeString("D", "yyMMdd-HHmmss"), "111022-112233");
    EQUAL(map.getDateTimeString("E", "yyMMdd-HHmmss"), "111122-223344");
    EQUAL(map.getDateTimeString("F", "yyMMdd-HHmmss"), "111122-012345");

    try {
      NG("" + map.getDateTimeString("G", "yyMMdd-HHmmss"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastDateTime);
      OK(e.toString());
    }

    try {
      NG("" + map.getDateTimeString("H", "yyMMdd-HHmmss"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastDateTime);
      OK(e.toString());
    }
  }

  public void getDateTimeString_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getDateTimeString("A", "yyyy-MM-dd HH:mm:ss"), "");
    EQUAL(map.getDateTimeString("B", "yyyy-MM-dd HH:mm:ss"), "");
  }

  public void getDateTimeString_notDate()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", new Object());
    map.put("B", "abc");

    try {
      NG("" + map.getDateTimeString("A", "yyyy-MM-dd HH:mm:ss"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastDateTime);
      OK(e.toString());
    }

    try {
      NG("" + map.getDateTimeString("B", "yyyy-MM-dd HH:mm:ss"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastDateTime);
      OK(e.toString());
    }
  }

  public void getDateTime()
  {
    MSG("IuWFNg擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    DateTime dttm0 = new DateTime(2011,6,5,1,22,33);
    DateTime dttm1 = new DateTime(2011,2,6,23,45,11);
    DateTime dttm2 = new DateTime(2011,1,4,11,11,11);
    DateTime dttm3 = new DateTime(2011,10,22,11,22,33);
    DateTime dttm4 = new DateTime(2011,11,22,22,33,44);

    map.put("A", dttm0);
    map.put("B", dttm1.getDate());
    map.put("C", dttm2.getSQLDate());
    map.put("D", dttm3.getSQLTime());
    map.put("E", dttm4.getSQLTimestamp());
    map.put("F", "2011-11-22 01:23:45");
    map.put("G", "2011-11-22");
    map.put("H", "01:23:45");

    EQUAL(map.getDateTime("A"), new DateTime(2011,6,5,1,22,33));
    EQUAL(map.getDateTime("B"), new DateTime(2011,2,6,23,45,11));
    EQUAL(map.getDateTime("C"), new DateTime(2011,1,4,11,11,11));
    EQUAL(map.getDateTime("D"), new DateTime(2011,10,22,11,22,33));
    EQUAL(map.getDateTime("E"), new DateTime(2011,11,22,22,33,44));
    EQUAL(map.getDateTime("F"), new DateTime(2011,11,22,1,23,45));

    try {
      NG("" + map.getDateTime("G"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastDateTime);
      OK(e.toString());
    }

    try {
      NG("" + map.getDateTime("H"));
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastDateTime);
      OK(e.toString());
    }
  }

  public void getDateTime_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getDateTime("A"), new DateTime(0L));
    EQUAL(map.getDateTime("B"), new DateTime(0L));
  }

  public void setDateTimeFormat_String()
  {
    MSG("ݒ肷郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    DateTime dttm = new DateTime(2010,10,22,10,20,30);
    map.put("A", dttm);
    EQUAL(map.getString("A"), "2010-10-22 10:20:30");

    map.setDateTimeFormat("yyyy/MM/dd HH=mm=ss");
    EQUAL(map.getString("A"), "2010/10/22 10=20=30");
  }

  public void setDateTimeFormat_String_null()
  {
    MSG("k̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    try {
      map.setDateTimeFormat((String) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void setDateTimeFormat_DateFormat()
  {
    MSG("IuWFNgݒ肷郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    DateTime dttm = new DateTime(2010,10,22,10,20,30);
    map.put("A", dttm);
    EQUAL(map.getString("A"), "2010-10-22 10:20:30");

    DateFormat df = new SimpleDateFormat("yyyy/MM/dd HH=mm=ss");
    map.setDateTimeFormat(df);
    EQUAL(map.getString("A"), "2010/10/22 10=20=30");
  }

  public void setDateTimeFormat_DateFormat_null()
  {
    MSG("k̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    try {
      map.setDateTimeFormat((DateFormat) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }

  }

  public void getList()
  {
    MSG("XgEIuWFNg擾郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    ArrayList<Integer> list0 = new ArrayList<Integer>();

    LinkedList<Integer> list1 = new LinkedList<Integer>();

    ArrayList<Integer> list2 = new ArrayList<Integer>();
    list2.add(3);
    list2.add(2);

    LinkedList<Integer> list3 = new LinkedList<Integer>();
    list3.add(8);
    list3.add(9);
    list3.add(3);

    TreeSet<Integer> coll0 = new TreeSet<Integer>();

    TreeSet<Integer> coll1 = new TreeSet<Integer>();
    coll1.add(5);
    coll1.add(8);
    coll1.add(3);

    map.put("A", list0);
    map.put("B", list1);
    map.put("C", list2);
    map.put("D", list3);
    map.put("E", coll0);
    map.put("F", coll1);

    List<? extends Object> list;
    list = map.getList("A");
    EQUAL(list.size(), 0);

    list = map.getList("B");
    EQUAL(list.size(), 0);

    list = map.getList("C");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), 3);
    EQUAL(list.get(1), 2);

    list = map.getList("D");
    EQUAL(list.size(), 3);
    EQUAL(list.get(0), 8);
    EQUAL(list.get(1), 9);
    EQUAL(list.get(2), 3);

    list = map.getList("E");
    EQUAL(list.size(), 0);

    list = map.getList("F");
    EQUAL(list.size(), 3);
    EQUAL(list.get(0), 3);
    EQUAL(list.get(1), 5);
    EQUAL(list.get(2), 8);
  }

  public void getList_array()
  {
    MSG("lz̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    String[] arr0 = new String[0];
    byte[] arr1 = new byte[0];
    short[] arr2 = new short[0];
    char[] arr3 = new char[0];
    int[] arr4 = new int[0];
    long[] arr5 = new long[0];
    float[] arr6 = new float[0];
    double[] arr7 = new double[0];
    boolean[] arr8 = new boolean[0];

    map.put("arr0", arr0);
    map.put("arr1", arr1);
    map.put("arr2", arr2);
    map.put("arr3", arr3);
    map.put("arr4", arr4);
    map.put("arr5", arr5);
    map.put("arr6", arr6);
    map.put("arr7", arr7);
    map.put("arr8", arr8);

    EQUAL(map.getList("arr0").size(), 0);
    EQUAL(map.getList("arr1").size(), 0);
    EQUAL(map.getList("arr2").size(), 0);
    EQUAL(map.getList("arr3").size(), 0);
    EQUAL(map.getList("arr4").size(), 0);
    EQUAL(map.getList("arr5").size(), 0);
    EQUAL(map.getList("arr6").size(), 0);
    EQUAL(map.getList("arr7").size(), 0);
    EQUAL(map.getList("arr8").size(), 0);

    arr0 = new String[]{"",""};
    arr1 = new byte[]{(byte)12, (byte)34};
    arr2 = new short[]{(short)123, (short)456};
    arr3 = new char[]{(char)55, (char)66};
    arr4 = new int[]{123456,7890};
    arr5 = new long[]{1234567890L, 987654321L};
    arr6 = new float[]{1.234f, 5.678f};
    arr7 = new double[]{1.23456789, 9.87654321};
    arr8 = new boolean[]{true, false, false, true};

    map.put("arr0", arr0);
    map.put("arr1", arr1);
    map.put("arr2", arr2);
    map.put("arr3", arr3);
    map.put("arr4", arr4);
    map.put("arr5", arr5);
    map.put("arr6", arr6);
    map.put("arr7", arr7);
    map.put("arr8", arr8);

    List list;
    list = map.getList("arr0");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), "");
    EQUAL(list.get(1), "");

    list = map.getList("arr1");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), (byte)12);
    EQUAL(list.get(1), (byte)34);

    list = map.getList("arr2");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), (short)123);
    EQUAL(list.get(1), (short)456);

    list = map.getList("arr3");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), (char)55);
    EQUAL(list.get(1), (char)66);

    list = map.getList("arr4");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), 123456);
    EQUAL(list.get(1), 7890);

    list = map.getList("arr5");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), 1234567890L);
    EQUAL(list.get(1), 987654321L);

    list = map.getList("arr6");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), 1.234f);
    EQUAL(list.get(1), 5.678f);

    list = map.getList("arr7");
    EQUAL(list.size(), 2);
    EQUAL(list.get(0), 1.23456789);
    EQUAL(list.get(1), 9.87654321);

    list = map.getList("arr8");
    EQUAL(list.size(), 4);
    EQUAL(list.get(0), true);
    EQUAL(list.get(1), false);
    EQUAL(list.get(2), false);
    EQUAL(list.get(3), true);
  }

  public void getList_null()
  {
    MSG("lk̏ꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("A", null);
    EQUAL(map.getList("A").size(), 0);
    EQUAL(map.getList("B").size(), 0);
  }

  public void getList_notCollection()
  {
    MSG("lRNVłzłȂꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();

    map.put("a", "ABC");
    map.put("b", 123);
    map.put("c", new DateTime(2011,1,2,3,4,5));

    List list;
    list = map.getList("a");
    EQUAL(list.size(), 1);
    EQUAL(list.get(0), "ABC");

    list = map.getList("b");
    EQUAL(list.size(), 1);
    EQUAL(list.get(0), 123);

    list = map.getList("c");
    EQUAL(list.size(), 1);
    EQUAL(list.get(0), new DateTime(2011,1,2,3,4,5));
  }

  public void toBigDecimal_str()
  {
    MSG("lBigDecimalIuWFNgɕϊ郁\bh̊mFB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    EQUAL(map.toBigDecimal(null), new BigDecimal(0));
    EQUAL(map.toBigDecimal(""), new BigDecimal(0));
    EQUAL(map.toBigDecimal("   "), new BigDecimal(0));

    EQUAL(map.toBigDecimal("0"), new BigDecimal(0));
    EQUAL(map.toBigDecimal("00000"), new BigDecimal(0));
    EQUAL(map.toBigDecimal("   00000  "), new BigDecimal(0));
    EQUAL(map.toBigDecimal("-0"), new BigDecimal(0));
    EQUAL(map.toBigDecimal("-00000"), new BigDecimal(0));
    EQUAL(map.toBigDecimal(" -  00000  "), new BigDecimal(0));

    EQUAL(map.toBigDecimal("1"), new BigDecimal(1));
    EQUAL(map.toBigDecimal("00123"), new BigDecimal(123));
    EQUAL(map.toBigDecimal("   00012  "), new BigDecimal(12));
    EQUAL(map.toBigDecimal("-8"), new BigDecimal(-8));
    EQUAL(map.toBigDecimal("-00089"), new BigDecimal(-89));
    EQUAL(map.toBigDecimal(" -  00987  "), new BigDecimal(-987));

    EQUAL(map.toBigDecimal(" 0.0111 "), new BigDecimal("0.0111"));
    EQUAL(map.toBigDecimal(" 0000.0222  "), new BigDecimal("0.0222"));
    EQUAL(map.toBigDecimal(" -0.0111 "), new BigDecimal("-0.0111"));
    EQUAL(map.toBigDecimal(" -  0000.0222  "), new BigDecimal("-0.0222"));
    EQUAL(map.toBigDecimal(" -.0111 "), new BigDecimal("-0.0111"));
    EQUAL(map.toBigDecimal(" -   .0222  "), new BigDecimal("-0.0222"));
  }

  public void toBigDecimal_str_FailToCastNumber()
  {
    MSG("lւ̕ϊɎsꍇB");

    TypedMap<String,Object> map = new TypedMap<String,Object>();
    try {
      map.toBigDecimal("a");
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), AbstractTypedGetter.Error.FailToCastNumber);
      OK(e.toString());
    }
  }
}
