/*
 * QueryParam class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Collection;
import java.util.Iterator;
import java.util.Arrays;

/**
 * クエリ・パラメータ・クラス。
 * <br>
 * クエリの実行内容に適用するパラメータを表すクラス。
 * パラメータの名前、値を属性に持ち、その他にデータ型などの情報を任意の属性
 * として保持することができる。
 * <br>
 * {@link #getValue()}メソッドは、値が単数の場合はその値を常に返す。値が複数の
 * 場合はそれらの値を順番に返して最後に至った後は最後の値を返し続ける。
 *
 * @author 佐藤隆之
 * @version $Id: QueryParam.java,v 1.7 2011-09-18 16:15:16 tayu Exp $
 */
public class QueryParam extends AbstractTypedGetter<String,Serializable>
{
  /** シリアル・バージョン番号。 */
  private static final long serialVersionUID = -7316624337998775137L;

  /** 入出力区分を表す列挙型。 */
  public enum IO {
    /** 入出力区分が入力であることを表す列挙値。 */
    IN,
    /** 入出力区分が出力であることを表す列挙値。 */
    OUT,
    /** 入出力区分が入力兼出力であることを表す列挙値。 */
    INOUT,
  }

  /** パラメータ名。 */
  private final String name;

  /** 入出力区分。 */
  private final IO io;

  /** パラメータ値。パラメータ値が複数の場合は最後の要素の値を保持する。 */
  private Serializable value = null;

  /** パラメータ値が複数の場合にそれらを順番に取り出すためのイテレータ。 */
  private Iterator<Serializable> iterator = null;

  /** パラメータ値が複数の場合にそれらを格納するコレクションへの参照。 */
  private Collection<Serializable> collection = null;

  /** 任意の属性を格納するマップ。 */
  private HashMap<String,Serializable> attrMap = null;

  /**
   * パラメータ名を引数にとるコンストラクタ。
   * <br>
   * 入出力区分は入力、値はヌルに指定される。 
   *
   * @param name パラメータ名。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryParam(String name)
  {
    this(name, IO.IN);
  }

  /**
   * パラメータ名と入出力区分を引数にとるコンストラクタ。
   * <br>
   * 値はヌルに指定される。
   *
   * @param name パラメータ名。
   * @param io 入出力区分。
   * @throws AsserionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryParam(String name, IO io)
  {
    assert (name != null && io != null) :
      (name == null) ? "@param:name is null." :
      (io   == null) ? "@param:io is null." :
      "";
    this.name = name;
    this.io = io;
  }

  /**
   * パラメータ名を取得する。
   *
   * @return パラメータ名。
   */
  public String getName()
  {
    return this.name;
  }

  /**
   * パラメータの入出力区分を取得する。
   *
   * @return パラメータの入出力区分。
   */
  public IO getIO()
  {
    return this.io;
  }

  /**
   * 指定されたキーに結びつけられた属性値を取得する。
   *
   * @param key キー。
   * @return 属性値。
   */
  @Override
  public Serializable get(String key)
  {
    return (this.attrMap == null) ? null : this.attrMap.get(key);
  }

  /**
   * 指定されたキーを属性名とする属性値を設定する。
   *
   * @param key キー。
   * @param attrValue 属性値。
   * @return 以前このキーに結びつけられていた属性値。
   */
  public Object put(String key, Serializable attrValue)
  {
    if (this.attrMap == null) {
      this.attrMap = new HashMap<String,Serializable>();
    }

    if (attrValue == null) {
      return this.attrMap.remove(key);
    }
    else {
      return this.attrMap.put(key, attrValue);
    }
  }

  /**
   * パラメータ値を設定する。
   * <br>
   * 引数はコレクション・オブジェクト又は配列の場合は複数の値を持つものとし、
   * それ以外は単数の値を持つものとして設定する。
   */
  public void setValue(Object paramValue)
  {
    if (paramValue == null) {
      this.value = null;
      this.collection = null;
      this.iterator = null;
    }
    else if (paramValue instanceof Collection) {
      this.value = null;
      this.collection = toCollection(paramValue);
      this.iterator = this.collection.iterator();
    }
    else if (paramValue instanceof Serializable[]) {
      this.value = null;
      this.collection = Arrays.asList(Serializable[].class.cast(paramValue));
      this.iterator = this.collection.iterator();
    }
    else {
      this.value = Serializable.class.cast(paramValue);
      this.collection = null;
      this.iterator = null;
    }
  }

  /**
   * パラメータ値を取得する。
   * <br>
   * 値が単数の場合はその値を常に返し、値が複数の場合はそれらの値を順番に返して
   * 最後に至った後は最後の値を返し続ける。
   */
  public Object getValue()
  {
    if (this.iterator != null && this.iterator.hasNext())
      this.value = this.iterator.next();

    return this.value;
  }

  /**
   * パラメータ値の数を取得する。
   *
   * @return パラメータ値の数。
   */
  public int countValues()
  {
    return (this.collection == null) ? 1 : this.collection.size();
  }

  /**
   * オブジェクトが{@link java.util.Collection}オブジェクトに型変換する。
   *
   * @param obj オブジェクト。
   * @return コレクション・オブジェクト。
   */
  @SuppressWarnings("unchecked")
  protected Collection<Serializable> toCollection(Object obj)
  {
    return (Collection<Serializable>) obj;
  }
}
