/*
 * Query class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import java.util.List;
import java.util.Map;

/**
 * クエリを実行するクラスのインターフェイス。
 * <br>
 * 指定された{@link QueryConnection}オブジェクトが示す接続先に対して、{@link
 * QueryResource}オブジェクトと入力パラメータ・マップから作成した実行内容を
 * 実行する。
 * クエリの実行結果は{@link QueryResult}オブジェクトを作成し、それに格納されて
 * 返される。
 * {@link #executeSet(String[], Map)}など、複数のクエリを実行
 * するメソッドは、各クエリの結果である{@link QueryResult}オブジェクトを
 * {@link QueryResultList}オブジェクトに格納して返される。
 * なお、この{@link QueryResult}オブジェクトは、そのクエリが属する{@link
 * QueryTransaction}オブジェクトにも登録され、クエリの実行履歴として取得する
 * ことができる。
 * <br>
 * <br>
 * クエリを実行する手順のイメージは以下のようになる：
 * <tt><pre>
 *     tran : QueryTransaction
 *     connAFactory, connBFactory : QueryConnectionFactory
 *     resA0, resA1, resB0 : QueryResource
 *     inputMap : Map<String,Object>
 *
 *     tran.addConnectionFactory("ConnA", connAFactory);
 *     tran.addConnectionFactory("ConnB", connBFactory);
 *     ...
 *
 *     try {
 *       trans.begin();
 *
 *       QueryConnection connA = tran.getConnection("ConnA");
 *       Query queryA0 = connA.getQuery(resA0);
 *       QueryResult rsltA00 = queryA0.execute("qidA00", inputMap);
 *       QueryResult rsltA01 = queryA0.execute("qidA01", inputMap);
 *       ...
 *
 *       Query queryA1 = connA.getQuery(resA1);
 *       QueryResultList rsltLstA1 = queryA1.executeSet(
 *         new String[]{ "qidA10", "qidA11" }, inputMap);
 *       QueryResult rsltA10 = rsltLst.getResultAt(0);
 *       QueryResult rsltA11 = rsltLst.getResultAt(1);
 *       ...
 *
 *       QueryConnection connB = tran.getConnection("ConnB");
 *       Query queryB0 = connB.getQuery(resB0);
 *       QueryResultList rsltB0 = queryB0.executeAll(inputMap);
 *       QueryResult rsltB00 = rsltLst.getResultAt(0);
 *       QueryResult rsltB01 = rsltLst.getResultAt(1);
 *       QueryResult rsltB02 = rsltLst.getResultAt(2);
 *       ...
 *
 *       tran.commit();
 *     }
 *     catch (Exception e) {
 *       tran.rollback();
 *     }
 *     finally {
 *       tran.end();
 *     }
 *
 *     QueryResultLst rsltHistory = trans.getResults();
 *
 *     QueryResult rslt0 = rsltHistory.getResultAt(0);
 *     QueryResult rslt1 = rsltHistory.getResultAt(1);
 *     Table<String,Object> table0 = rslt0.getResultTable();
 *     ...
 *
 *     QueryResult rsltA01 = rsltHistory.getFirstResult("qidA01");
 *     int resultCount = rsltA01.getResultCount();
 *     ...
 * </pre></tt>
 *
 * @author 佐藤隆之
 * @version $Id: Query.java,v 1.8 2011-09-18 16:15:16 tayu Exp $
 */
public interface Query
{
  /** このクラスで発生しうるエラーを示す列挙型。 */
  enum Error {
    /** トランザクション・タイムアウトの場合。 */
    Timeout,

    /** クエリ実行時エラーの場合。 */
    FailToExecute,
  }

  /**
   * このオブジェクトが保持するクエリの中から、指定されたクエリIDに対する1つの
   * クエリを実行する。
   *
   * @param queryId クエリID。
   * @param inputMap 入力パラメータを格納したマップ。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  QueryResult execute(String queryId, Map<String,Object> inputMap)
    throws ReasonedException;

  /**
   * このオブジェクトが保持するクエリの中から、指定された配列内のクエリIDに
   * 対する複数のクエリを順番に実行する。
   *
   * @param qidArr クエリIDの配列。
   * @param inputMap 入力パラメータを格納したマップ。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  QueryResultList executeSet(String[] qidArr, Map<String,Object> inputMap)
    throws ReasonedException;

  /**
   * このオブジェクトが保持するクエリの中から、指定されたリスト内のクエリIDに
   * 対する複数のクエリを順番に実行する。
   *
   * @param qidLst クエリIDのリスト。
   * @param inputMap 入力パラメータを格納したマップ。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  QueryResultList executeSet(List<String> qidLst, Map<String,Object> inputMap)
    throws ReasonedException;

  /**
   * このオブジェクトが保持する全てのクエリを順番に実行する。
   * 
   * @param inputMap 入力パラメータを格納したマップ。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  QueryResultList executeAll(Map<String,Object> inputMap)
    throws ReasonedException;
}
