/*
 * QueryParamValueIterator class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import java.util.Collection;
import java.util.Iterator;
import java.util.Arrays;

/**
 * パラメータの値を格納するイテレータ・クラス。
 * <br>
 * {@link #next()}メソッドは、値が単数の場合はその値を常に返し、値が複数の場合
 * はそれらの値を順番に返して最後に至った後は最後の値を返し続ける。
 *
 * @author 佐藤隆之。
 * @version $Id: QueryParamValueIterator.java,v 1.1 2011-08-01 15:25:37 tayu Exp $
 */
public class QueryParamValueIterator
{
  /** 単値、又は複数の値の最後の要素を保持するオブジェクト。 */
  private Object last;

  /** 複数の値を格納するコレクション・オブジェクト。 */
  private Collection<? extends Object> collection;

  /** 複数の値を格納するイテレータ・オブジェクト。 */
  private Iterator<? extends Object> iterator;

  /**
   * パラメータの値を引数にとるコンストラクタ。
   * <br>
   * 引数がコレクション・オブジェクト又は配列の場合は複数の値を持つものとし、
   * それ以外は単数の値を持つものとしてインスタンスを作成する。
   *
   * @param value 値オブジェクト。
   */
  public QueryParamValueIterator(Object value)
  {
    if (value == null) {
      this.last = null;
      this.collection = null;
      this.iterator = null;
    }
    else if (value instanceof Collection) {
      this.last = null;
      this.collection = toCollection(value);
      this.iterator = this.collection.iterator();
    }
    else if (value instanceof Object[]) {
      this.last = null;
      this.collection = Arrays.asList(Object[].class.cast(value));
      this.iterator = this.collection.iterator();
    }
    else {
      this.last = value;
      this.collection = null;
      this.iterator = null;
    }
  }

  /**
   * 値の数を取得する。
   *
   * @return 値の数。
   */
  public int count()
  {
    return (this.collection == null) ? 1 : this.collection.size();
  }

  /**
   * 値を順に取得していくメソッド。
   * <br>
   * 値が単数の場合はその値を常に返し、値が複数の場合はそれらの値を順番に返して
   * 最後に至った場合は最後の値を返し続ける。
   *
   * @return 値オブジェクト。
   */
  public Object next()
  {
    if (this.iterator != null && this.iterator.hasNext()) {
      this.last = this.iterator.next();
    }
    return this.last;
  }


  /**
   * オブジェクトが{@link java.util.Collection}オブジェクトに型変換する。
   *
   * @param obj オブジェクト。
   * @return コレクション・オブジェクト。
   */
  @SuppressWarnings("unchecked")
  protected Collection<Object> toCollection(Object obj)
  {
    return (Collection<Object>) obj;
  }
}
