/*
 * QueryOutput class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import java.util.Map;
import java.util.HashMap;

/**
 * クエリ出力項目クラス。
 * <br>
 * クエリの実行結果として取得できる出力項目を表すクラス。
 * 出力項目の名前を属性に持ち、その他にデータ型などを任意の属性として保持する
 * ことができる。
 *
 * @author 佐藤隆之
 * @version $Id: QueryOutput.java,v 1.1 2011-08-09 14:08:20 tayu Exp $
 */
public class QueryOutput extends AbstractTypedGetter<String,Object>
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 3690826582113281193L;

  /** 出力項目名。 */
  private final String name;

  /** 任意の属性を格納するマップ。 */
  private Map<String,Object> attrMap = null;

  /**
   * 出力項目名を引数にとるコンストラクタ。
   * <br>
   * 入出力区分は入力、値はヌルに指定される。
   *
   * @param name 出力項目名。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryOutput(String name)
  {
    assert (name != null) : "@param:name is null.";
    this.name = name;
  }

  /**
   * 出力項目名を取得する。
   *
   * @return 出力項目名。
   */
  public String getName()
  {
    return this.name;
  }

  /**
   * このオブジェクトのハッシュ・コードを取得する。
   *
   * @return このオブジェクトのハッシュ・コード。
   */
  @Override
  public int hashCode()
  {
    return ("QueryOutput" + this.name).hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj)
  {
    if (obj == null)
      return false;

    if (! (obj instanceof QueryOutput))
      return false;

    QueryOutput output = QueryOutput.class.cast(obj);
    if (! this.name.equals(output.name))
      return false;

    return true;
  }

  /** 
   * 指定されたキーに結びつけられた属性値を取得する。
   *
   * @param key キー。
   * @return 属性値。
   */
  @Override
  public Object get(String key)
  {
    return (this.attrMap == null) ? null : this.attrMap.get(key);
  }

  /** 
   * 指定されたキーを属性名とする属性値を設定する。
   *
   * @param key キー。
   * @param attrValue 属性値。
   * @return 以前このキーに結びつけられていた属性値。
   */
  public Object put(String key, Object attrValue)
  {
    if (this.attrMap == null) {
      this.attrMap = new HashMap<String,Object>();
    }

    if (attrValue == null) {
      return this.attrMap.remove(key);
    }
    else {
      return this.attrMap.put(key, attrValue);
    }
  }
}
