/*
 * AbstractQuery class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.resource.Resource;
import ts.util.table.Header;
import ts.util.table.Table;
import ts.util.table.ArrayListTable;
import java.util.List;
import java.util.Map;
import java.util.Enumeration;

/**
 * {@link Query}インターフェイスに対する抽象クラス。
 *
 * @author 佐藤隆之
 * @version $Id: AbstractQuery.java,v 1.5 2011-08-09 14:24:17 tayu Exp $
 */
public abstract class AbstractQuery implements Query
{
  /** クエリを実行する際の接続先を示す{@link QueryConnection}オブジェクト。 */
  private final QueryConnection connection;

  /**
   * クエリの実行内容の元となる情報を保持する{@link ts.util.resource.Resource}
   * オブジェクト。
   */
  private final Resource resource;

  /**
   * 接続先を示す{@link QueryConnection}オブジェクトと、実行内容の元情報を
   * 保持する{@link ts.util.resource.Resource}オブジェクトを引数にとる
   * コンストラクタ。
   *
   * @param conn {@link QueryConnection}オブジェクト。
   * @param res  {@link ts.util.resource.Resource}オブジェクト。
   */
  public AbstractQuery(QueryConnection conn, Resource res)
  {
    assert (conn != null && res != null) :
      (conn == null) ? "@param:conn is null." :
      (res  == null) ? "@param:res is null." : "";

    this.connection = conn;
    this.resource = res;
  }

  /**
   * クエリを実行する接続先を示す{@link QueryConnection}オブジェクトを取得する。
   *
   * @return クエリを実行する接続先を示す{@link QueryConnection}オブジェクト。
   */
  protected QueryConnection getConnection()
  {
    return this.connection;
  }

  /**
   * クエリの実行内容の元情報を保持する{@link ts.util.resource.Resource}
   * オブジェクトを取得する。
   *
   * @return クエリの実行内容の元情報を保持する{@link ts.util.resource.Resource}
   *           オブジェクト。
   */
  protected Resource getResource()
  {
    return this.resource;
  }

  /**
   * このオブジェクトが保持するクエリの中から、指定された配列内のIDに対する
   * 複数のクエリを順番に実行する。
   *
   * @param queryIds クエリIDの配列。
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  public void execute(String[] queryIds,
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException
  {
    for (String qid : queryIds) {
      execute(qid, inputMap, resultLst);
    }
  }

  /**
   * このオブジェクトが保持するクエリの中から、指定されたリスト内のIDに対する
   * 複数のクエリを順番に実行する。
   *
   * @param queryIdLst クエリIDのリスト。
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  public void execute(List<String> queryIdLst,
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException
  {
    for (String qid : queryIdLst) {
      execute(qid, inputMap, resultLst);
    }
  }

  /**
   * このオブジェクトが保持する全てのクエリを、クエリ・リソースの中で指定されて
   * いる順番に実行する。
   *
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  @Override
  public void execute(
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException
  {
    for (String qid : listAllQueryIds()) {
      execute(qid, inputMap, resultLst);
    }
  }

  /**
   * 指定されたクエリ・リソースに含まれる全てのクエリのIDを格納したリストを
   * 取得する。
   *
   * @return クエリIDのリスト。
   */
  public abstract List<String> listAllQueryIds();

  /**
   * このオブジェクトが保持するクエリの中から、指定されたIDに対する1つのクエリ
   * を実行する。
   *
   * @param queryId クエリID。
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  public void execute(String queryId,
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException
  {
    assert (queryId != null && inputMap != null && resultLst != null) :
      (queryId  == null) ? "@param:queryId is null." :
      (inputMap == null) ? "@param:inputMap is null." :
      (resultLst== null) ? "@param:resultLst is null." : "";

    long stime = System.currentTimeMillis();

    Resource res = getResource();
    QueryConnection conn = getConnection();
    QueryContext ctx = createContext(conn.getId(), queryId);

    QueryResult rslt = new QueryResult(ctx);
    try {
      checkTimeout(ctx);

      List<QueryParam> paramLst = prepareQuery(ctx, inputMap, resultLst);

      checkTimeout(ctx);

      if (ctx.hasResultTable()) {
        Table<String,Object> tbl = createResultTable(ctx);
        int cnt = executeQuery(ctx, paramLst, tbl);
        rslt.setResultTable(tbl);
        rslt.setResultCount(cnt);
        rslt.setSuccess(true);
      }
      else {
        int cnt = executeQuery(ctx, paramLst);
        rslt.setResultCount(cnt);
        rslt.setSuccess(true);
      }
    }
    catch (ReasonedException e) {
      rslt.setSuccess(false);
      rslt.setException(e);
      throw e;
    }
    catch (ReasonedRuntimeException e) {
      rslt.setSuccess(false);
      ReasonedException exc = new ReasonedException(e.getReason(), e);
      exc.setStackTrace(e.getStackTrace());
      rslt.setException(exc);
      throw exc;
    }
    catch (Exception e) {
      rslt.setSuccess(false);
      ReasonedException exc = new ReasonedException(Error.FailToExecute, e);
      exc.setStackTrace(e.getStackTrace());
      rslt.setException(exc);
      throw exc;
    }
    finally {
      long etime = System.currentTimeMillis();
      rslt.setSpentTimeMillis(etime - stime);
      resultLst.addResult(rslt);
    }
  }

  /**
   * タイムアウトの判定を実行する。
   *
   * @param ctx 実行されるクエリのコンテキスト情報オブジェクト。
   * @throws ReasonedException タイムアウト時刻を超えた場合。
   */
  protected void checkTimeout(QueryContext ctx) throws ReasonedException
  {
    long limitTime = getConnection().getLimitTimeMillis();
    if (limitTime >= 0L) {
      if (limitTime <= System.currentTimeMillis())
        throw new ReasonedException(Error.Timeout, ctx.getQueryId());
    }
  }

  /**
   * 実行されるクエリのコンテキスト情報を作成する。
   *
   * @param connId コネクションID。
   * @param queryId クエリID。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected QueryContext createContext(String connId, String queryId)
  {
    return new QueryContext(connId, queryId);
  }

  /**
   * クエリの結果データを格納するテーブルを作成する。
   *
   * @param ctx クエリのコンテキスト情報オブジェクト。
   * @return 結果データを格納するテーブル。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  protected Table<String,Object> createResultTable(QueryContext ctx)
  {
    assert (ctx != null) : "@param:ctx is null.";

    int nCol = ctx.countOutputNames();
    Header<String> hdr = new ArrayListTable.Header<String>(nCol);
    Enumeration<String> enm = ctx.enumOutputNames();
    while (enm.hasMoreElements()) {
      hdr.addColumn(enm.nextElement());
    }
    return new ArrayListTable<String,Object>(hdr);
  }

  /**
   * クエリ実行の準備処理を行う。
   *
   * @param ctx クエリのコンテキスト情報。
   * @param inputMap 入力パラメータ・マップ。
   * @param resultLst 結果リスト。
   * @return クエリ実行の際に使用する入力パラメータを使用順に格納したリスト。
   * @throws ReasonedException クエリの準備処理に失敗した場合。
   */
  protected abstract List<QueryParam> prepareQuery(
    QueryContext ctx, Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException;

  /**
   * 結果データのあるクエリを実行する。
   *
   * @param ctx クエリのコンテキスト情報オブジェクト。
   * @param paramLst 入力パラメータのリスト。
   * @param tbl 結果データを格納するテーブル。
   * @return クエリの処理件数。
   * @throws ReasonedException クエリの実行に失敗した場合。
   */
  protected abstract int executeQuery(
    QueryContext ctx, List<QueryParam> paramLst, Table<String,Object> tbl
  ) throws ReasonedException;

  /**
   * 結果データのないクエリを実行する。
   *
   * @param ctx クエリのコンテキスト情報オブジェクト。
   * @param paramLst 入力パラメータのリスト。
   * @return クエリの処理件数。
   * @throws ReasonedException クエリの実行に失敗した場合。
   */
  protected abstract int executeQuery(
    QueryContext ctx, List<QueryParam> paramLst
  ) throws ReasonedException;
}
