/*
 * Element class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.resource;

import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.LinkedHashMap;

public class Element
{
  private String name_ ;

  private String value_ = null;

  private AttributeMap attrMap_ = new AttributeMap();

  private Element parent_ = null;

  Map<String, List<Element>> childMap_ =
    new LinkedHashMap<String, List<Element>>();

  public static Element createRootElement()
  {
    return new Element() {
        @Override public String setValue(String value) {
          assert false;
          return "";
        }
        @Override public String putAttribute(String name, String value) {
          assert false;
          return "";
        }
      };
  }

  private Element()
  {
    name_ = "";
    parent_ = null;
  }

  public Element(String name)
  {
    assert (name != null) : "@param:name is null.";
    assert (name.length() > 0) : "@param:name is empty.";

    name_ = name;
  }

  public String getName()
  {
    return name_ ;
  }

  public String getValue()
  {
    return value_ ;
  }

  public String setValue(String value)
  {
    String oldValue = value_ ;
    value_ = value;
    return oldValue;
  }

  public Element getParent()
  {
    return parent_ ;
  }

  public AttributeMap getAttributes()
  {
    return attrMap_ ;
  }

  public String putAttribute(String name, String value)
  {
    assert (name != null) : "@param;name is null.";
    assert (value != null) : "@param:value is null";
    assert (name.length() > 0) : "@param:name is empty";

    return attrMap_.put(name, value);
  }

  public void addChild(Element child)
  {
    assert (child != null) : "@param:child is null.";

    List<Element> eL = childMap_.get(child.getName());
    if (eL == null) {
      eL = new LinkedList<Element>();
      childMap_.put(child.getName(), eL);
    }
    child.parent_ = this;
    eL.add(child);
  }

  public void removeChildren(String name)
  {
    assert (name != null) : "@param:name is null.";

    childMap_.remove(name);
  }

  public void removeAllChildren()
  {
    childMap_.clear();
  }

  public Element getFirstChild(String name)
  {
    assert (name != null) : "@param:name is null.";

    List<Element> eL = childMap_.get(name);
    if (eL == null || eL.size() == 0) {
      return null;
    }
    return eL.get(0);
  }

  public List<Element> getChildren(String name)
  {
     assert (name != null) : "@param:name is null.";

     List<Element> eL = childMap_.get(name);
     return (eL != null) ? eL : new LinkedList<Element>();
  }

  public List<Element> getAllChildren()
  {
    List<Element> allL = new LinkedList<Element>();
    for (List<Element> eL : childMap_.values()) {
      for (Element e : eL) {
        allL.add(e);
      }
    }
    return allL;
  }
}
