require "rubygems"
require "active_record"

# テーブルに格納されている時刻を透過的に扱うためのモジュール。
module TimestampsAsString
  def self.included(base) #:nodoc:
    super

    base.alias_method_chain :create, :string_timestamps
    base.alias_method_chain :update, :string_timestamps

    base.record_timestamps = false
    base.class_inheritable_accessor :record_string_timestamps, :instance_writer => false
    base.record_string_timestamps = true

    base.extend(ClassMethods)
  end

  module ClassMethods
    def timestamps_as_string
      class_eval do
        alias :created_at :created_at_with_string_timestamps
        alias :updated_at :updated_at_with_string_timestamps
      end
    end
  end

  # 作成の際にカラム <tt>created_at</tt> や <tt>updated_at</tt> に日時を内部表現で格納する。
  def create_with_string_timestamps #:nodoc:
    if record_string_timestamps
      s = Time.now.utc.strftime("%Y%m%d%H%M%S")
      write_attribute("created_at", s) if self.class.column_names.include?("created_at") && created_at.nil?
      write_attribute("updated_at", s) if self.class.column_names.include?("updated_at")
    end
    create_without_string_timestamps
  end

  # 更新の際にカラム <tt>updated_at</tt> に日時を内部表現で格納する。
  def update_with_string_timestamps #:nodoc:
    if record_string_timestamps
      s = Time.now.utc.strftime("%Y%m%d%H%M%S")
      write_attribute("updated_at", s) if self.class.column_names.include?("updated_at")
    end
    update_without_string_timestamps
  end

  # <tt>created_at</tt> を置き換える。
  # 内部表現から時刻オブジェクトへ変換して返す。
  def created_at_with_string_timestamps
    return convert_from_string_timestamp(read_attribute_before_type_cast("created_at"))
  end

  # <tt>updated_at</tt> を置き換える。
  # 内部表現から時刻オブジェクトへ変換して返す。
  def updated_at_with_string_timestamps
    return convert_from_string_timestamp(read_attribute_before_type_cast("updated_at"))
  end

  module_function

  def convert_from_string_timestamp(str)
    if str.nil?
      return nil
    elsif /\A(....)(..)(..)(..)?(..)?(..)?\z/ =~ str
      return Time.utc(*$~.captures.compact).localtime
    else
      raise ArgumentError, "invalid timestamp #{str.inspect}"
    end
  end

end

ActiveRecord::Base.class_eval do
  include TimestampsAsString
end
