# -*- coding: utf-8 -*-
# データに対する権限を実装するモジュール。
module ActiveRecord
  module Acts
    module Permissible
      def self.included(base)
        base.extend(ClassMethods)
      end

      module ClassMethods
        # あるモデルクラスでこの <tt>acts_as_permissible</tt> を呼び出すと、
        # そのクラスの <tt>find</tt>、<tt>count</tt> および <tt>with_scope</tt> を置き換える。
        # それぞれのメソッドで権限によって許可されているデータのみが取得され、
        # それ以外のデータは透過的に存在しないものとされる。
        def acts_as_permissible(options = {})
          return if permissible?

          write_inheritable_attribute(:acts_as_permissible_options,
                                      { :parent => options[:parent]
                                      })
          class_inheritable_reader :acts_as_permissible_options

          class << self
            alias_method_chain :validate_find_options, :permission
            alias_method :administrator_find, :find
            alias_method :count_with_permission, :count
            alias_method :administrator_with_scope, :with_scope
          end
          has_many :grant_ons, :as => :grant_targettable
          has_many :permissions, :as => :grant_targettable
          include InstanceMethods
        end

        def permissible?
          self.included_modules.include? ActiveRecord::Acts::Permissible::InstanceMethods
        end

        protected

        VALID_FIND_OPTIONS_WITH_PERMISSION =
          class << ::ActiveRecord::Base
            VALID_FIND_OPTIONS
          end + [:with_permission]

        def validate_find_options_with_permission(options)
          options.assert_valid_keys(VALID_FIND_OPTIONS_WITH_PERMISSION)
        end

        private

        # see the original version of activerecord-2.1.2/lib/active_record/associations.rb
        def configure_dependency_for_has_many(reflection)
          if reflection.options.include?(:dependent)
            # Add polymorphic type if the :as option is present
            dependent_conditions = []
            dependent_conditions << "#{reflection.primary_key_name} = \#{record.quoted_id}"
            dependent_conditions << "#{reflection.options[:as]}_type = '#{base_class.name}'" if reflection.options[:as]
            dependent_conditions << sanitize_sql(reflection.options[:conditions]) if reflection.options[:conditions]
            dependent_conditions = dependent_conditions.collect {|where| "(#{where})" }.join(" AND ")

            case reflection.options[:dependent]
            when :destroy
              method_name = "has_many_dependent_destroy_for_#{reflection.name}".to_sym
              define_method(method_name) do
                if reflection.klass.permissible?
                  x = reflection.klass.find_with_permission(:all, :conditions => {reflection.primary_key_name => self.id})
                else
                  x = send("#{reflection.name}")
                end
                x.each(&:destroy)
              end
              before_destroy method_name
            when :delete_all
              module_eval "before_destroy { |record| #{reflection.class_name}.delete_all(%(#{dependent_conditions})) }"
            when :nullify
              module_eval "before_destroy { |record| #{reflection.class_name}.update_all(%(#{reflection.primary_key_name} = NULL),  %(#{dependent_conditions})) }"
            else
              raise ArgumentError, "The :dependent option expects either :destroy, :delete_all, or :nullify (#{reflection.options[:dependent].inspect})"
            end
          end
        end

        # see the original version of activerecord-2.1.2/lib/active_record/associations.rb
        def configure_dependency_for_has_one(reflection)
          if reflection.options.include?(:dependent)
            case reflection.options[:dependent]
            when :destroy
              method_name = "has_one_dependent_destroy_for_#{reflection.name}".to_sym
              define_method(method_name) do
                if reflection.klass.permissible?
                  association = reflection.klass.find_with_permission(:first, :conditions => {reflection.primary_key_name => self.id})
                else
                  association = send("#{reflection.name}")
                end
                association.destroy unless association.nil?
              end
              before_destroy method_name
            when :delete
              method_name = "has_one_dependent_delete_for_#{reflection.name}".to_sym
              define_method(method_name) do
                if reflection.klass.permissible?
                  association = reflection.klass.find_with_permission(:first, :conditions => {reflection.primary_key_name => self.id})
                else
                  association = send("#{reflection.name}")
                end
                association.class.delete(association.id) unless association.nil?
              end
              before_destroy method_name
            when :nullify
              method_name = "has_one_dependent_nullify_for_#{reflection.name}".to_sym
              define_method(method_name) do
                if reflection.klass.permissible?
                  association = reflection.klass.find_with_permission(:first, :conditions => {reflection.primary_key_name => self.id})
                else
                  association = send("#{reflection.name}")
                end
                association.update_attribute("#{reflection.primary_key_name}", nil) unless association.nil?
              end
              before_destroy method_name
            else
              raise ArgumentError, "The :dependent option expects either :destroy, :delete or :nullify (#{reflection.options[:dependent].inspect})"
            end
          end
        end
      end

      module InstanceMethods
        def self.included(base)
          base.extend(ClassMethods)
        end

        module ClassMethods
          def find(*args)
            options = args.extract_options!
            call_administrator_find = lambda { administrator_find(*(args << options)) }
            if options[:with_permission]
              return call_administrator_find.call
            else
              return with_permission_scope { call_administrator_find.call }
            end
          end

          def find_with_permission(*args)
            return administrator_find(*(args << args.extract_options!.merge(:with_permission => true)))
          end

          def count(*args)
            return with_permission_scope { count_with_permission(*args) }
          end

          # 条件を表すテンプレートの文字列とパラメータのハッシュの組を返す。
          def permission_conditions(user_id, mode)
            if parent = acts_as_permissible_options[:parent]
              a = reflect_on_association(parent.to_sym)
              raise ArgumentError, "unknown association: #{parent}" unless a
              parent_class = a.class_name.constantize
              foreign_key = a.primary_key_name
              parent_template, parent_params = parent_class.permission_conditions(user_id, mode)
              parent_quoted_table_name = parent_class.quoted_table_name
              inner_template = permission_inner_template(mode, parent)
              template = <<-SQL
(#{inner_template}) and
exists (select 1 from #{parent_quoted_table_name} where #{parent_quoted_table_name}.id = #{quoted_table_name}.#{foreign_key} and (#{parent_template}))
SQL
              [template, parent_params]
            else
              [permission_inner_template(mode), {:user_id => user_id, :date => Time.now.strftime("%Y%m%d")}]
            end
          end

          protected

          def permission_roleable_template(name, product)
            type = "Domain"
            if product
              x = product.__send__(name)
              type = x unless x.blank?
            end
            case type
            when "Domain"
              rest = <<-SQL
exists (select 1 from users u1, users u2 where
         u1.domain_id = u2.domain_id and
         u1.id = #{quoted_table_name}.created_by and
         u2.id = :user_id
        )
SQL
            when "Company"
              rest = <<-SQL
exists (select 1 from companies c, users u1, users u2, company_members cm1, company_members cm2 where
          cm1.company_id = c.id and
          cm2.company_id = c.id and
          u1.id = #{quoted_table_name}.created_by and
          u2.id = :user_id and
          cm1.person_id = u1.person_id and
          cm2.person_id = u2.person_id and
          :date between cm1.inception and cm1.expiry and
          :date between cm2.inception and cm2.expiry
        )
SQL
            when "Organization"
              rest = <<-SQL
exists (select 1 from orgainizations c, users u1, users u2, organization_members om1, organization_members om2 where
          om1.organization_id = c.id and
          om2.organization_id = c.id and
          u1.id = #{quoted_table_name}.created_by and
          u2.id = :user_id and
          om1.person_id = u1.person_id and
          om2.person_id = u2.person_id and
          :date between om1.inception and om1.expiry and
          :date between om2.inception and om2.expiry
        )
SQL
            when "Group"
              rest = <<-SQL
exists (select 1 from groups c, users u1, users u2, group_members gm1, group_members gm2 where
          gm1.group_id = c.id and
          gm2.group_id = c.id and
          u1.id = #{quoted_table_name}.created_by and
          u2.id = :user_id and
          gm1.person_id = u1.person_id and
          gm2.person_id = u2.person_id and
          :date between gm1.inception and gm1.expiry and
          :date between gm2.inception and gm2.expiry
        )
SQL
            when "Person"
              rest = <<-SQL
exists (select 1 from users u1, users u2, people p where
          u1.person_id = p.id and
          u2.person_id = p.id and
          u1.id = #{quoted_table_name}.created_by and
          u2.id = :user_id
        )
SQL
            else
              raise ArgumentError, "unknown roleable type: #{type}"
            end
            return "#{quoted_table_name}.created_by IS NULL or (#{rest})"
          end

          def permission_inner_template(mode, parent=nil)
            product = Product.find(:first, :conditions => {:model_name => to_s, Product.inheritance_column => %w|ProductSingle ProductDetailed|})
            # 該当する product が存在し permission_enabled? が真ならば permissions での権限の条件のテンプレートを返す。
            if product && product.permission_enabled?
              quoted_model_name = quote_value(to_s)
              case mode
              when :read
                temp1 = "p1.value = 'visible' or p1.value = 'full'"
                temp2 = "p2.value = 'invisible'"
              when :write
                temp1 = "p1.value = 'full'"
                temp2 = "p2.value = 'invisible' or p2.value = 'visible'"
              else
                raise ArgumentError, "unknown mode: #{mode}"
              end
              # NOTE: priority が小さい値のものを優先する
              if parent
                return <<-SQL
not exists (select 1 from permissions p2 where
             p2.user_id = :user_id and
             :date between p2.inception and p2.expiry and
             p2.grant_targettable_type = #{quoted_model_name} and
             p2.grant_targettable_id = #{quoted_table_name}.id and
             (#{temp2}) and
             (not exists (select 1 from permissions p1 where 
                           p1.user_id = :user_id and
                           :date between p1.inception and p1.expiry and
                           p1.grant_targettable_type = #{quoted_model_name} and
                           p1.grant_targettable_id = #{quoted_table_name}.id and
                           (#{temp1}) and
                           p1.priority <= p2.priority)))
SQL
              else
                return <<-SQL
exists (select 1 from permissions p1 where
         p1.user_id = :user_id and
         :date between p1.inception and p1.expiry and
         p1.grant_targettable_type = #{quoted_model_name} and
         p1.grant_targettable_id = #{quoted_table_name}.id and
         (#{temp1}) and
         (not exists (select 1 from permissions p2 where 
                       p2.user_id = :user_id and
                      :date between p2.inception and p2.expiry and
                       p2.grant_targettable_type = #{quoted_model_name} and
                       p2.grant_targettable_id = #{quoted_table_name}.id and
                       (#{temp2}) and
                       p2.priority < p1.priority)))
SQL
              end
            else
              # スコープをあらわす条件のテンプレートを返す。
              s = permission_roleable_template(:scope_roleable_type, product)
              return s if mode == :read
              t = permission_roleable_template(:initial_roleable_type, product)
              return "(#{s}) and (#{t})"
            end
          end

          def with_permission_scope
            return yield unless User.current
            return yield if User.admin?
            administrator_with_scope({:find => {:conditions => permission_conditions(User.current_id, :read)}}) do
              yield
            end
          end
        end

        def readable?
          accessible?(:read)
        end

        def writable?
          accessible?(:write)
        end

        def readable_for?(user_id)
          accessible_for?(:read, user_id)
        end

        def writable_for?(user_id)
          accessible_for?(:write, user_id)
        end

        private

        def accessible?(mode)
          return true unless User.current
          return true if User.admin?
          return true if new_record?
          template, params = self.class.permission_conditions(User.current_id, mode)
          (self.class.count_with_permission(:conditions => ["#{self.class.quoted_table_name}.id = #{self.id} and (#{template})", params]) > 0)
        end

        def accessible_for?(mode, user_id)
          return true if user_id.blank?
          return true if User.exists?(:id => user_id, :admin => true)
          return true if new_record?
          template, params = self.class.permission_conditions(user_id, mode)
          (self.class.count_with_permission(:conditions => ["#{self.class.quoted_table_name}.id = #{self.id} and (#{template})", params]) > 0)
        end

      end
    end
  end
end
