# -*- coding: utf-8 -*-
# app/models/product.rb のクラス Product をテストする。

require 'test_helper'

class ProductTest < ActiveSupport::TestCase
  fixtures(:users, :products, :displays, :documents, :document_edges, :document_paths)

  def setup
    CacheEachRequest.clear
  end

  def teardown
    CacheEachRequest.clear
  end

  # Product#root がルートになるメニューを返すことをテストする。
  def test_root
    assert_equal products(:root), Product.root
  end

  # Product#root が(ドメイン ID が3の場合に)ルートになるメニューを返すことをテストする。
  def test_root__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_equal products(:root_for_domain_3), Product.root
  end

  # Product#level がメニューの深さを返すことをテストする。
  def test_level
    assert_equal 0, products(:root).level
    assert_equal 1, products(:top_1).level
    assert_equal 4, products(:about).level
  end

  # Product#level が(ドメイン ID が3の場合に)メニューの深さを返すことをテストする。
  def test_level__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_equal 0, products(:root_for_domain_3).level
    assert_equal 1, products(:menu_for_domain_3).level
  end

  # Product#ancestors が祖先のメニューを返すことをテストする。
  def test_ancestors
    assert_array_as_nested_set(:ancestors,
                               :root => [],
                               :top_2 => [:root],
                               :about => [:root, :top_1, :sub_1_1, :sub_1_1_1])
  end

  # Product#ancestors が(ドメイン ID が3の場合に)祖先のメニューを返すことをテストする。
  def test_ancestors__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:ancestors,
                               :root_for_domain_3 => [],
                               :menu_for_domain_3 => [:root_for_domain_3])
  end

  # Product#self_and_ancestors が自分自身および祖先のメニューを返すことをテストする。
  def test_self_and_ancestors
    assert_array_as_nested_set(:self_and_ancestors,
                               :root => [:root],
                               :top_2 => [:root, :top_2],
                               :about => [:root, :top_1, :sub_1_1, :sub_1_1_1, :about])
  end

  # Product#self_and_ancestors が(ドメイン ID が3の場合に)自分自身および祖先のメニューを返すことをテストする。
  def test_self_and_ancestors__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:self_and_ancestors,
                               :root_for_domain_3 => [:root_for_domain_3],
                               :menu_for_domain_3 => [:root_for_domain_3, :menu_for_domain_3])
  end

  # Product#siblings が兄弟関係のメニューを返すことをテストする。
  def test_siblings
    assert_array_as_nested_set(:siblings,
                               :root => [],
                               :top_1 => [:top_2, :top_3, :sample_list, :portal, :grant_on_for_dev, :product_detailed_0, :workshop, :demo, :demo2, :grant_product, :grant_data, :example_project, :example_work, :example_issue, :example_quality_assurance, :matter, :grant_item],
                               :sub_1_1 => [:sub_1_2],
                               :about => [])
  end

  # Product#siblings が(ドメイン ID が3の場合に)兄弟関係のメニューを返すことをテストする。
  def test_siblings__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:siblings,
                               :root_for_domain_3 => [],
                               :menu_for_domain_3 => [])
  end

  # Product#self_and_siblings が自分自身および兄弟関係のメニューを返すことをテストする。
  def test_self_and_siblings
    assert_array_as_nested_set(:self_and_siblings,
                               :root => [:root],
                               :top_1 => [:top_1, :top_2, :top_3, :sample_list, :portal, :grant_on_for_dev, :product_detailed_0, :workshop, :demo, :demo2, :grant_product, :grant_data, :example_project, :example_work, :example_issue, :example_quality_assurance, :matter, :grant_item],
                               :sub_1_1 => [:sub_1_1, :sub_1_2],
                               :about => [:about])
  end

  # Product#self_and_siblings が(ドメイン ID が3の場合に)自分自身および兄弟関係のメニューを返すことをテストする。
  def test_self_and_siblings__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:self_and_siblings,
                               :root_for_domain_3 => [:root_for_domain_3],
                               :menu_for_domain_3 => [:menu_for_domain_3])
  end

  # Product#parent が親メニューを返すことをテストする。
  def test_parent
    assert_nil products(:root).parent
    assert_equal products(:root), products(:top_1).parent
    assert_equal products(:sub_1_1_1), products(:about).parent
  end

  # Product#parent が(ドメイン ID が3の場合に)親メニューを返すことをテストする。
  def test_parent__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_equal products(:root_for_domain_3), products(:menu_for_domain_3).parent
  end

  # Product#children_count が子メニューの個数を返すことをテストする。
  def test_children_count
    assert_equal 18, products(:root).children_count
    assert_equal  2, products(:top_1).children_count
    assert_equal  0, products(:about).children_count
  end

  # Product#children_count が(ドメイン ID が3の場合に)子メニューの個数を返すことをテストする。
  def test_children_count__for_domain3
    User.current = User.find_by_login("geeko")
    assert_equal 1, products(:root_for_domain_3).children_count
    assert_equal 0, products(:menu_for_domain_3).children_count
  end

  # Product#children が子メニューを返すことをテストする。
  def test_children
    assert_array_as_nested_set(:children,
                               :root => [:top_1, :top_2, :top_3, :sample_list, :portal, :grant_on_for_dev, :product_detailed_0, :workshop, :demo, :demo2, :grant_product, :grant_data, :example_project, :example_work, :example_issue, :example_quality_assurance, :matter, :grant_item],
                               :top_1 => [:sub_1_1, :sub_1_2],
                               :about => [])
  end

  # Product#children が(ドメイン ID が3の場合に)子メニューを返すことをテストする。
  def test_children__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:children,
                               :root_for_domain_3 => [:menu_for_domain_3],
                               :menu_for_domain_3 => [])
  end

  # Product#all_children が子メニューを返すことをテストする。
  def test_all_children
    assert_array_as_nested_set(:all_children,
                               :top_1 => [:sub_1_1, :sub_1_1_1, :about, :sub_1_2],
                               :about => [])
  end

  # Product#all_children が(ドメイン ID が3の場合に)子メニューを返すことをテストする。
  def test_all_children__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:all_children,
                               :root_for_domain_3 => [:menu_for_domain_3])
  end

  # Product#fullset が自分自身および子孫のメニューを返すことをテストする。
  def test_full_set
    assert_array_as_nested_set(:full_set,
                               :top_1 => [:top_1, :sub_1_1, :sub_1_1_1, :about, :sub_1_2],
                               :about => [:about])
  end

  # Product#fullset が(ドメイン ID が3の場合に)自分自身および子孫のメニューを返すことをテストする。
  def test_full_set__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:full_set,
                               :root_for_domain_3 => [:root_for_domain_3, :menu_for_domain_3])
  end

  # Product 間の大小関係をテストする。
  def test_position
    top_2 = products(:top_2)
    children = top_2.children
    assert children[0].id > children[1].id
    assert_equal "SubMenu 2 1", children[0].title
    assert_equal "SubMenu 2 2", children[1].title
  end

  # Product#url_options の動作をテストする。
  def test_url_options_without_controller
    top_1 = products(:top_1)
    expected = {:controller => "menu", :action => "mock", :id => "2"}
    assert_equal expected, top_1.url_options
    assert top_1.with_ajax?
  end

  # Product#url_options の動作をテストする。
  def test_url_options_with_controller
    portal = products(:portal)
    expected = {:controller => "portal", :action => "index"}
    assert_equal expected, portal.url_options
    assert portal.with_ajax?
  end

  # Product#url_options の動作をテストする。
  def test_url_options_without_ajax
    grant_item = products(:grant_item)
    expected = {:controller => "product", :action => "list", :product_id => "24"}
    assert_equal expected, grant_item.url_options
    assert !grant_item.with_ajax?
  end

  # Product#permissible および invisible がユーザーの変更にともなって動作することをテストする。
  def test_sharp135
    User.current = users(:admin)
    x = Product.permissible
    assert Product.invisible.empty?
    CacheEachRequest.current[:product] = nil
    User.current.admin = false
    assert x != Product.permissible
  end

  # アプリケーションの管理者によって用意される画面を返す。
  def test_master_displays
    classes = [DisplayToList, DisplayToShow, DisplayToEdit, DisplayToNew]
    Product.find(:all).each do |product|
      product.master_displays.each do |display|
        assert classes.include?(display.class)
      end
    end
  end

  DOCUMENTS_FOR = [
    [999, MatterReport, 1, 1],
    [999, MatterReport, 2, 2],
    [8, Domain, 2, 0],
  ]

  # Product#documents_for() は関連文書の配列を返す。
  def test_documents_for
    DOCUMENTS_FOR.each do |id, model_class, model_id, expected|
      product = Product.find(id)
      x = model_class.find(model_id)
      docs = product.documents_for(x)
      assert_kind_of Array, docs
      assert_equal expected, docs.size
    end
  end

  DOCUMENT_NODES_FOR = [
    [999, MatterReport, 1, 1],
    [999, MatterReport, 2, 3],
    [8, Domain, 2, 0],
  ]

  # Product#document_nodes_for() は関連文書のノードの配列を返す。
  def test_document_nodes_for
    DOCUMENT_NODES_FOR.each do |id, model_class, model_id, expected|
      product = Product.find(id)
      x = model_class.find(model_id)
      docs = product.document_nodes_for(x)
      assert_kind_of Array, docs
      assert_equal expected, docs.size
    end
  end

  private

  def assert_array_as_nested_set(x, h)
    h.each_pair do |k, v|
      expected = v.map {|s| products(s).id}
      value    = products(k).__send__(x).map {|m| m.id}
      assert_equal expected, value
    end
  end
end
