# -*- coding: utf-8 -*-
# app/models/item.rb のクラス Item をテストする。

require 'test_helper'

class ItemTest < ActiveSupport::TestCase
  fixtures :domains, :items, :people, :users, :companies, :company_members, :input_options # avoid using User
  fixtures :matter_reports, :workshops

  def setup
    CacheEachRequest.clear
  end

  def teardown
    CacheEachRequest.clear
  end

  # Item#readable? の動作をテストする。
  def test_readable?
    User.current = users(:tesla)
    assert items(:free).readable?
    assert !items(:admin_only).readable?
  end

  def test_admin_readable?
    User.current = users(:admin)
    assert items(:admin_only).readable?
  end

  # Item#writable? の動作をテストする。
  def test_writable?
    User.current = users(:tesla)
    assert items(:free).writable?
    assert !items(:admin_only).writable?
  end

  def test_admin_writable?
    User.current = users(:admin)
    assert items(:admin_only).writable?
  end

  # Item に対する各種検証の動作をテストする。
  def test_validations
    item = nil
    assert_nothing_raised do
      item = ItemPseudo.create!({
          "domain_id"=>1,
          "name_po"=>0,
          "code"=>"dummy",
          "display_id"=>1,
          "adapter_name"=>"dummy",
          "model_name"=>"Domain",
        })
    end
    item.column_name = "data"
    item.class.column_names.grep(/\Avalidates_[a-z_]+\z/).each do |v|
      item[v] = true
    end
    item.validates_inclusion_chars_1 = "abc"
    item.validates_minimum_length_1 = 1
    item.validates_maximum_length_1 = 9
    item.validates_integral_length_1 = 1
    item.validates_integral_length_2 = 9
    item.validates_fractional_length_1 = 1
    item.validates_fractional_length_2 = 9
    item.validates_minimum_number_1 = 1
    item.validates_period_1 = "19870605"
    item.validates_period_2 = "20010203"
    item.validates_row_1 = "domains"
    item.validates_row_2 = "id"
    item.validates_isolated_1 = "User#domain_id,Company#domain_id"
    assert item.save

    o = Object.new
    class << o
      include CustomValidations::Validateable
      extend CustomValidations
    end
    o.extend CustomValidations::SingletonValidateable

    assert_nothing_raised do
      item.validations(:save).each do |args|
        o.__send__(*args)
      end
    end
  end

  # Item#extract で出力される SQL 文をテストする。
  def test_extract
    quoted_table_name = Domain.quoted_table_name
    i = ItemProper.new do |i|
      i.model_name = "Domain"
      i.column_name = "code"
      i.condition_value = "abc  123 xyz "
    end
    i.condition_pattern = "none"
    assert !i.extract
    i.condition_pattern = "eq"
    v_abc = ItemProper.quote_value('abc')
    v_123 = ItemProper.quote_value('123')
    v_xyz = ItemProper.quote_value('xyz')
    assert_equal "(#{quoted_table_name}.code = #{v_abc} OR #{quoted_table_name}.code = #{v_123} OR #{quoted_table_name}.code = #{v_xyz})", i.extract
    i.condition_pattern = "neq"
    assert_equal "(#{quoted_table_name}.code <> #{v_abc} AND #{quoted_table_name}.code <> #{v_123} AND #{quoted_table_name}.code <> #{v_xyz})", i.extract
    i.condition_pattern = "head"
    v = ItemProper.quote_value('abc  123 xyz %')
    assert_equal "(#{quoted_table_name}.code LIKE #{v})", i.extract
    i.condition_pattern = "tail"
    v = ItemProper.quote_value('%abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.code LIKE #{v})", i.extract
    i.condition_pattern = "include"
    v = ItemProper.quote_value('%abc  123 xyz %')
    assert_equal "(#{quoted_table_name}.code LIKE #{v})", i.extract
    i.condition_pattern = "exclude"
    v = ItemProper.quote_value('%abc  123 xyz %')
    assert_equal "(#{quoted_table_name}.code IS NULL OR #{quoted_table_name}.code NOT LIKE #{v})", i.extract
    i.condition_pattern = "ge"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.code >= #{v})", i.extract
    i.condition_pattern = "le"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.code <= #{v})", i.extract
    i.condition_pattern = "gt"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.code > #{v})", i.extract
    i.condition_pattern = "lt"
    v = ItemProper.quote_value('abc  123 xyz ')
    assert_equal "(#{quoted_table_name}.code < #{v})", i.extract
    i.condition_pattern = "null"
    assert_equal "(#{quoted_table_name}.code IS NULL OR #{quoted_table_name}.code = '')", i.extract
    i.condition_pattern = "not-null"
    assert_equal "(#{quoted_table_name}.code IS NOT NULL AND #{quoted_table_name}.code <> '')", i.extract
  end

  # Item#extract で option_category がエスケープされていることをテストする。
  def test_extract__no565
    item = items(:category9991)
    item.condition_pattern = "eq"
    item.condition_value = "direct"
    m_name = MatterReport.quoted_table_name
    c = ItemProper.quote_value("matter_report_category")
    assert_equal "(EXISTS (SELECT 1 FROM input_options WHERE category = #{c} AND name_po = 999811 AND value = #{m_name}.category))", item.extract
  end

  # Item#inline_values の checkbox パターンをテストする。
  def test_inline_values
    i = ItemProper.new do |i|
      i.condition_pattern = "eq"
      i.condition_value = "aaa   on off bbb"
      i.input_type = "checkbox"
      i.input_parameter = "aaa,bbb"
    end
    on = Item.connection.quoted_true
    off = Item.connection.quoted_false
    assert_equal [on,on,off,off], i.inline_values
  end

  # Item#coordinates で出力されるセルのラベルをテストする。
  def test_coordinates
    assert_equal "A1", Item.find(1).coordinates
    assert_equal "B1", Item.find(2).coordinates
    assert_equal "A2", Item.find(3).coordinates
    assert_equal "A1", Item.find(841).coordinates
    assert_equal "B2", Item.find(842).coordinates
    assert_equal "C3", Item.find(843).coordinates
  end

  # Item#coordinate_alphabet で出力されるセルのラベルのアルファベット部分をテストする。
  def test_coordinate_alphabet
    assert_equal "A", Item.find(1).coordinate_alphabet
  end

  # Item#coordinate_number で出力されるセルのラベルを数値部分をテストする。
  def test_coordinate_number
    assert_equal 1, Item.find(1).coordinate_number
  end

  # Item#link で URL を含むリンクの情報が返ることをテストする。
  def test_link__link
    item = Item.new(:linked => true, :link_parameter => '/foo/bar')
    it = mock('link', :id => 1234321)
    item.stubs(:to_data).with(it).returns("data")
    assert_equal [:link_to, '/foo/bar', nil], item.link(it)
  end

  # Item#link で mailto を含むリンクの情報が返ることをテストする。
  def test_link__mailto_simple
    item = Item.new(:linked => true, :link_parameter => 'mailto')
    it = mock('mailto_simple')
    item.stubs(:to_data).with(it).returns('foo@example.com')
    assert_equal [:mail_to, 'foo@example.com'], item.link(it)
  end

  # Item#link で mailto を含むリンクの情報が返ることをテストする。
  def test_link__mailto_address
    item = Item.new(:linked => true, :link_parameter => 'mailto:bar%{id}@example.com')
    it = mock('mailto_address', :id => 1)
    item.stubs(:to_data).with(it).returns('x')
    assert_equal [:mail_to, 'bar1@example.com', nil], item.link(it)
  end

  # Item#link_url で出力される URL をテストする。
  def test_link_url
    assert_equal false, Item.find(1).link_url(Domain.find(1))
    assert_equal ["http://example.com/2/Domain/D000000002", nil], Item.find(842).link_url(Domain.find(2))
  end

  # Item#link_url で出力される URL をテストする。
  def test_link_url_with_label
    item = ItemProper.new(:model_name => "Product", :column_name => "id", :linked => true, :link_parameter => "/product/1102/list?target_product_id=%{id}>GrantItem")
    assert_equal ['/product/1102/list?target_product_id=2', 'GrantItem'], item.link_url(Product.find(2))
  end

  # Item#link_options が返す link のオプションをテストする。
  def test_link_options
    item = Item.new(:linked => false)
    assert !item.link_options
    item.linked = true
    assert_equal({}, item.link_options)
    item.link_target = "_blank"
    assert_equal({:target => "_blank"}, item.link_options)
  end

  # private_copy で個人利用のためのコピーが返ることをテストする。
  def test_private_copy
    source = items(:one)
    item = source.private_copy(30)
    assert_kind_of ItemProper, item
    assert_equal "C_ITEM_1", item.code
    assert_equal "domain_id", item.column_name
    assert 0 != item.name_po
    assert 8 != item.name_po
  end

  # ItemProper: set_attributes で適切な属性が設定される。
  def test_set_attributes__proper
    item = items(:name821)
    d = Domain.new
    item.set_attributes(d, {:domain_id => 123456, :name => "xxx", :code => "yyy"})
    assert_equal "xxx", d.attributes['name']
  end

  # ItemPseudo: set_attrubutes で何もしない。
  def test_set_attributes__pseudo
    item = items(:pseudo823)
    d = Domain.new
    item.set_attributes(d, {:domain_id => 123456, :name => "xxx", :code => "yyy"})
    assert_nil d.attributes['name']
  end

  # ItemProper#to_data は表示に適したデータを返す。
  def test_to_data__person
    item = ItemProper.new(:column_name => "person_id")
    assert_equal "NEKOの子猫", item.to_data(users(:geeko))
    item = ItemProper.new(:column_name => "default_company_id")
    person = Person.new(:default_company_id => 1)
    assert_equal "ミッドガルドシステムズ", item.to_data(person)
    item = ItemProper.new(:column_name => "search", :input_type => "checkbox", :input_parameter => "aaa,bbb", :search => true)
    assert_equal "aaa", item.to_data(item)
  end

  # Item#reference は外部参照を行うカラムに対応している場合はそのクラスを返す。
  # さもなくば false を返す。
  def test_reference
    item = ItemProper.find(99913)
    assert_equal false, item.__send__(:reference)
    item = ItemProper.find(99924)
    assert_equal Company, item.__send__(:reference)
    item = ItemProper.find(99935)
    assert_equal Person, item.__send__(:reference)
  end

  # ItemProper#option_category は InputOption を参照する場合はその種類を文字列で返す。
  # さもなくば false を返す。
  def test_option_category
    item = items(:category9991)
    assert_equal "matter_report_category", item.option_category
    item = items(:name9991)
    assert !item.option_category
    item = ItemProper.new(:domain_id => 1, :input_type => "select", :input_parameter => "pj_input_option(foo, bar)")
    assert !item.option_category
  end

  # ItemProper#split_into_input_options は選択肢を返す。
  def test_split_into_input_options
    item = items(:category9991)
    options = item.split_into_input_options(item.model_class.find(:first))
    assert_kind_of Array, options
    assert_equal %w|1 2 3 4 5|, options.map(&:last)
    item = ItemProper.new(:domain_id => 1, :model_name => "Person", :input_parameter => "a:alice,b:bob,c:charlie")
    options = item.split_into_input_options(item.model_class.find(:first))
    assert_kind_of Array, options
    assert_equal %w|a b c|, options.map(&:last)
    item = ItemProper.new(:domain_id => 1, :model_name => "Person", :input_parameter => "User#login")
    options = item.split_into_input_options(item.model_class.find(:first))
    assert_kind_of Array, options
    assert options.map(&:first).include?("admin")
  end

  # checkbox 形式を利用するかどうかを判定する。
  def test_checkbox?
    item = ItemProper.new(:input_type => "checkbox")
    assert item.__send__(:checkbox?)
    item = ItemProper.find(1)
    assert_equal false,  item.__send__(:checkbox?)
  end

  # ItemPlural#datum_class がデータの単位となるモデルのクラスを返す。
  def test_datum_class
    assert_equal Person, ItemPlural.find(88814).datum_class
  end

  # ItemPlural#data_method_name がデータを返すためのメソッド名を返す。
  def test_data_method_name
    assert_equal "people", ItemPlural.find(88824).data_method_name
  end

  # ItemProper#method_chain を設定したときと設定していないときのテスト。
  def test_method_chain_of_proper
    matter_report = MatterReport.find(1)
    [
      ItemProper.find(99915), # 詳細画面
      ItemProper.find(99945), # 一覧画面
    ].each do |item|
      assert_equal "米畑鉄矢", item.to_data(matter_report)
      item.update_attributes!(:method_chain => "Person#mail_address")
      assert_equal "id\@a.example.com", item.to_data(matter_report)
      item.update_attributes!(:method_chain => "Person#email_address")
      assert_raise(NoMethodError) do
        item.to_data(matter_report)
      end
      item.update_attributes!(:method_chain => "Person#default_organization_id.Organization#name")
      assert_nil item.to_data(matter_report)
      item.update_attributes!(:method_chain => "name")
      e = assert_raise(Item::Error) do
        item.to_data(matter_report)
      end
      assert_match(/invalid method_chain/, e.message)
    end
  end

  # ItemPlural#method_chain を設定したときと設定していないときのテスト。
  def test_method_chain_of_plural
    workshop = Workshop.find(1)
    [
      ItemPlural.find(88814), # 詳細画面
      ItemPlural.find(88844), # 一覧画面
    ].each do |item|
      assert_equal "ホームラン王,米畑鉄矢,ひがしたいよう", item.to_data(workshop)
      item.update_attributes!(:method_chain => "Person#mail_address")
      assert_equal "king\@a.example.com,id\@a.example.com,higashi\@b.example.com", item.to_data(workshop)
      item.update_attributes!(:method_chain => "Person#email_address")
      assert_raise(NoMethodError) do
        item.to_data(workshop)
      end
      item.update_attributes!(:method_chain => "Person#default_organization_id.Organization#name")
      # NOTE: 以下のように複数の空文字列(この場合は3つ)を連結する可能性がある
      assert_equal ",,", item.to_data(workshop)
      item.update_attributes!(:method_chain => "name")
      e = assert_raise(Item::Error) do
        item.to_data(workshop)
      end
      assert_match(/invalid method_chain/, e.message)
    end
  end

  # ItemPolymorphic#method_chain を設定したときと設定していないときのテスト。
  def test_method_chain_of_polymorphic
    workshop = Workshop.find(1)
    [
      ItemPolymorphic.find(88813), # 詳細画面
      ItemPolymorphic.find(88843), # 一覧画面
    ].each do |item|
      assert_equal "ミッドガルドシステムズ", item.to_data(workshop)
      item.update_attributes!(:method_chain => "name")
      assert_equal "ミッドガルドシステムズ", item.to_data(workshop)
      item.update_attributes!(:method_chain => "Person#email_address")
      e = assert_raise(Item::Error) do
        item.to_data(workshop)
      end
      assert_match(/invalid method_chain/, e.message)
      item.update_attributes!(:method_chain => "domain_id.Domain#name")
      assert_equal "wonda", item.to_data(workshop)
      item.update_attributes!(:method_chain => "domain.Domain#name")
      e = assert_raise(Item::Error) do
        item.to_data(workshop)
      end
      assert_match(/invalid id/, e.message)
    end
  end

  # 存在しない person_id を指定した場合のテスト
  def test_method_chain_of_proper__missing_id
    MatterReport.find(1).update_attributes!(:order_person_id => 1111)
    matter_report = MatterReport.find(1)
    [
      ItemProper.find(99915), # 詳細画面
      ItemProper.find(99945), # 一覧画面
    ].each do |item|
      assert_equal 1111, item.to_data(matter_report)
      item.update_attributes!(:method_chain => "Person#mail_address")
      assert_equal "", item.to_data(matter_report)
      item.update_attributes!(:method_chain => "Person#default_organization_id.Organization#name")
      assert_equal "",  item.to_data(matter_report)
      item.update_attributes!(:method_chain => "name")
      e = assert_raise(Item::Error) do
        item.to_data(matter_report)
      end
      assert_match(/invalid method_chain/, e.message)
    end
  end

  # input_initializer に today を指定した場合のテスト。
  def test_initialize_column_of__today
    item = Item.new(:input_initializer => "today",
                    :input_initial_value => "yyyy/mm/dd",
                    :column_name => "abc")
    item.stubs(:writable?).returns(true)
    it = mock()
    it.expects(:[]=).with("abc", Time.now.strftime("%Y/%m/%d"))
    item.initialize_column_of(it)

    item.input_initial_value = ""
    it = mock()
    it.expects(:[]=).with("abc", Time.now.strftime("%Y%m%d"))
    item.initialize_column_of(it)
  end

  # input_initializer に counter を指定した場合のテスト。
  def test_fill_column_of__counter
    item = Item.new(:input_initializer => "counter",
                    :input_initial_value => "f=yyyymm,-障害-,s=FOOS.5",
                    :column_name => "abc")
    item.stubs(:writable?).returns(true)
    it = mock()
    it.expects(:[]=).with("abc", Date.today.strftime("%Y%m-障害-00001"))
    assert item.fill_column_of(it)

    item.stubs(:writable?).returns(false)
    it = mock()
    assert !item.fill_column_of(it)

    item = Item.new(:input_initial_value => "today")
    item.stubs(:writable?).returns(true)
    it = mock()
    assert !item.fill_column_of(it)
  end
end
