# -*- coding: utf-8 -*-
# lib/document_styles.rb のモジュール DocumentStyles をテストする。

require 'test_helper'

class FooBar
  include DocumentStyles::StyleBorderWidth
end
class BarBaz
  include DocumentStyles::StyleMargin
end
class BazFoo
  include DocumentStyles::StyleAlignment
end

class DocumentStylesTest < ActiveSupport::TestCase

  VALID_COLOR = [
    [0, 0, 0],
    [255, 255, 255],
    [123, 213, 132],
    [0xff, 0xcc, 0xcc],
    ["255", "255", "0255"],
  ]

  INVALID_COLOR = [
    [256, 0, 0],
    [-2, 123, 123],
  ]

  # DocumentStyles::Color の動作をテストする。
  def test_color
    color = DocumentStyles::Color.new # without arguments
    assert_kind_of DocumentStyles::Color, color
    VALID_COLOR.each do |r, g, b|
      color = DocumentStyles::Color.new(r, g, b)
      assert_kind_of DocumentStyles::Color, color
      assert_equal r.to_i, color.red
      assert_equal g.to_i, color.green
      assert_equal b.to_i, color.blue
    end
    INVALID_COLOR.each do |r, g, b|
      assert_raise(RuntimeError) do
        DocumentStyles::Color.new(r, g, b)
      end
    end
  end

  BORDER_WIDTH = {
    :top    => 0.5,
    :bottom => 1,
    :left   => 0.3,
    :right  => 0.1,
  }

  # DocumentStyles::BorderWidth の動作をテストする。
  def test_border_width
    bw = DocumentStyles::BorderWidth.new # without arguments
    [:top, :bottom, :left, :right].each do |k|
      assert_equal 0.2, bw[k]
    end
    assert_equal 0.2, bw.common_width
    BORDER_WIDTH.each do |k, v|
      bw[k] = v
      assert_equal v, bw[k]
    end
    assert_nil bw.common_width
    assert_raise(RuntimeError) do
      bw[:no_such_key] = 0.1
    end
    assert_raise(RuntimeError) do
      bw[:no_such_key]
    end
  end

  # DocumentStyles::StyleBorderWidth の動作をテストする。
  def test_style_border_width
    foo = FooBar.new
    foo.border_width = DocumentStyles::BorderWidth.new
    assert_kind_of DocumentStyles::BorderWidth, foo.border_width
    assert_raise(RuntimeError)do
      foo.border_width = 1
    end
  end

  MARGIN = {
    :top    => 0.5,
    :bottom => 1,
    :left   => 0.3,
    :right  => 0.1,
  }

  # DocumentStyles::Margin の動作をテストする。
  def test_margin
    margin = DocumentStyles::Margin.new # without arguments
    [:top, :bottom, :left, :right].each do |k|
      assert_equal 0.0, margin[k]
    end
    MARGIN.each do |k, v|
      margin[k] = v
      assert_equal v, margin[k]
    end
    assert_raise(RuntimeError) do
      margin[:no_such_key]
    end
    assert_raise(RuntimeError) do
      margin[:no_such_key] = 0
    end
  end

  # DocumentStyles::StyleMargin の動作をテストする。
  def test_style_margin
    bar = BarBaz.new
    bar.margin = DocumentStyles::Margin.new
    assert_kind_of DocumentStyles::Margin, bar.margin
    assert_raise(RuntimeError)do
      bar.margin = 1
    end
  end

  # DocumentStyles::StyleAlignment の動作をテストする。
  def test_style_alignment
    baz = BazFoo.new
    {
      :left => "L",
      :right => "R",
      :center => "C",
    }.each do |k, v|
      baz.align = k
      assert_equal k, baz.align
      assert_equal v, baz.align_expr
    end
    assert_raise(RuntimeError) do
      baz.align = :no_such_key
    end
  end
end
