# -*- coding: utf-8 -*-
# 権限の制御をテストする。

require File.dirname(__FILE__) + '/../test_helper'

class ActsAsPermissibleTest < ActionController::IntegrationTest
  fixtures(:permissions, :products,
           :users, :people, :domains,
           :companies, :company_members,
           :organizations, :organization_members,
           :groups, :group_members,
           :matters, :workshops)
  self.use_transactional_fixtures = false

  def setup
    CacheEachRequest.clear
    @logistics = matters(:logistics)
    @skynet    = matters(:skynet)
    @tesla = users(:tesla)
    @admin = users(:admin)
  end

  def teardown
    CacheEachRequest.clear
  end

  # ActsAsPermissible による制御の下で限られたオブジェクトのみが find で利用できることをテストする。
  def test_find
    # User.current = nil
    assert_equal @logistics, Matter.find(1)
    assert_equal @skynet, Matter.find(2)
  end

  # ActsAsPermissible による制御の下で限られたオブジェクトのみが find で利用できることをテストする。
  def test_find__tesla
    User.current = @tesla
    assert_equal @skynet, Matter.find(2)
    assert !Matter.find(:first, :conditions => {:anken_code => "M1"})
  end

  # ActsAsPermissible による制御の下で限られたオブジェクトのみが find で利用できることをテストする。
  def test_find__admin
    User.current = @admin
    assert_equal @logistics, Matter.find(1)
    assert_equal @skynet, Matter.find(:first, :conditions => {:anken_code => "M2"})
  end

  # ActsAsPermissible による制御の下で find_by_* のふるまいをテストする。
  # NOTICE:
  # Find_by_*'s behavior differs from simply find's one!
  # i.e. you will find all as if using `administrator_find'.
  def test_find_by
    # User.current = nil
    assert_equal @logistics, Matter.find_by_id(1)
    assert_equal @skynet, Matter.find_by_id(2)
  end

  # ActsAsPermissible による制御の下で find_by_* のふるまいをテストする。
  def test_find_by__tesla
    User.current = @tesla
    assert_equal @logistics, Matter.find_by_id(1)
    assert_equal @skynet, Matter.find_by_anken_code("M2")
  end

  # ActsAsPermissible による制御の下で find_by_* のふるまいをテストする。
  def test_find_by__admin
    User.current = @admin
    assert_equal @logistics, Matter.find_by_id(1)
    assert_equal @skynet, Matter.find_by_anken_code("M2")
  end

  # ActsAsPermissible による制御の影響を受けない find_with_permission のふるまいをテストする。
  def test_find_with_permission
    # User.current = nil
    assert_equal 102, Matter.find_with_permission(:all).size
  end

  # ActsAsPermissible による制御の影響を受けない find_with_permission のふるまいをテストする。
  def test_find_with_permission__tesla
    User.current = @tesla
    assert_equal 22, Matter.find_with_permission(:all).size
  end

  # ActsAsPermissible による制御の影響を受けない find_with_permission のふるまいをテストする。
  def test_find_with_permission__admin
    User.current = @admin
    assert_equal 22, Matter.find_with_permission(:all).size
  end

  # ActsAsPermissible による制御の下で count が限られたオブジェクトのみを数えることをテストする。
  def test_count
    # User.current = nil
    assert_equal 102, Matter.count(:all, {})
  end

  # ActsAsPermissible による制御の下で count が限られたオブジェクトのみを数えることをテストする。
  def test_count__tesla
    User.current = @tesla
    assert_equal 1, Matter.count(:all, {})
  end

  # ActsAsPermissible による制御の下で count が限られたオブジェクトのみを数えることをテストする。
  def test_count__admin
    User.current = @admin
    assert_equal 22, Matter.count(:all, {})
  end

  # ActsAsPermissible が提供するインスタンスメソッド readable? が
  # 利用者の権限を反映していることをテストする。
  def test_readable?
    # User.current = nil
    assert @skynet.readable?
  end

  # ActsAsPermissible が提供するインスタンスメソッド readable? が
  # 利用者の権限を反映していることをテストする。
  def test_tesla_readable?
    User.current = @tesla
    assert !@logistics.readable?
    assert @skynet.readable?
  end

  # ActsAsPermissible が提供するインスタンスメソッド readable? が
  # 利用者の権限を反映していることをテストする。
  def test_admin_readable?
    User.current = @admin
    assert @skynet.readable?
  end

  # ActsAsPermissible が提供するインスタンスメソッド writable? が
  # 利用者の権限を反映していることをテストする。
  def test_writable?
    # User.current = nil
    assert @skynet.writable?
  end

  # ActsAsPermissible が提供するインスタンスメソッド writable? が
  # 利用者の権限を反映していることをテストする。
  def test_tesla_writable?
    User.current = @tesla
    assert !@logistics.writable?
    assert !@skynet.writable?
  end

  # ActsAsPermissible が提供するインスタンスメソッド writable? が
  # 利用者の権限を反映していることをテストする。
  def test_admin_writable?
    User.current = @admin
    assert @skynet.writable?
  end

  # Product#permission_enabled? に応じて権限を参照するかどうかを
  # 決定することをテストする。
  def test_permission_enabled_readable?
    m = matters(:anken_6)
    User.current = users(:tesla4)
    assert !m.readable?
    Product.update_all("permission_enabled = #{Product.connection.quoted_false}", "model_name = #{Product.quote_value('Matter')}")
    assert m.readable?
  end

  # Product#permission_enabled? に応じて権限を参照するかどうかを
  # 決定することをテストする。
  def test_permission_enabled_writable?
    m = matters(:anken_6)
    User.current = users(:tesla4)
    assert !m.writable?
    Product.update_all("permission_enabled = #{Product.connection.quoted_false}", "model_name = #{Product.quote_value('Matter')}")
    assert m.writable?
  end

  # ActsAsPermissible が提供するインスタンスメソッド readable_for? が
  # 利用者の権限を反映していることをテストする。
  def test_readable_for?
    assert @skynet.readable_for?(nil)
  end

  # ActsAsPermissible が提供するインスタンスメソッド readable_for? が
  # 利用者の権限を反映していることをテストする。
  def test_tesla_readable_for?
    assert !@logistics.readable_for?(@tesla.id)
    assert @skynet.readable_for?(@tesla.id)
  end

  # ActsAsPermissible が提供するインスタンスメソッド readable_for? が
  # 利用者の権限を反映していることをテストする。
  def test_admin_readable_for?
    assert @skynet.readable_for?(@admin.id)
  end

  # ActsAsPermissible が提供するインスタンスメソッド writable_for? が
  # 利用者の権限を反映していることをテストする。
  def test_writable_for?
    assert @skynet.writable_for?(nil)
  end

  # ActsAsPermissible が提供するインスタンスメソッド writable_for? が
  # 利用者の権限を反映していることをテストする。
  def test_tesla_writable_for?
    assert !@logistics.writable_for?(@tesla.id)
    assert !@skynet.writable_for?(@tesla.id)
  end

  # ActsAsPermissible が提供するインスタンスメソッド writable_for? が
  # 利用者の権限を反映していることをテストする。
  def test_admin_writable_for?
    assert @skynet.writable_for?(@admin.id)
  end

  # permission_enabled? が真でなければ initial_roleable に関わらず読み取りができることをテストする。
  def test_tesla_readable_for__initial_roleable
    assert workshops(:one).readable_for?(@tesla.id)
    assert workshops(:two).readable_for?(@tesla.id)
    assert workshops(:one).writable_for?(@tesla.id)
    assert !workshops(:two).writable_for?(@tesla.id)
  end

end
