# -*- coding: utf-8 -*-
require 'test_helper'

class Pick::ProjectControllerTest < ActionController::TestCase
  fixtures(:domains, :config_pickers, :users,
           :projects, :project_segments, :retrieval_divisions)

  def setup
    @request.session[:user_id] = User.find_by_login("tesla")
  end

  # rfw パラメータなしで search を呼ぶとエラーになる。
  def test_search__without_rfw
    assert_raise(ApplicationController::ClientError) do
      post :search
    end
  end

  # code も name も指定しない場合全件を表示する。
  def test_search__with_neither_code_nor_name
    post :search, :rfw => {:project_name => "xxx_project", :project_id => "xxx_project_id"}
    assert_response :success
    assert_template "search"
    assert_select "table.result>tr.project", :count => 4
  end

  # code で search を絞り込む。
  def test_search__with_code
    post :search, :rfw => {:project_name => "xxx_project", :project_id => "xxx_project_id"}, :code => "project1"
    assert_response :success
    assert_template "search"
    assert_select "table.result>tr.project", :count => 1
  end

  # name で search を絞り込む。
  def test_search__with_name
    post :search, :rfw => {:project_name => "yyy_project", :project_id => "yyy_project_id"}, :name => "プロジェクト2"
    assert_response :success
    assert_template "search"
    assert_select "table.result>tr.project", :count => 1
  end

  # code と name で絞り込む。
  def test_search__with_code_and_name
    post :search, :rfw => {:project_name => "zzz_project", :project_id => "zzz_project_id"}, :code => "project1", :name => "プロジェクト2"
    assert_response :success
    assert_template "search"
    assert_select "table.result>tr.project", :count => 0
  end

  # writable? でない場合は表示されない。
  def test_search__writable_only
    Project.any_instance.stubs(:writable?).returns(false)
    post :search, :rfw => {:project_name => "wo_project", :project_id => "wo_project_id"}
    assert_response :success
    assert_template "search"
    assert_select "table.result>tr.project", :count => 0
  end

  # rfw パラメータなしで search_projects を呼ぶとエラーになる。
  def test_search_projects__without_rfw
    assert_raise(ApplicationController::ClientError) do
      post :search_projects
    end
  end

  # code も name も指定しない場合 search_projects は全件を表示する。
  def test_search_projects__with_neither_code_nor_name
    post :search_projects, :rfw => {:project_name => "aaa_project", :project_id => "aaa_project_id"}
    assert_response :success
    assert_template "search_projects"
    assert_select "table.result_projects>tr.project", :count => 4
  end

  # code で search_projects を絞りこむ。
  def test_search_projects__with_code
    post :search_projects, :rfw => {:project_name => "aaa_project", :project_id => "aaa_project_id"}, :code => "project1"
    assert_response :success
    assert_template "search_projects"
    assert_select "table.result_projects>tr.project", :count => 1
  end

  # name で search_projects を絞りこむ。
  def test_search_projects__with_name
    post :search_projects, :rfw => {:project_name => "aaa_project", :project_id => "aaa_project_id"}, :name => "プロジェクト2"
    assert_response :success
    assert_template "search_projects"
    assert_select "table.result_projects>tr.project", :count => 1
  end

  # code と name で search_projects を絞りこむ。
  def test_search_projects__with_code_and_name
    post :search_projects, :rfw => {:project_name => "aaa_project", :project_id => "aaa_project_id"}, :code => "project2", :name => "プロジェクト1"
    assert_response :success
    assert_template "search_projects"
    assert_select "table.result_projects>tr.project", :count => 0
  end

  # writable? でない場合は表示されない。
  def test_search_projects__writable_only
    Project.any_instance.stubs(:writable?).returns(false)
    post :search_projects, :rfw => {:project_name => "wo_project", :project_id => "wo_project_id"}
    assert_response :success
    assert_template "search_projects"
    assert_select "table.result_projects>tr.project", :count => 0
  end

  # rfw パラメータなしで search_segments を呼ぶとエラーになる。
  def test_search_segments__without_rfw
    assert_raise(ApplicationController::ClientError) do
      post :search_segments
    end
  end

  # search_segments は project_id で絞りこむ。
  def test_search_segments__with_project_id
    post :search_segments, :rfw => {:project_name => "bbb_project", :project_id => "bbb_project_id"}, :project_id => 2
    assert_response :success
    assert_template "search_segments"
    assert_select "tr.segment", :count => 3
  end

  # writable? でない場合は表示されない。
  def test_search_segments__writable_only
    ProjectSegment.any_instance.stubs(:writable?).returns(false)
    post :search_segments, :rfw => {:project_name => "wo_project", :project_id => "wo_project_id"}, :project_id => 2
    assert_response :success
    assert_template "search_segments"
    assert_select "tr.segment", :count => 0
  end

  # rfw パラメータなしで select_retrieval_divisions を呼び出すとエラーになる。
  def test_select_retrieval_divisions__without_rfw
    assert_raise(ApplicationController::ClientError) do
      post :select_retrieval_divisions
    end
  end

  # select_retrieval_divisions はパラメータを入力欄にあらかじめ表示する。
  def test_select_retrieval_divisions__with_value
    rfw = {
      :project_retrieval_divisions => "ccc_project_retrieval_divisions",
    }
    (1..10).each do |i|
      rfw["project_retrieval_division_#{i}"] = "ccc_project_retrieval_division_#{i}"
    end
    post :select_retrieval_divisions, :rfw => rfw, :ccc_project_retrieval_division_1 => "XXX"
    assert_response :success
    assert_template "select_retrieval_divisions"
    assert_select "input[name=retrieval_division_1]" do
      assert_select "[value=?]", "XXX"
    end
  end

end
