# -*- coding: utf-8 -*-
# app/controllers/design_controller.rb にあるクラス DesignController をテストする。
# テストの対象となる実装は app/apis/design_api.rb にある。

require File.dirname(__FILE__) + '/../test_helper'
require 'design_controller'

class DesignController; def rescue_action(e) raise e end; end

class DesignControllerApiTest < Test::Unit::TestCase
  fixtures(:domains, :users, :people, :languages, :mail_formats,
           :po_messages, :po_translations, :po_arguments,
           :products, :displays, :items,
           :application_tables, :table_columns,
           :design_tickets)

  VALID_CLIENT_IDENTIFIER = {
    :client_identifier_x => 2,
    :client_identifier_y => 1,
    :client_identifier_z => 'unique0'
  }

  def setup
    @controller = DesignController.new
    @request    = ActionController::TestRequest.new
    @response   = ActionController::TestResponse.new
    CacheEachRequest.clear
  end

  def teardown
    CacheEachRequest.clear
  end

  # DesignApi の login が適切な入力で成功することをテストする。
  def test_login__success
    proposal = LoginStruct::Proposal.new(:parameter_0 => "admin", :parameter_1 => "atest")
    result = invoke_delegated :design, :login, proposal
    assert_kind_of LoginStruct::Approval, result
    assert_equal 2, result.client_identifier_x
    assert_equal 1, result.client_identifier_y
    assert result.client_identifier_z.length > 16
  end

  # DesignApi の login が不正な入力で失敗することをテストする。
  def test_login__failure
    proposal = LoginStruct::Proposal.new(:parameter_0 => "admin", :parameter_1 => "ruby")
    result = invoke_delegated :design, :login, proposal
    assert !result
  end

  # DesignApi が認証していないクライアントから呼び出された場合に
  # 例外を投げることをテストする。
  def test_authenticate__failure
    book = ConfigLanguageStruct::Book.new
    assert_raise(ActionWebService::Dispatcher::DispatcherError) do
      invoke_delegated :language, :import, book
    end
  end

  # DesignApi の logout が認証済みのクライアントから呼び出された場合に
  # 成功することをテストする。
  def test_logout__success
    proposal = LogoutStruct::Proposal.new(VALID_CLIENT_IDENTIFIER)
    result = invoke_delegated :design, :logout, proposal
    assert_kind_of LogoutStruct::Approval, result
  end

  # DesignApi の logout が不正な入力で失敗することをテストする。
  def test_logout__bad_parameters
    proposal = LogoutStruct::Proposal.new
    result = invoke_delegated :design, :logout, proposal
    assert !result
  end

  # Language#language_labels の動作をテストする。
  def test_language_labels
    labels = Language.language_labels
    labels.each do |label|
      assert_kind_of DesignStruct::LanguageLabel, label
    end
    assert_equal 10, labels.size
  end

  # Language#to_language_conf_item が動作することをテストする。
  def test_language_to_language_conf_item
    language = Language.find(1)
    item = language.to_language_conf_item
    assert_kind_of ConfigLanguageStruct::LanguageConfItem, item
    assert_equal 1, item.language_id
    assert_equal "ja", item.language_suffix
    assert_equal "日本語", item.language_name
  end

  # ConfigLanguageStruct::MainSheet#push が動作することをテストする。
  def test_language_push
    sheet = ConfigLanguageStruct::MainSheet.new
    ja = Language.find(1)
    sheet.push(ja)
    zh = ConfigLanguageStruct::LanguageConfItem.new(:language_id => 2, :language_suffix => "zh")
    sheet.push(zh)
    assert_equal 2, sheet.language_conf_items.size
  end

  # Language.to_book を引数無しで呼んだ場合の動作をテストする。
  def test_language_to_book__0
    book = Language.to_book
    assert_kind_of ConfigLanguageStruct::Book, book
    assert_equal 10, book.main_sheet.language_conf_items.size
  end

  # Language.to_book を1引数で呼んだ場合の動作をテストする。
  def test_language_to_book__1
    book = Language.to_book([1,3,6])
    assert_kind_of ConfigLanguageStruct::Book, book
    assert_equal 3, book.main_sheet.language_conf_items.size
  end

  # LanguageApi の import が適切なパラメータが与えられた場合に成功することをテストする。
  def test_language_import
    book = ConfigLanguageStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    result = invoke_delegated :language, :import, book
    assert_kind_of ConfigLanguageStruct::Book, result
  end

  # LanguageApi の export が新しい行を追加することをテストする。
  def test_language_export__insert
    book = ConfigLanguageStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    book.main_sheet = ConfigLanguageStruct::MainSheet.new
    item = ConfigLanguageStruct::LanguageConfItem.new(:language_id => 0, :language_suffix => "en")
    book.main_sheet.push(item)
    result = invoke_delegated :language, :export, book
    assert_kind_of ConfigLanguageStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal 1, result.main_sheet.language_conf_items.size
  end

  # LanguageApi の export が既存の行を更新することをテストする。
  def test_language_export__update
    book = Language.to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    book.main_sheet.language_conf_items.each {|item| item.language_suffix.upcase!}
    result = invoke_delegated :language, :export, book
    assert_kind_of ConfigLanguageStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal "JA", Language.find(1).code
  end

  # LanguageApi の export が不正な入力で失敗することをテストする。
  def test_language_export__failure
    book = ConfigLanguageStruct::Book.new
    assert_raise(ActionWebService::Dispatcher::DispatcherError) do
      invoke_delegated :language, :export, book
    end
  end

  # MailFormat#to_mail_conf_item が動作することをテストする。
  def test_mail_format_to_mail_conf_item
    f = MailFormat.find(1)
    item = f.to_mail_conf_item
    assert_kind_of MailFormatStruct::MailConfItem, item
    assert_equal 1, item.mail_id
    assert_equal "ProjectCollaboration( m=frm_p_name )", item.from_format
  end

  # MailFormatStruct::MainSheet#push が動作することをテストする。
  def test_mail_format_push
    sheet = MailFormatStruct::MainSheet.new
    item = MailFormatStruct::MailConfItem.new(:mail_id => 2,
                                              :mail_mode => "mode",
                                              :from_format => "abc",
                                              :recipients_format => "efg",
                                              :subject_format => "hij",
                                              :body_format => "klm")
    sheet.push(item)
    assert_equal 1, sheet.mail_conf_items.size
    assert_equal 2, sheet.mail_conf_items[0].mail_id
  end

  # MailFormat.to_book を引数無しで呼んだ場合の動作をテストする。
  def test_mail_format_to_book__0
    book = MailFormat.to_book
    assert_kind_of MailFormatStruct::Book, book
    assert_equal 1, book.main_sheet.mail_conf_items.size
  end

  # MailFormat.to_book を1引数で呼んだ場合の動作をテストする。
  def test_mail_format_to_book__1
    book = MailFormat.to_book([1])
    assert_kind_of MailFormatStruct::Book, book
    assert_equal 1, book.main_sheet.mail_conf_items.size
  end

  # MailApi の export が新しい行を追加することをテストする。
  def test_mail_export__insert
    book = MailFormatStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    book.main_sheet = MailFormatStruct::MainSheet.new
    item = MailFormatStruct::MailConfItem.new(:mail_id => 0,
                                              :mail_mode => "111",
                                              :from_format => "222",
                                              :recipients_format => "333",
                                              :subject_format => "444",
                                              :body_format => "555")
    book.main_sheet.push(item)
    book.main_sheet.push(MailFormat.find(1))
    result = invoke_delegated :mail, :export, book
    assert_kind_of MailFormatStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal 2, result.main_sheet.mail_conf_items.size
  end

  # MailApi の export が既存の行を更新することをテストする。
  def test_mail_export__update
    book = MailFormat.to_book
    book.main_sheet.mail_conf_items.each {|item| item.body_format = "No comment."}
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    result = invoke_delegated :mail, :export, book
    assert_kind_of MailFormatStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal 1, result.main_sheet.mail_conf_items.size
    assert_equal "No comment.", result.main_sheet.mail_conf_items[0].body_format
  end

  # MailApi の export が不正な入力で失敗することをテストする。
  def test_mail_export__failure
    book = MailFormatStruct::Book.new
    assert_raise(ActionWebService::Dispatcher::DispatcherError) do
      invoke_delegated :mail, :export, book
    end
  end

  # PoMessage.to_book が適切な CommonLabelStruct::Book のインスタンスを返すことをテストする。
  def test_po_message_to_book
    book = PoMessage.to_book
    assert_kind_of CommonLabelStruct::Book, book
    #assert (2049..2051).include?(book.main_sheet.common_labels.size)
    #print "size: ", book.main_sheet.common_labels.size, "\n"
  end

  # CommonLabelApi の import の動作をテストする。
  def test_common_label_import
    book = CommonLabelStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    result = invoke_delegated :common_label, :import, book
    assert_kind_of CommonLabelStruct::Book, result
    assert_equal "success", result.message_code
    #assert (2049..2051).include?(result.main_sheet.common_labels.size)
    #print "size: ", result.main_sheet.common_labels.size, "\n"
  end

  # CommonLabelApi の export が新しい msgid を追加することをテストする。
  def test_common_label_export__insert_msgid
    book = PoMessage.to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    label = CommonLabelStruct::CommonLabel.new
    label.type = "PoMessageSingular"
    label.msgid = "coffee or tea?"
    label.common_values = [CommonLabelStruct::CommonValue.new(:language_id => 1, :common_value => "コーヒーまたは紅茶どちらにしますか?")]
    book.main_sheet.common_labels << label
    result = invoke_delegated :common_label, :export, book
    assert_kind_of CommonLabelStruct::Book, result
    assert_equal "success", result.message_code
    #assert (2033..2042).include?(result.main_sheet.common_labels.size)
    #print "size: ", result.main_sheet.common_labels.size, "\n"
  end

  # CommonLabelApi の export が新しい msgstr を追加することをテストする。
  def test_common_label_export__insert_msgstr
    book = PoMessage.to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    label = book.main_sheet.common_labels[0]
    label.common_values << CommonLabelStruct::CommonValue.new(:language_id => 4, :common_value => "coffee or tea?")
    result = invoke_delegated :common_label, :export, book
    assert_kind_of CommonLabelStruct::Book, result
    assert_equal "success", result.message_code
    #assert (2033..2051).include?(result.main_sheet.common_labels.size)
    #print "size: ", result.main_sheet.common_labels.size, "\n"
    assert_equal "こんにちは、世界!", result.main_sheet.common_labels[0].common_values[0].common_value
    assert_equal "coffee or tea?",    result.main_sheet.common_labels[0].common_values[1].common_value
  end

  # CommonLabelApi の export が既存の msgstr を更新することをテストする。
  def test_common_label_export__update_msgstr
    book = PoMessage.to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    label = book.main_sheet.common_labels[0]
    label.common_values = [CommonLabelStruct::CommonValue.new(:language_id => 1, :common_value => "コーヒーまたは紅茶どちらにしますか?")]
    result = invoke_delegated :common_label, :export, book
    assert_kind_of CommonLabelStruct::Book, result
    assert_equal "success", result.message_code
    #assert (2033..2051).include?(result.main_sheet.common_labels.size)
    #print "size: ", result.main_sheet.common_labels.size
    assert_equal "コーヒーまたは紅茶どちらにしますか?", result.main_sheet.common_labels[0].common_values[0].common_value
  end

  # MessageApi の import の動作をテストする。
  def test_message_import
    book = MessageStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    result = invoke_delegated :message, :import, book
    assert_kind_of MessageStruct::Book, result
    assert_equal "success", result.message_code
    #assert (1..3).include?(result.main_sheet.messages.size)
  end

  # MessageApi の export が新しい msgid を追加することをテストする。
  def test_message_export__insert_msgid
    book = MessageStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    book.main_sheet = MessageStruct::MainSheet.new
    newbie = MessageStruct::Message.new(:message_msgid => "foo bar baz")
    newbie.message_values = []
    (1..10).each do |i|
      value = MessageStruct::MessageValue.new(:language_id => i, :message_value => "translated #{i}")
      newbie.message_values << value
    end
    book.main_sheet.messages = [newbie]
    result = invoke_delegated :message, :export, book
    assert_kind_of MessageStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal 1, result.main_sheet.messages.size
  end

  # MessageApi の export が新しい msgstr を追加することをテストする。
  def test_message_export__insert_msgstr
    book = PoMessageSingular.to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    messages_size = book.main_sheet.messages.size
    values_size = book.main_sheet.messages[0].message_values.size
    book.main_sheet.messages[0].message_values << MessageStruct::MessageValue.new(:language_id => 5, :message_value => "here you are.")
    result = invoke_delegated :message, :export, book
    assert_kind_of MessageStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal messages_size, result.main_sheet.messages.size
    assert_equal values_size + 1, result.main_sheet.messages[0].message_values.size
    assert_equal "here you are.", result.main_sheet.messages[0].message_values[1].message_value
  end

  # MessageApi の export が既存の msgstr を更新することをテストする。
  def test_message_export__update_msgstr
    book = PoMessageSingular.to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    messages_size = book.main_sheet.messages.size
    values_size = book.main_sheet.messages[0].message_values.size
    book.main_sheet.messages[0].message_values[0] = MessageStruct::MessageValue.new(:language_id => 5, :message_value => "here am I?")
    result = invoke_delegated :message, :export, book
    assert_kind_of MessageStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal values_size, result.main_sheet.messages[0].message_values.size
    assert_equal "here am I?", result.main_sheet.messages[0].message_values[0].message_value
  end

  # Item#to_item が適切な DisplayStruct::Item のインスタンスを返すことをテストする。
  def test_item_to_item
    item = items(:one).to_item
    assert_kind_of DisplayStruct::Item, item
    assert_equal 1, item.item_id
    assert_equal 1, item.display_id
    assert_equal "domain_id", item.column_name
    assert_equal "C_ITEM_1", item.item_code
  end

  # Display#to_display_item が適切な DisplayStruct::DisplayItem のインスタンスを返すことをテストする。
  def test_display_to_dipslay_item
    display_item = Display.find(81).to_display_item
    assert_kind_of DisplayStruct::DisplayItem, display_item
  end

  # Product#to_item_sheet が適切な DisplayStruct::ItemSheet のインスタンスを返すことをテストする。
  def test_product_to_item_sheet
    product = Product.find(8)
    item_sheet = product.to_item_sheet
    assert_kind_of DisplayStruct::ItemSheet, item_sheet
  end

  # Product#to_book が適切な DisplayStruct::Book のインスタンスを返すことをテストする。
  def test_product_to_book
    book = Product.find(8).to_book
    assert_kind_of DisplayStruct::Book, book
  end

  # DisplayApi の import が動作することをテストする。
  def test_display_import
    book = DisplayStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    book.product_id = 8
    result = invoke_delegated :display, :import, book
    assert_kind_of DisplayStruct::Book, result
    assert_equal "success", result.message_code
    (0..3).each do |i|
      assert_equal 81+i, result.main_sheet.display_items[i].display_id
    end
  end

  # DisplayApi の export が display の内容を更新することをテストする。
  def test_display_export__update_display
    book = Product.find(8).to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    book.new_sheet.new_items[0].width = "333px"
    result = invoke_delegated :display, :export, book
    assert_kind_of DisplayStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal "333px", result.new_sheet.new_items[0].width
  end
  
  # DisplayApi の export が既存の item を更新することをテストする。
  def test_display_export__update_item
    book = Product.find(8).to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    items_size = book.item_sheet.items.size
    item = book.item_sheet.items[0]
    item_id = item.item_id
    assert (0 < Item.find(item_id).name_po)
    item.item_names = [DisplayStruct::ItemName.new(:language_id => 1, :item_name => "あいうえお")]
    result = invoke_delegated :display, :export, book
    assert_kind_of DisplayStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal items_size, result.item_sheet.items.size
    assert_equal item_id, result.item_sheet.items[0].item_id
    assert_equal "code", result.item_sheet.items[0].column_name
    assert_equal 1, result.item_sheet.items[0].item_names[0].language_id
#    assert_equal "あいうえお", result.item_sheet.items[0].item_names[0].item_name
  end

  # DisplayApi の export が新しい item を追加することをテストする。
  def test_display_export__insert_item
    book = Product.find(8).to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    size = book.item_sheet.items.size
    item = DisplayStruct::Item.new
    item.item_id = 0
    item.item_names = [DisplayStruct::ItemName.new(:language_id => 1, :item_name => "一時的な名前")]
    book.item_sheet.items << item
    %w|new edit show list|.each do |name|
      items = book.__send__("#{name}_sheet").__send__("#{name}_items")
      x = items[0].clone
      x.item_id = 0
      x.code = "our_new_item_for_#{name}"
      x.layout = 4
      x.column_name = "updated_at"
      book.__send__("#{name}_sheet").__send__("#{name}_items").<<(x)
    end
    result = invoke_delegated :display, :export, book
    assert_kind_of DisplayStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal size+4, result.item_sheet.items.size
  end

  # Product.to_product_book が適切な ProductStruct::Book のインスタンスを返すことをテストする。
  def test_product_to_product_book
    book = Product.to_product_book(1)
    assert_kind_of ProductStruct::Book, book
    assert_equal 1, book.main_sheet.product_items[0].product_id
    assert_equal "P_ROOT", book.main_sheet.product_items[0].product_code
  end

  # ProductService#import が適切に動作することをテストする。
  def test_product_import
    book = ProductStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    result = invoke_delegated :product, :import, book
    assert_kind_of ProductStruct::Book, result
    assert_equal "success", result.message_code
  end

  # ProductService#export が既存の product を更新することをテストする。
  def test_product_export__update_product
    book = Product.to_product_book(1)
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    size = book.main_sheet.product_items.size
    book.main_sheet.product_items[10].csv = true
    result = invoke_delegated :product, :export, book
    assert_kind_of ProductStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal size, result.main_sheet.product_items.size
    assert_equal true, result.main_sheet.product_items[10].csv
  end

  # ProductService#export が既存の product の名前を更新することをテストする。
  def test_product_export__update_product_name
    book = Product.to_product_book(1)
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    size = book.main_sheet.product_items.size
    book.main_sheet.product_items[10].product_names = [ProductStruct::ProductName.new(:language_id => 1, :product_name => "Windows")]
    result = invoke_delegated :product, :export, book
    assert_kind_of ProductStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal size, result.main_sheet.product_items.size
    assert_equal "Windows", result.main_sheet.product_items[10].product_names[0].product_name
  end

  # ProductService#export が新しい product を追加することをテストする。
  def test_product_export__insert_product
    book = Product.to_product_book(1)
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    size = book.main_sheet.product_items.size
    item = ProductStruct::ProductItem.new
    item.product_id = 0
    item.product_code = "foo_bar_baz"
    item.product_model = 'Matter'
    item.workflow_enabled = false
    item.workflow = 'abc'
    item.workflow_body_method = 'xxx'
    item.mail = true
    item.mail_skip_auth = false
    item.document = false
    item.document_name_method = 'id'
    item.document_number_method = ''
    item.document_content_method = ''
    item.attachment = true
    item.search = true
    item.csv = false
    item.initial_roleable_type = "Company"
    item.scope_roleable_type = "Person"
    item.product_names = [ProductStruct::ProductName.new(:language_id => 1, :product_name => "11111"),
                          ProductStruct::ProductName.new(:language_id => 2, :product_name => "22222"),
                         ]
    book.main_sheet.product_items << item
    result = invoke_delegated :product, :export, book
    assert_kind_of ProductStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal size+1, result.main_sheet.product_items.size
    assert_equal false, result.main_sheet.product_items[size].csv
    assert_equal "11111", result.main_sheet.product_items[size].product_names[0].product_name
    assert_equal "22222", result.main_sheet.product_items[size].product_names[1].product_name
  end

  # Product.to_menu_book が適切な MenuStruct::Book インスタンスを返すことをテストする。
  def test_product_to_menu_book
    ids = Product.find(:all, :conditions => {:domain_id => 1}, :order => "lft").select do |product|
      level = product.ancestors.size
      1 <= level && level <= 3
    end.map(&:id)
    book = Product.to_menu_book(1)
    assert_kind_of MenuStruct::Book, book
    assert_equal ids, book.main_sheet.menu_items.map(&:product_id)
  end

  # MenuService#import が適切な MenuStruct::Book インスタンスを返すことをテストする。
  def test_menu_import
    ids = Product.find(:all, :conditions => {:domain_id => 1}, :order => "lft").select do |product|
      level = product.ancestors.size
      1 <= level && level <= 3
    end.map(&:id)
    book = Product.to_menu_book(1)
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    result = invoke_delegated :menu, :import, book
    assert_kind_of MenuStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal ids, result.main_sheet.menu_items.map(&:product_id)
  end

  # MenuService#export がメニューのレイアウトを更新することをテストする。
  def test_menu_export
    book = Product.to_menu_book(1)
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    # transpose two menus of the same level.
    book.main_sheet.menu_items[0], book.main_sheet.menu_items[4] = book.main_sheet.menu_items[4], book.main_sheet.menu_items[0]
    result = invoke_delegated :menu, :export, book
    assert_kind_of MenuStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal 3, result.main_sheet.menu_items[0].product_id
    assert_equal 2, result.main_sheet.menu_items[4].product_id
  end

  # MenuService#export が新しい ProductPseudo を追加することをテストする。
  def test_menu_export__new_product_pseudo
    book = Product.to_menu_book(1)
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    menu_item = MenuStruct::MenuItem.new(:product_id => 0,
                                         :menu_hierarchy_level_1 => 0,
                                         :menu_hierarchy_level_2 => 1,
                                         :menu_hierarchy_level_3 => 0,
                                         :product_code => "NEWBIE",
                                         :product_name => "NEWBIE",
                                         :menu_type => true)
    book.main_sheet.menu_items << menu_item
    result = invoke_delegated :menu, :export, book
    assert_kind_of MenuStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal "NEWBIE", result.main_sheet.menu_items.last.product_name
    assert result.main_sheet.menu_items.last.menu_type
  end

  # ApplicationTableService#import が適切な ApplicationTableStruct::Book インスタンスを返すことをテストする。
  def test_application_table_import
    book = ApplicationTableStruct::Book.new(VALID_CLIENT_IDENTIFIER)
    result = invoke_delegated :application_table, :import, book
    assert_kind_of ApplicationTableStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal 10, result.main_sheet.tables.size
  end

  # ApplicationTableService#export が成功することをテストする。
  def test_application_table_export__success
    book = ApplicationTable.to_book
    VALID_CLIENT_IDENTIFIER.each {|k,v| book.__send__ "#{k}=", v}
    result = invoke_delegated :application_table, :export, book
    assert_kind_of ApplicationTableStruct::Book, result
    assert_equal "success", result.message_code
    assert_equal 10, result.main_sheet.tables.size
  end
end
